/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import static org.nuiton.i18n.I18n._;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * Class containing the methods to manage the forms :
 * - creation, update, deletion, retrieving
 * - binding forms with clients whose queries are returning the forms
 *
 * @author schorlet
 * @date 2010-01-27 11:03:15
 * @version $Revision: 474 $ $Date: 2010-02-07 10:52:07 +0100 (dim., 07 févr. 2010) $
 */
public class FormManager {
    private static final Log log = LogFactory.getLog(FormManager.class);

    private final WikittyProxy proxy;
    private final ClientManager clientManager;
    private final ThesaurusManager thesaurusManager;
    private final SearchManager searchManager;

    public final static String TO_TREAT_STATUS_NAME = _("vradi.status.toTreat.name");
    public final static String TO_TREAT_STATUS_DESCRIPTION = _("vradi.status.toTreat.description");
    public final static String VALIDATED_STATUS_NAME = _("vradi.status.validated.name");
    public final static String VALIDATED_STATUS_DESCRIPTION = _("vradi.status.validated.description");
    public final static int TO_TREAT_STATUS_VALUE = 0;
    public final static int VALIDATED_STATUS_VALUE = 1;
    
    /**
     * Form Id date format
     */
    public static final SimpleDateFormat FORM_ID_DATE_FORMAT =
            new SimpleDateFormat("yyyy-MM-dd");
    
    /**
     * default value of the sending status
     */
    public static final int DEFAULT_SENDING_STATUS = 0;
    
    /**
     * default value of the sending paragraph : cannot be null or empty
     * to be able to look for it in the solr index
     */
    public static final String DEFAULT_SENDING_PARAGRAPH = "null";
    
    public FormManager(WikittyProxy proxy, ClientManager clientManager,
            ThesaurusManager thesaurusManager, SearchManager searchManager) {
        this.proxy = proxy;
        this.clientManager = clientManager;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    public FormManager(ClientManager clientManager, ThesaurusManager thesaurusManager,
            SearchManager searchManager) {
        proxy = ServiceHelper.getWikittyProxy();
        this.clientManager = clientManager;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    public List<Form> getAllForms() {
        if (log.isDebugEnabled()) {
            log.debug("getAllForms()");
        }
        
        Search search = Search.query().eq(Element.ELT_EXTENSION, Form.EXT_FORM);
        Criteria criteria = search.criteria();
        
        PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
        List<Form> all = forms.getAll();
        
        return all;
    }
    
    public List<Form> getForms(List<String> formIds) {
        if (log.isDebugEnabled()) {
            log.debug("getForms(" + formIds + ")");
        }

        List<Form> forms = proxy.restore(Form.class, formIds);
        return forms;
    }
    
    public Form getForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("getForm(" + formId + ")");
        }
        
        Form form = proxy.restore(Form.class, formId);
        return form;
    }
    
    public void deleteForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("deleteForm(" + formId + ")");
        }
        
        proxy.delete(formId);
    }
    
    public Form updateForm(Form form) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateForm(form)");
        }
        
        List<Form> forms = Arrays.asList(form);
        List<Form> updateForms = updateForms(forms);
        
        Form updatedForm = updateForms.get(0);
        return updatedForm;
    }
    
    public List<Form> updateForms(List<Form> forms) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateForms(forms)");
        }
        
        List<TreeNode> thesaurusList = new ArrayList<TreeNode>();
        
        for (Form form : forms) {
            if (log.isDebugEnabled()) {
                log.debug("updating form: " + form.getId());
            }
            
            if (form.getCreationDate() == null) {
                form.setCreationDate(new Date());
                
            } else {
                if(!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATION_TAG)) {
                    form.addExtension(ModificationTag.MODIFICATION_TAG);
                }
                form.setField(ModificationTag.EXT_MODIFICATION_TAG,
                        ModificationTag.FIELD_LAST_MODIFIED, new Date());
            }
        
            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus != null) {
                for (String thesaurusId : thesaurus) {
                    
                    TreeNode node = thesaurusManager.getThesaurus(thesaurusId);
                    
                    if (node.getChildren() == null
                            || !node.getChildren().contains(form.getWikittyId())) {
                        
                        node.addChildren(form.getWikittyId());
                        
                        if (!thesaurusList.contains(node)) {
                            thesaurusList.add(node);
                        }
                    }
                }
            }
            
//            String incrementMajorRevision = WikittyUtil.incrementMajorRevision(
//                    form.getWikittyVersion());
//            form.setWikittyVersion(incrementMajorRevision);
        }
        
        proxy.store(thesaurusList);
        List<Form> result = proxy.store(forms);
        
        return result;
    }
    
    public Map<Form, List<Client>> getClientsByForms(String dateType,
            Date beginDate, Date endDate, WikittyExtension extension) {
        if (log.isDebugEnabled()) {
            log.debug("getClientsByForms(dateType, beginDate, endDate, extension)");
        }

        Map<Form, List<Client>> clientsByForm = new HashMap<Form, List<Client>>();
        
        List<Form> forms = getAllForms();
        for (Form form : forms) {

            String dateExtension = null;
            String dateTypeName = null;

            if (dateType != null) {
                int dot = dateType.indexOf('.');
                if (dot > 0) {
                    dateExtension = dateType.substring(0, dot);
                    dateTypeName = dateType.substring(dot + 1);
                }
            }

            Date date = (Date) form.getField(dateExtension, dateTypeName);
            
            if ((extension == null || form.getExtensions().contains(extension))
                    && (dateType == null || beginDate == null || !beginDate.after(date))
                    && (dateType == null || endDate == null || !endDate.before(date))) {

                List<String> clientIds = new ArrayList<String>();
                Criteria criteria = Search.query()
                        .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                        .eq(Sending.FQ_FIELD_FORM, form.getWikittyId())
                        .criteria();
                
                PagedResult<Sending> sendings = proxy.findAllByCriteria(Sending.class, criteria);
                List<Sending> sendingList = sendings.getAll();

                for (Sending sending : sendingList) {
                    clientIds.add(sending.getClient());
                }

                if (!clientIds.isEmpty()) {
                    List<Client> clients = proxy.restore(Client.class, clientIds);
                    clientsByForm.put(form, clients);
                }
            }
        }
        
        return clientsByForm;
    }

    public List<VradiSendingDTO> getFormsByClients(String dateType,
            Date beginDate, Date endDate, WikittyExtension extension,
            Boolean receptionProof, Boolean paragraph, int status)
            throws TechnicalException {

        if (log.isDebugEnabled()) {
            log.debug("getFormsByClients(dateType, beginDate, endDate, extension, receptionProof, paragraph, status)");
        }
        
        List<VradiSendingDTO> formsByClients = new ArrayList<VradiSendingDTO>();
        List<Client> clients = clientManager.getAllClients();
        
        for (Client client : clients) {
            Search search = Search.query()
                    .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                    .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId());
        
            if (receptionProof != null) {
                search.eq(Sending.FQ_FIELD_RECEPTIONPROOF, receptionProof.toString());
            }
            
            if (paragraph != null) {
                if (paragraph) {
                    search.neq(Sending.FQ_FIELD_PARAGRAPH, DEFAULT_SENDING_PARAGRAPH);
                } else {
                    search.eq(Sending.FQ_FIELD_PARAGRAPH, DEFAULT_SENDING_PARAGRAPH);
                }
            }
            
            if (status >= 0) {
                search.eq(Sending.FQ_FIELD_STATUS, String.valueOf(status));
            }
            
            Criteria criteria = search.criteria();
            PagedResult<Sending> sendings = proxy.findAllByCriteria(Sending.class, criteria);
            List<Sending> sendingsList = sendings.getAll();
            
            for (Sending sending : sendingsList) {
                Set<String> sendingForm = sending.getForm();
                List<Form> forms = new ArrayList<Form>();
                
                if (sendingForm != null && !sendingForm.isEmpty()) {
                    List<Form> forms2 = getForms(new ArrayList<String>(sendingForm));
                    forms.addAll(forms2);
                }
                
                String dateExtension = null;
                String dateTypeName = null;
                
                if (dateType != null) {
                    int dot = dateType.indexOf('.');
                    if (dot > 0) {
                        dateExtension = dateType.substring(0, dot);
                        dateTypeName = dateType.substring(dot + 1);
                    }
                }
                
                List<Form> formsToSend = new ArrayList<Form>();
                
                for (Form form : forms) {
                    Date date = (Date) form.getField(dateExtension, dateTypeName);
                    
                    if ((extension == null || form.getExtensions().contains(extension))
                            && (dateType == null || beginDate == null || !beginDate.after(date))
                            && (dateType == null || endDate == null || !endDate.before(date))) {
                        
                        formsToSend.add(form);
                    }
                }
                
                if (!formsToSend.isEmpty()) {
                    VradiSendingDTO formsByClient = new VradiSendingDTO(sending);
                    formsByClient.setFormDTOs(formsToSend);
                    formsByClients.add(formsByClient);
                }
            }
        }
        
        return formsByClients;
    }
    
    public void bindFormsToClients() {
        if (log.isDebugEnabled()) {
            log.debug("bindFormsToClients()");
        }
        
        List<Sending> toSend = new ArrayList<Sending>();
        try {
            List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
            
            //get all the clients
            List<Client> clients = clientManager.getAllClients();
            for (Client client : clients) {
            
                // look for all the already bound forms
                Criteria criteria = Search.query()
                        .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                        .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId())
                        .criteria();
                
                List<Sending> sendings = proxy
                        .findAllByCriteria(Sending.class, criteria).getAll();
                
                // List of all the already bound forms
                List<String> alreadyBoundFormIds = new ArrayList<String>();
                for (Sending sending : sendings) {
                    if (sending.getForm() != null) {
                        alreadyBoundFormIds.addAll(sending.getForm());
                    }
                }
                
                queryMakers.add(client);
                
                // get all the client users
                List<User> users = clientManager.getClientUsers(client.getWikittyId());
                queryMakers.addAll(users);
                
                // List of the forms to bind ( = not yet bound)
                List<String> formsToBindIds = new ArrayList<String>();
                
                // iterate on the client and its users
                for (QueryMaker queryMaker : queryMakers) {
                
                    //if the query maker has queries
                    if (queryMaker.getQueries() != null) {
                        Set<String> queries = queryMaker.getQueries();
                        
                        for (String query : queries) {
                            //create a new VradiFormPageDTO to find
                            //  all the forms corresponding to the query
                            VradiFormPageDTO vradiFormPage = new VradiFormPageDTO();
                            try {
                                searchManager.findForms(query, vradiFormPage);
                                
                                List<Form> forms = vradiFormPage.getFormsToShow();
                                for (Form form : forms) {
                                    //if the form has not yet been bound,
                                    //  add it to the forms to bind
                                    if (!alreadyBoundFormIds.contains(form.getWikittyId())) {
                                        formsToBindIds.add(form.getWikittyId());
                                    }
                                }
                            } catch (UnsupportedQueryException e) {
                                log.warn("it seems that a persisted query is not supported anymore", e);
                            }
                        }
                    }
                }
                
                // create a new Sending
                Sending newSending = new SendingImpl();
                
                // look for a sending with not yet sent forms
                for (Sending sending : sendings) {
                    if (sending.getStatus() == DEFAULT_SENDING_STATUS) {
                        newSending = sending;
                        break;
                    }
                }
                
                // if some forms are to be bound, bind them
                if (!formsToBindIds.isEmpty()) {
                    newSending.setClient(client.getWikittyId());
                    
                    for (String formsToSendId : formsToBindIds) {
                        newSending.addForm(formsToSendId);
                        alreadyBoundFormIds.add(formsToSendId);
                    }
                
                    newSending.setSentDate(null);
                    newSending.setReceptionDate(null);
                    newSending.setParagraph(DEFAULT_SENDING_PARAGRAPH);
                    newSending.setReceptionProof(false);
                    newSending.setStatus(DEFAULT_SENDING_STATUS);
                    toSend.add(newSending);
                }
                
                queryMakers.clear();
            }
            
            // store the data
            proxy.store(toSend);

        } catch (TechnicalException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
    }

    public Status getStatus(String statusId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getStatus(" + statusId + ")");
        }
        Status status = proxy.restore(Status.class, statusId);
        return status;
    }

    public List<Status> getStatuses(List<String> statusIds) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getStatuses(statusIds)");
        }
        List<Status> statuses = proxy.restore(Status.class, statusIds);
        return new ArrayList<Status>(statuses);
    }

    public List<Status> getAllStatuses() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllStatus()");
        }

        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        Criteria criteria = search.criteria();

        PagedResult<Status> statuses = proxy.findAllByCriteria(Status.class,
                criteria);
        List<Status> all = statuses.getAll();

        if(all == null || all.isEmpty()) {
            log.warn("no status found, creating ...");
            all = new ArrayList<Status>();
            all.add(getToTreatStatus());
            all.add(getValidatedStatus());
        }
        return new ArrayList<Status>(all);
    }

    public Status getToTreatStatus() {
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, TO_TREAT_STATUS_NAME);
        Criteria criteria = search.criteria();

        Status toTreatStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(toTreatStatus == null) {
            toTreatStatus = new StatusImpl();
            toTreatStatus.setName(TO_TREAT_STATUS_NAME);
            toTreatStatus.setValue(TO_TREAT_STATUS_VALUE);
            toTreatStatus.setDescription(TO_TREAT_STATUS_DESCRIPTION);
            toTreatStatus.setModifiable(false);

            toTreatStatus = proxy.store(toTreatStatus);
        }
        return toTreatStatus;
    }

    public Status getValidatedStatus() {
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Status.EXT_STATUS);
        search.eq(Status.FQ_FIELD_NAME, VALIDATED_STATUS_NAME);
        Criteria criteria = search.criteria();

        Status validatedStatus = proxy.findByCriteria(Status.class,
                criteria);
        if(validatedStatus == null) {
            validatedStatus = new StatusImpl();
            validatedStatus.setName(VALIDATED_STATUS_NAME);
            validatedStatus.setValue(VALIDATED_STATUS_VALUE);
            validatedStatus.setDescription(VALIDATED_STATUS_DESCRIPTION);
            validatedStatus.setModifiable(false);

            validatedStatus = proxy.store(validatedStatus);
        }
        return validatedStatus;
    }

    public List<Status> updateStatuses(List<Status> statuses)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatuses(statuses)");
        }
        List<Status> result = proxy.store(statuses);

        return result;
    }

    public Status updateStatus(Status status) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatus(status)");
        }

        List<Status> statuses = Arrays.asList(status);
        List<Status> updateStatuses = updateStatuses(statuses);

        Status updatedStatus = updateStatuses.get(0);
        return updatedStatus;
    }

    public void deleteStatuses(List<String> statusIds) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("deleteStatus(" + statusIds + ")");
        }

        proxy.delete(statusIds);
    }
}
