/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jayway.restassured.config;

import static com.jayway.restassured.config.MatcherConfig.ErrorDescriptionType.REST_ASSURED;
import static com.jayway.restassured.internal.assertion.AssertParameter.notNull;

/**
 * Allow you to configure settings for response matchers.
 */
public class MatcherConfig implements Config {

    private final ErrorDescriptionType errorDescriptionType;
    private final boolean isUserDefined;

    /**
     * Creates a new instance of {@link com.jayway.restassured.config.MatcherConfig} that uses {@link ErrorDescriptionType#REST_ASSURED} as
     * error description type.
     */
    public MatcherConfig() {
        this(REST_ASSURED, false);
    }

    /**
     * Creates a new instance of {@link com.jayway.restassured.config.MatcherConfig} that uses the supplied <code>errorDescriptionType</code> as
     * error description type.
     *
     * @param errorDescriptionType The error description type to use.
     */
    public MatcherConfig(ErrorDescriptionType errorDescriptionType) {
        this(errorDescriptionType, true);
    }

    private MatcherConfig(ErrorDescriptionType errorDescriptionType, boolean isUserDefined) {
        notNull(errorDescriptionType, ErrorDescriptionType.class);
        this.errorDescriptionType = errorDescriptionType;
        this.isUserDefined = isUserDefined;
    }

    /**
     * @param errorDescriptionType The error description type to use.
     * @return A new instance of MatcherConfig.
     */
    public MatcherConfig errorDescriptionType(ErrorDescriptionType errorDescriptionType) {
        return new MatcherConfig(errorDescriptionType, true);
    }

    /**
     * @return The error description type.
     */
    public ErrorDescriptionType errorDescriptionType() {
        return errorDescriptionType;
    }

    /**
     * Returns <code>true</code> if this config has the supplied <code>errorDescriptionType</code>, <code>false</code> otherwise.
     *
     * @param errorDescriptionType The error description type to check for.
     * @return <code>true</code> if this config has the supplied <code>errorDescriptionType</code>, <code>false</code> otherwise.
     */
    public boolean hasErrorDescriptionType(ErrorDescriptionType errorDescriptionType) {
        return this.errorDescriptionType == errorDescriptionType;
    }

    public boolean isUserConfigured() {
        return isUserDefined;
    }

    /**
     * Represents the different error description types available in REST Assured. This determines how error messages are generated when assertions
     * fail.
     */
    public static enum ErrorDescriptionType {
        /**
         * The error descriptions are generated by REST Assured in order to make them easier to understand. In some situations error descriptions
         * generated with this approach will have the opposite effect and make them harder to understand. In that case you can configure REST Assured to use {@link #HAMCREST}
         * message descriptions instead.
         */
        REST_ASSURED,
        /**
         * Tries to stay as close to the original Hamcrest error messages as possible.
         */
        HAMCREST
    }

    public static MatcherConfig matcherConfig() {
        return new MatcherConfig();
    }
}
