/*
 * #%L
 * SGQ :: Business
 * $Id: PresentationDAOImpl.java 394 2013-06-19 08:44:58Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1/sgq-business/src/main/java/com/herbocailleau/sgq/entities/PresentationDAOImpl.java $
 * %%
 * Copyright (C) 2012, 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.model.PresentationModel;

public class PresentationDAOImpl<E extends Presentation> extends PresentationDAOAbstract<E> {

    /**
     * Find all presentation for non expired and valid batch.
     * 
     * @param productName 
     * @return
     * @throws TopiaException 
     */
    public List<Presentation> findAllNonExpiredValid(String productName) throws TopiaException {
        String query = "from " + Presentation.class.getName() +
                " where " + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and " + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_INVALID + " = false";

        List<Presentation> result = null;
        if (StringUtils.isNotBlank(productName)) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive(Presentation.PROPERTY_BATCH +
                        "." + Batch.PROPERTY_PRODUCT + "." + Product.PROPERTY_NAME, ":product");
            result = context.findAll(query, "product", "%" + StringUtils.stripAccents(productName) + "%");
        } else {
            result = context.findAll(query);
        }

        return result;
    }
    
    /**
     * Find all presentation for non expired and valid batch.
     * 
     * @param product product name or code
     * @return
     * @throws TopiaException 
     */
    public List<PresentationModel> findAllNonExpiredValidModel(String product) throws TopiaException {
        String query = "select P, PP.place," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :ze)," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :zp) + " +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_DESTINATION + " = :zp)," +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_DESTINATION + " = :zc)" +
                " from " + Presentation.class.getName() + " P," +
                " " + ProductPlace.class.getName() + " PP" +
                " where P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_INVALID + " = false" +
                " and P.batch.product = PP.product" + 
                " and PP.presentationCode = P.presentationCode";

        List<Object[]> lines = null;
        if (StringUtils.isNotBlank(product)) {
            query += " and (" + BatchDAOImpl.getFieldLikeInsensitive("P." + Presentation.PROPERTY_BATCH +
                        "." + Batch.PROPERTY_PRODUCT + "." + Product.PROPERTY_NAME, ":product") +
                     " or " + BatchDAOImpl.getFieldLikeInsensitive("P." + Presentation.PROPERTY_BATCH +
                             "." + Batch.PROPERTY_PRODUCT + "." + Product.PROPERTY_CODE, ":product") + ")";
            lines = context.findAll(query, "product", "%" + StringUtils.stripAccents(product) + "%",
                    "ze", Zone.ZE, "zc", Zone.ZC, "zp", Zone.ZP);
        } else {
            lines = context.findAll(query, "ze", Zone.ZE, "zc", Zone.ZC, "zp", Zone.ZP);
        }

        // les lignes doivent être triées par Zone pour l'algorithme
        // les lignes ZE, ZP, ZC dans cet ordre
        Collections.sort(lines, new Comparator<Object[]>() {
            @Override
            public int compare(Object[] o1, Object[] o2) {
                int result = -1;
                Zone o1Zone = ((Place)o1[1]).getZone();
                Zone o2Zone = ((Place)o2[1]).getZone();
                // fix Java 7 Comparison method violates its general contract!
                if (o1Zone == o2Zone) {
                    result = 0;
                } else {
                    switch (((Place)o1[1]).getZone()) {
                    case ZC:
                        result = 1;
                        break;
                    case ZP:
                        if (o2Zone == Zone.ZE) {
                            result = 1;
                        }
                        break;
                    }
                }
                return result;
            }
        });
        
        // les résultats doivent être trié par le code presentation pour le rendu
        Map<Presentation, PresentationModel> presentationCache = new HashMap<Presentation, PresentationModel>();
        for (Object[] line : lines) {
            Presentation presentation = (Presentation)line[0];
            Place place = (Place)line[1];
            long labelToZeCount = ((Number)line[2]).longValue();
            long labelToZpCount = ((Number)line[3]).longValue();
            long labelToZcCount = ((Number)line[4]).longValue();
            PresentationModel model = presentationCache.get(presentation);
            if (model == null) {
                model = new PresentationModel(presentation);
                presentationCache.put(presentation, model);
            }
            if ((place.getZone() == Zone.ZE && (presentation.isOriginal() || labelToZeCount > 0)) ||
                    (place.getZone() == Zone.ZP && ((model.getPlaces().isEmpty() && presentation.isOriginal()) || labelToZpCount > 0)) ||
                    (place.getZone() == Zone.ZC && ((model.getPlaces().isEmpty() && presentation.isOriginal()) || labelToZcCount > 0))) {
                model.addPlace(place);
            }
        }

        // FIXME echatellier 20121123, il y a un bug dans la requete précédente
        // elle ne retourne pas les présentations dans les emplacements ne peuvent
        // pas être déterminé :(
        // on corrige cela avec une seconde requete
        query = "select P" +
                " from " + Presentation.class.getName() + " P" +
                " where P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_INVALID + " = false" +
                " and P.batch.product not in (select PP.product FROM " + ProductPlace.class.getName() + " PP" +
                "  where PP.presentationCode = P.presentationCode)";
        List<Presentation> presentations = (List<Presentation>)context.findAll(query);
        for (Presentation presentation : presentations) {
            PresentationModel model = new PresentationModel(presentation);
            presentationCache.put(presentation, model);
        }
        // fin de la correction

        // build new result list sorted by presentation code
        List<PresentationModel> result = new ArrayList<PresentationModel>(presentationCache.values());
        Collections.sort(result, new Comparator<PresentationModel>() {
            @Override
            public int compare(PresentationModel o1, PresentationModel o2) {
                int result = o1.getPresentation().getPresentationCode().ordinal()
                        - o2.getPresentation().getPresentationCode().ordinal();
                return result;
            }
        });

        // ajout des emplacements d'inventaires
        for (PresentationModel model : result) {
            Presentation presentation = model.getPresentation();
            if (presentation.getInventoryPlaces() != null) {
                for (Place place : presentation.getInventoryPlaces()) {
                    if (!model.getPlaces().contains(place)) {
                        model.getPlaces().add(place);
                    }
                }
            }
        }
                
        return result;
    }
    
    /**
     * Retourne les presentations d'un lot ordonnées par le code presentation,
     * avec pour chaque présentation, ses emplacements théoriques disponibles.
     * 
     * Pour la détermination des emplacements:
     * <ul>
     * <li>plante qui arrive par ordre de présence des zones: ZE, ZP, ZC
     * <li>une plante est présente en ZC s'il y a un mouvement pour le client 99997
     * <li>une plante est présente en ZP s'il y a un mouvement pour un client
     * </ul>
     * 
     * @param batch batch
     * @return batch presentation
     * @throws TopiaException 
     */
    public List<PresentationModel> findAllByBatchOrderByPresentation(Batch batch) throws TopiaException {
        String query = "select P, PP.place," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :ze)," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :zp) + " +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_DESTINATION + " = :zp)," +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_DESTINATION + " = :zc)" +
                " from " + Presentation.class.getName() + " P," +
                " " + ProductPlace.class.getName() + " PP" +
                " where P." + Presentation.PROPERTY_BATCH + " = :batch" +
                " and P.batch.product = PP.product" +
                " and PP.presentationCode = P.presentationCode";

        // les lignes doivent être triées par Zone pour l'algorithme
        // les lignes ZE, ZP, ZC dans cet ordre
        List<Object[]> lines = (List<Object[]>)context.findAll(query, "batch", batch, "ze", Zone.ZE, "zp", Zone.ZP, "zc", Zone.ZC);
        Collections.sort(lines, new Comparator<Object[]>() {
            @Override
            public int compare(Object[] o1, Object[] o2) {
                int result = -1;
                Zone o1Zone = ((Place)o1[1]).getZone();
                Zone o2Zone = ((Place)o2[1]).getZone();
                // fix Java 7 Comparison method violates its general contract!
                if (o1Zone == o2Zone) {
                    result = 0;
                } else {
                    switch (((Place)o1[1]).getZone()) {
                    case ZC:
                        result = 1;
                        break;
                    case ZP:
                        if (o2Zone == Zone.ZE) {
                            result = 1;
                        }
                        break;
                    }
                }
                return result;
            }
        });

        // les résultats doivent être trié par le code presentation pour le rendu
        Map<Presentation, PresentationModel> presentationCache = new HashMap<Presentation, PresentationModel>();
        for (Object[] line : lines) { 
            Presentation presentation = (Presentation)line[0];
            Place place = (Place)line[1];
            long labelToZeCount = ((Number)line[2]).longValue();
            long labelToZpCount = ((Number)line[3]).longValue();
            long labelToZcCount = ((Number)line[4]).longValue();
            PresentationModel model = presentationCache.get(presentation);
            if (model == null) {
                model = new PresentationModel(presentation);
                presentationCache.put(presentation, model);
            }
            if ((place.getZone() == Zone.ZE && (presentation.isOriginal() || labelToZeCount > 0)) ||
                    (place.getZone() == Zone.ZP && ((model.getPlaces().isEmpty() && presentation.isOriginal()) || labelToZpCount > 0)) ||
                    (place.getZone() == Zone.ZC && ((model.getPlaces().isEmpty() && presentation.isOriginal()) || labelToZcCount > 0))) {
                model.addPlace(place);
            }
        }

        // FIXME echatellier 20121123, il y a un bug dans la requete précédente
        // elle ne retourne pas les présentations dans les emplacements ne peuvent
        // pas être déterminé :(
        // on corrige cela avec une seconde requete
        query = "select P" +
                " from " + Presentation.class.getName() + " P" +
                " where P." + Presentation.PROPERTY_BATCH + " = :batch" +
                " and P.batch.product not in (select PP.product FROM " + ProductPlace.class.getName() + " PP" +
                "  where PP.presentationCode = P.presentationCode)";
        List<Presentation> presentations = (List<Presentation>)context.findAll(query, "batch", batch);
        for (Presentation presentation : presentations) {
            PresentationModel model = new PresentationModel(presentation);
            presentationCache.put(presentation, model);
        }
        // fin de la correction

        // build new result list sorted by presentation code
        List<PresentationModel> result = new ArrayList<PresentationModel>(presentationCache.values());
        Collections.sort(result, new Comparator<PresentationModel>() {
            @Override
            public int compare(PresentationModel o1, PresentationModel o2) {
                int result = o1.getPresentation().getPresentationCode().ordinal()
                        - o2.getPresentation().getPresentationCode().ordinal();
                return result;
            }
        });
        
        // ajout des emplacements d'inventaires
        for (PresentationModel model : result) {
            Presentation presentation = model.getPresentation();
            if (presentation.getInventoryPlaces() != null) {
                for (Place place : presentation.getInventoryPlaces()) {
                    if (!model.getPlaces().contains(place)) {
                        model.getPlaces().add(place);
                    }
                }
            }
        }

        return result;
    }

    /**
     * Retourne les presentations d'un lot.
     * 
     * @param batch batch
     * @return batch presentation
     * @throws TopiaException 
     */
    public List<Presentation> findAllPresentationForbatch(Batch batch) throws TopiaException {
        String query = "select P" +
                " from " + Presentation.class.getName() + " P" +
                " where P." + Presentation.PROPERTY_BATCH + " = :batch";

        List<Presentation> result = (List<Presentation>)context.findAll(query, "batch", batch);
        return result;
    }

} //PresentationDAOImpl<E extends Presentation>
