/*
 * #%L
 * SGQ :: Business
 * $Id: AnalyzeImportModel.java 320 2013-04-17 08:26:15Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.2/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/csv/AnalyzeImportModel.java $
 * %%
 * Copyright (C) 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */
package com.herbocailleau.sgq.business.services.csv;

import static org.nuiton.i18n.I18n._;

import java.text.ParseException;
import java.util.Date;
import java.util.Map;

import org.nuiton.topia.TopiaException;
import org.nuiton.util.csv.ValueParser;
import org.nuiton.util.csv.ext.AbstractImportModel;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchAnalyze;
import com.herbocailleau.sgq.entities.BatchAnalyzeImpl;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.Supplier;
import com.herbocailleau.sgq.entities.SynthesisMention;

/**
 * Modèle d'import unique (reprise) du fichier des analyses.
 * 
 * N°Lot;Catégorie;Type;Qualification;Commentaire;Prestataire;Date d'envoi de l'échantillon;Date réception de l'échantillon
 * 
 * @author echatellier
 */
public class AnalyzeImportModel extends AbstractImportModel<BatchAnalyze> {

    public AnalyzeImportModel(final BatchDAO batchDAO, final Map<String, AnalyzeType> analyzeTypes,
            final Map<String, Supplier> suppliers) {
        super(';');
        
        newMandatoryColumn("N°Lot", BatchAnalyze.PROPERTY_BATCH, new ValueParser<Batch>() {
            @Override
            public Batch parse(String value) throws ParseException {
                int number = Integer.parseInt(value);
                Batch result = null;
                try {
                    result = batchDAO.findByNumber(number);
                } catch (TopiaException e) {
                    throw new SgqBusinessException("Can't get batch", e);
                }
                return result;
            }
        });
        
        newIgnoredColumn("Catégorie");
        
        newMandatoryColumn("Type", BatchAnalyze.PROPERTY_ANALYZE_TYPE, new ValueParser<AnalyzeType>() {
            @Override
            public AnalyzeType parse(String value) throws ParseException {
                AnalyzeType type = analyzeTypes.get(value);
                if (type == null) {
                    throw new SgqBusinessException(_("Le type d'analyze est obligatoire %s", value));
                }
                return type;
            }
        });
        newMandatoryColumn("Qualification", BatchAnalyze.PROPERTY_SYNTHESIS_MENTION, new ValueParser<SynthesisMention>() {
            @Override
            public SynthesisMention parse(String value) throws ParseException {
                SynthesisMention result;
                // Conforme, Acceptable, Non acceptable
                if ("Conforme".equals(value)) {
                    result = SynthesisMention.CONFORM;
                } else if ("Acceptable".equals(value)) {
                    result = SynthesisMention.ACCEPTABLE;
                } else if ("Non acceptable".equals(value)) {
                    result = SynthesisMention.NOT_ACCEPTABLE;
                } else {
                    throw new SgqBusinessException(_("Qualification de l'analyze inconnue", value));
                }
                return result;
            }
        });
        newMandatoryColumn("Commentaire", BatchAnalyze.PROPERTY_SYNTHESIS);
        newMandatoryColumn("Prestataire", BatchAnalyze.PROPERTY_SUPPLIER, new ValueParser<Supplier>() {
            @Override
            public Supplier parse(String value) throws ParseException {
                Supplier supplier = suppliers.get(value);
                if (supplier == null) {
                    throw new SgqBusinessException(_("Prestataire inconnu %s", value));
                }
                return supplier;
            }
        });
        newMandatoryColumn("Date d'envoi de l'échantillon", BatchAnalyze.PROPERTY_SENT_DATE, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                return SgqUtils.parseSgqDate(value);
            }
        });
        newMandatoryColumn("Date réception de l'échantillon", BatchAnalyze.PROPERTY_RECEIPT_DATE, new ValueParser<Date>() {
            @Override
            public Date parse(String value) throws ParseException {
                return SgqUtils.parseSgqDate(value);
            }
        });
    }

    @Override
    public BatchAnalyze newEmptyInstance() {
        return new BatchAnalyzeImpl();
    }
}
