/*
 * #%L
 * SGQ :: Business
 * $Id: BatchAnalyzeDAOImpl.java 258 2013-01-30 11:38:59Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.0/sgq-business/src/main/java/com/herbocailleau/sgq/entities/BatchAnalyzeDAOImpl.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;

import com.herbocailleau.sgq.business.model.AnalyzeFilter;

public class BatchAnalyzeDAOImpl<E extends BatchAnalyze> extends BatchAnalyzeDAOAbstract<E> {

    /**
     * Retourne l'ensemble des analyzes a preparer et non encore expédiées.
     * 
     * @param analyzeFilter filter to filter on
     * @throws TopiaException 
     */
    public List<BatchAnalyze> findAllAnalyzeToSend(AnalyzeFilter analyzeFilter) throws TopiaException {
        String query = "from " + BatchAnalyze.class.getName() +
            " where " + BatchAnalyze.PROPERTY_SENT_DATE + " is null";
        return findAllAnalyzeWithQuery(analyzeFilter, query);
    }

    /**
     * Retourne l'ensemble des analyzes expédiées en attente de reception.
     * 
     * @param analyzeFilter filter to filter on
     * @throws TopiaException 
     */
    public List<BatchAnalyze> findAllAnalyzeToReceive(AnalyzeFilter analyzeFilter) throws TopiaException {
        String query = "from " + BatchAnalyze.class.getName() +
            " where " + BatchAnalyze.PROPERTY_RECEIPT_DATE + " is null" +
            " and " + BatchAnalyze.PROPERTY_SENT_DATE + " is not null";
        return findAllAnalyzeWithQuery(analyzeFilter, query);
    }
    
    /**
     * Retourne l'ensemble des analyzes suivant un filtre.
     * 
     * @param analyzeFilter filter to filter on
     * @throws TopiaException 
     */
    public List<BatchAnalyze> findAllAnalyze(AnalyzeFilter analyzeFilter) throws TopiaException {
        String query = "from " + BatchAnalyze.class.getName() + " where 1=1";
        return findAllAnalyzeWithQuery(analyzeFilter, query);
    }
    
    /**
     * Retourne l'ensemble des analyzes en utilisant le filtre et le
     * debut de requete.
     * 
     * @param analyzeFilter filter
     * @param queryPrefix query prefix
     * @throws TopiaException 
     */
    protected List<BatchAnalyze> findAllAnalyzeWithQuery(AnalyzeFilter analyzeFilter, String queryPrefix) throws TopiaException {
        String query = queryPrefix;
        
        // build query and params
        List<Object> params = new ArrayList<Object>();
        if (analyzeFilter.getAnalyzeType() != null) {
            query += " and " + BatchAnalyze.PROPERTY_ANALYZE_TYPE + " = :analyzeType";
            params.add("analyzeType");
            params.add(analyzeFilter.getAnalyzeType());
        }
        if (analyzeFilter.getBatchNumberBegin() > 0) {
            query += " and :batchNumberBegin <= " + BatchAnalyze.PROPERTY_BATCH + "."  + Batch.PROPERTY_NUMBER;
            params.add("batchNumberBegin");
            params.add(analyzeFilter.getBatchNumberBegin());
        }
        if (analyzeFilter.getBatchNumberEnd() > 0) {
            query += " and " + BatchAnalyze.PROPERTY_BATCH + "."  + Batch.PROPERTY_NUMBER + " <= :batchNumberEnd";
            params.add("batchNumberEnd");
            params.add(analyzeFilter.getBatchNumberEnd());
        }
        if (StringUtils.isNotBlank(analyzeFilter.getSupplier())) {
            query += " and " + BatchDAO.getFieldLikeInsensitive(BatchAnalyze.PROPERTY_SUPPLIER +
                    "." + Supplier.PROPERTY_NAME, ":supplier"); 
            params.add("supplier");
            params.add("%" + StringUtils.stripAccents(analyzeFilter.getSupplier()) + "%");
        }
        if (analyzeFilter.getSentDateBegin() != null) {
            query += " and :datebegin <= dateadd('day', -analyzeType.durationDays, batch.dpmes)";
            params.add("datebegin");
            params.add(analyzeFilter.getSentDateBegin());
        }
        if (analyzeFilter.getSentDateEnd() != null) {
            query += " and dateadd('day', -analyzeType.durationDays, batch.dpmes) <= :dateend";
            params.add("dateend");
            params.add(analyzeFilter.getSentDateEnd());
        }
        query += " order by " + BatchAnalyze.PROPERTY_BATCH + "." + Batch.PROPERTY_DPMES;
        query += " , " + BatchAnalyze.PROPERTY_BATCH + "." + Batch.PROPERTY_NUMBER;
        query += " , " + TopiaEntity.TOPIA_ID;

        // execute query
        List<BatchAnalyze> result = context.findAll(query, params.toArray());
        
        return result;
    }

} //AnalyzeDAOImpl<E extends Analyze>
