/*
 * #%L
 * SGQ :: Business
 * $Id: ReferentialService.java 385 2013-06-04 09:59:12Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.0/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ReferentialService.java $
 * %%
 * Copyright (C) 2012, 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.csv.AbstractImportErrorInfo;
import org.nuiton.util.csv.Import2;
import org.nuiton.util.csv.ImportConf;
import org.nuiton.util.csv.ImportRow;

import au.com.bytecode.opencsv.CSVReader;
import au.com.bytecode.opencsv.CSVWriter;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.model.ImportLog;
import com.herbocailleau.sgq.business.model.ProductSearchModel;
import com.herbocailleau.sgq.business.services.csv.ClientImportModel;
import com.herbocailleau.sgq.business.services.csv.PlaceBean;
import com.herbocailleau.sgq.business.services.csv.PlaceImportModel;
import com.herbocailleau.sgq.business.services.csv.SupplierImportModel;
import com.herbocailleau.sgq.entities.AnalyzeCategory;
import com.herbocailleau.sgq.entities.AnalyzeCategoryDAO;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.AnalyzeTypeDAO;
import com.herbocailleau.sgq.entities.Client;
import com.herbocailleau.sgq.entities.ClientDAO;
import com.herbocailleau.sgq.entities.Place;
import com.herbocailleau.sgq.entities.PlaceDAO;
import com.herbocailleau.sgq.entities.Product;
import com.herbocailleau.sgq.entities.ProductDAO;
import com.herbocailleau.sgq.entities.ProductPlace;
import com.herbocailleau.sgq.entities.ProductPlaceDAO;
import com.herbocailleau.sgq.entities.ProductStatus;
import com.herbocailleau.sgq.entities.ProductStatusDAO;
import com.herbocailleau.sgq.entities.Supplier;
import com.herbocailleau.sgq.entities.SupplierDAO;
import com.herbocailleau.sgq.entities.Zone;

/**
 * Service gérant les elements faisant partit du référentiel du SGQ.
 * 
 * À savoir:
 * <ul>
 * <li>les produits
 * <li>les clients
 * <li>les emplacements
 * <li>les fournisseurs
 * <li>les analyses et catégories d'analyses
 * </ul>
 * 
 * @author echatellier
 */
public class ReferentialService extends SgqService {

    private static final Log log = LogFactory.getLog(ReferentialService.class);

    /**
     * Import client dbf file.
     * 
     * @param file dbf file
     */
    public List<ImportLog<Client>> importClients(File file) {
        
        if (log.isInfoEnabled()) {
            log.info("Importing client from file " + file.getAbsolutePath());
        }

        List<ImportLog<Client>> importLogs = new ArrayList<ImportLog<Client>>();

        Reader reader = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            reader = new InputStreamReader(new FileInputStream(file), config.getImportCsvFileEncoding());
            ClientImportModel model = new ClientImportModel();

            boolean noErrors = true;
            ImportConf conf = new ImportConf();
            conf.setStrictMode(false);
            Import2<Client> importCsv = Import2.newImport(conf, model, reader);

            for (ImportRow<Client> row : importCsv) {

                Client bean = row.getBean();
                if (row.isValid()) {
                    ImportLog<Client> importLog = new ImportLog<Client>();
                    
                    Client client = clientDAO.findByCode(bean.getCode());
                    if (client == null) {
                        client = clientDAO.create(Client.PROPERTY_CODE, bean.getCode());
                        importLog.setMessage(_("Creation d'un nouveau client"));
                    } else {
                        importLog.setMessage(_("Mise à jour du client"));
                    }
                    client.setName(bean.getName());

                    importLog.setBean(client);
                    importLog.setLine(row.getLineNumber());
                    importLog.setCode(client.getCode());
                    importLogs.add(importLog);
                } else {
                    noErrors = false;
                    Set<AbstractImportErrorInfo<Client>> rowErrors = row.getErrors();
                    for (AbstractImportErrorInfo<Client> rowError : rowErrors) {
                        ImportLog<Client> importLog = new ImportLog<Client>(true);
                        importLog.setLine(row.getLineNumber());
                        importLog.setCode(row.getBean().getCode());
                        importLog.setMessage(rowError.getCause().getMessage());
                        importLogs.add(importLog);
                    }
                }
            }

            if (noErrors) {
                daoHelper.commit();
            }
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't commit modification in database", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't read csv file", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }
        
        return importLogs;
    }

    public List<ImportLog<Product>> importProducts(File file) {
        if (log.isInfoEnabled()) {
            log.info("Importing product from file " + file.getAbsolutePath());
        }
 
        List<ImportLog<Product>> importLogs = new ArrayList<ImportLog<Product>>();
        CSVReader csvReader = null;
        try {

            ProductDAO productDAO = daoHelper.getProductDAO();
            ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();

            csvReader = new CSVReader(new InputStreamReader(new FileInputStream(file),
                    config.getImportCsvFileEncoding()), ';');

            // Code Produit;Nom Produit;Catégorie;Désignation Latine;Famille
            // ensuite, dynamiquement
            // des statuts et des analyses a déterminer suivant le meta header
            // sauf les header commencant par "Commentaire"
            // enfin, controle identification et biologique
            // en un commentaire final
            String[] metaHeaders = csvReader.readNext();
            String[] headers = csvReader.readNext();

            // check headers
            String[] mandatoryHeaders = ArrayUtils.subarray(headers, 0, 5);
            String[] expHeaders = new String[] {
                    "Code Produit", "Nom Produit", "Cat\u00e9gorie", "Désignation Latine", "Famille"};
            if (!Arrays.equals(mandatoryHeaders, expHeaders)) {
                throw new SgqBusinessException(_("Le fichier ne contient pas au moins les entetes obligatoires %s, trouvés %s",
                        StringUtils.join(expHeaders, ';'), StringUtils.join(mandatoryHeaders, ';'))); 
            }

            String[] row = null;
            while ((row = csvReader.readNext()) != null) {

                // la derniere ligne peut être vide
                if (row.length <= 1) {
                    continue;
                }

                String code = row[0];
                Product product = productDAO.create(Product.PROPERTY_CODE, code);
                product.setName(row[1]);
                product.setCategory(row[2]);
                product.setLatinName(row[3]);
                product.setFamily(row[4]);

                StringBuilder commentaires = new StringBuilder();
                for (int index = 5 ; index < row.length ; index++) {
                    String data = row[index].trim();
                    String metaHeader = metaHeaders[index].trim();
                    String header = headers[index].trim();

                    // dynamic status case
                    if (header.startsWith("Commentaire")) {
                        if (StringUtils.isNotBlank(data)) {
                            commentaires.append(header + "\n");
                            commentaires.append("-----------\n");
                            commentaires.append(data + "\n");
                        }
                    } else if (header.startsWith("Lieu d'analyse ")) {
                        // skip this one
                    } else if ("statut".equalsIgnoreCase(metaHeader)) {
                        if (data.equalsIgnoreCase("Oui") || data.equalsIgnoreCase("Oui avec limite")) {
                            ProductStatus status = productStatusDAO.findByName(header);
                            if (status == null) {
                                status = productStatusDAO.create();
                                status.setName(header);
                            }
                            product.addProductStatus(status);
                            if (data.equalsIgnoreCase("Oui avec limite")) {
                                commentaires.append(header + " : " + data + "\n");
                            }
                        }
                    } else if ("analyse".equalsIgnoreCase(metaHeader)) {
                        String[] analyzes = data.split("\\s*[,;]\\s*");
                        for (String analyze : analyzes) {
                            if (!analyze.isEmpty()) {
                                AnalyzeType analyzeType = analyzeTypeDAO.findByName(analyze);
                                if (analyzeType == null) {
                                    AnalyzeCategory analyzeCategory = analyzeCategoryDAO.findByName(header);
                                    if (analyzeCategory == null) {
                                        analyzeCategory = analyzeCategoryDAO.create();
                                        analyzeCategory.setName(header);
                                    }
                                    analyzeType = analyzeTypeDAO.create();
                                    analyzeType.setName(analyze);
                                    analyzeType.setAnalyzeCategory(analyzeCategory);
                                    // config par defaut : 30 jours
                                    analyzeType.setDurationDays(30);
                                }
                                product.addAnalyzeType(analyzeType);
                            }
                        }
                        
                    } else if ("contrôle".equalsIgnoreCase(metaHeader)) {
                        if (data.equalsIgnoreCase("Oui")) {
                            if ("Botanique".equalsIgnoreCase(header)) {
                                product.setBotanicControl(true);
                            } else if ("Identification".equalsIgnoreCase(header)) {
                                product.setIdentificationControl(true);
                            }
                        }
                    }
                }
                product.setComment(commentaires.toString());
                productDAO.update(product);
            }

            csvReader.close();

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import product file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import product file", ex);
        } finally {
            IOUtils.closeQuietly(csvReader);
        }
        
        return importLogs;
    }

    /**
     * Import place csv file.
     * 
     * @param file csv file
     * @return import log
     */
    public List<ImportLog<Place>> importPlaces(File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing places from file " + file.getAbsolutePath());
        }

        List<ImportLog<Place>> importLogs = new ArrayList<ImportLog<Place>>();

        Reader reader = null;
        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            ProductPlaceDAO productPlaceDAO = daoHelper.getProductPlaceDAO();

            // add some cache
            ProductDAO productDAO = daoHelper.getProductDAO();
            Map<String, Product> products = new HashMap<String, Product>();
            for (Product product : productDAO.findAll()) {
                products.put(product.getCode(), product);
            }

            reader = new InputStreamReader(new FileInputStream(file), config.getImportCsvFileEncoding());
            PlaceImportModel model = new PlaceImportModel(products);

            ImportConf conf = new ImportConf();
            conf.setStrictMode(false);
            Import2<PlaceBean> importCsv = Import2.newImport(conf, model, reader);

            // remove all product places and place
            productPlaceDAO.deleteAll();
            placeDAO.deleteAll();

            for (ImportRow<PlaceBean> row : importCsv) {

                PlaceBean bean = row.getBean();
                String placeCode = bean.getCode();

                if (row.isValid()) {

                    ImportLog<Place> importLog = new ImportLog<Place>();
                    importLog.setLine(row.getLineNumber());
                    importLog.setCode(placeCode);

                    // save place bean
                    Place place = placeDAO.findByCode(placeCode);
                    if (place == null) {
                        place = placeDAO.createByNaturalId(placeCode);
                        importLog.setMessage(_("L'emplacement %s a été créé dans le référentiel pour la plante %s et sa présentation %c",
                                placeCode, bean.getProduct().getCode(), bean.getPresentationCode().getCode()));
                    } /* Aucun message car on affiche tout ensuite
                    else {
                        importLog.setMessage(_("Mise à jour de l'emplacement : %s", placeCode));
                    }*/
                    importLog.setBean(place);
                    place.setZone(bean.getZone());
                    place.setName(bean.getName());
                    place = placeDAO.update(place);

                    // enregistement du produit dans l'emplacement
                    ProductPlace productPlace = productPlaceDAO.create();
                    productPlace.setPlace(place);
                    productPlace.setProduct(bean.getProduct());
                    productPlace.setPresentationCode(bean.getPresentationCode());

                    importLogs.add(importLog);
                } else {
                    Set<AbstractImportErrorInfo<PlaceBean>> rowErrors = row.getErrors();
                    for (AbstractImportErrorInfo<PlaceBean> rowError : rowErrors) {
                        ImportLog<Place> importLog = new ImportLog<Place>(true);
                        importLog.setLine(row.getLineNumber());
                        importLog.setCode(placeCode);
                        importLog.setMessage(rowError.getCause().getMessage());
                        importLogs.add(importLog);
                    }
                }
            }

            // on ne commit que s'il n'y a aucune erreur
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't commit modification in database", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import csv file", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }

        return importLogs;
    }

    /**
     * Import suppliers dbf file.
     * 
     * @param file dbf file
     */
    public List<ImportLog<Supplier>> importSuppliers(File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing suppliers from file " + file.getAbsolutePath());
        }

        List<ImportLog<Supplier>> importLogs = new ArrayList<ImportLog<Supplier>>();

        Reader reader = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();

            reader = new InputStreamReader(new FileInputStream(file), config.getImportCsvFileEncoding());
            SupplierImportModel model = new SupplierImportModel();
            ImportConf conf = new ImportConf();
            conf.setStrictMode(false);
            Import2<Supplier> importCsv = Import2.newImport(conf, model, reader);

            for (ImportRow<Supplier> row : importCsv) {

                Supplier bean = row.getBean();
                String supplierCode = bean.getCode();

                if (row.isValid()) {
                    ImportLog<Supplier> importLog = new ImportLog<Supplier>();
                    importLog.setLine(row.getLineNumber());
                    importLog.setCode(supplierCode);

                    Supplier supplier = supplierDAO.findByCode(supplierCode);
                    if (supplier == null) {
                        supplier = supplierDAO.createByNaturalId(bean.getCode());
                        importLog.setMessage(_("Creation d'un nouveau fournisseur : %s", supplierCode));
                    } else {
                        importLog.setMessage(_("Mise à jour du fournisseur : %s", supplierCode));
                    }
                    supplier.setName(bean.getName());
                    supplier.setLaboratory(bean.isLaboratory());
                    supplierDAO.update(supplier);

                    importLogs.add(importLog);
                } else {
                    Set<AbstractImportErrorInfo<Supplier>> rowErrors = row.getErrors();
                    for (AbstractImportErrorInfo<Supplier> rowError : rowErrors) {
                        ImportLog<Supplier> importLog = new ImportLog<Supplier>(true);
                        importLog.setLine(row.getLineNumber());
                        importLog.setCode(supplierCode);
                        importLog.setMessage(rowError.getCause().getMessage());
                        importLogs.add(importLog);
                    }
                }
            }

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't read csv file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't commit modification in database", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }
        
        return importLogs;
    }

    public long getClientsCount() {
        long result = 0;

        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            result = clientDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get clients ordered by code.
     * 
     * @param offset offset
     * @param count count
     * @return count clients
     */
    public List<Client> getClients(int offset, int count) {
        List<Client> results = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            results = clientDAO.findAllByQueryWithBound("from " + Client.class.getName() +
                    " order by " + Client.PROPERTY_CODE, offset, offset + count - 1);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }

    public Client findClientById(String id) {
        Client result = null;
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            result = clientDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get client", ex);
        }
        return result;
    }

    public void saveClient(Client client) {
        try {
            ClientDAO clientDAO = daoHelper.getClientDAO();
            if (StringUtils.isEmpty(client.getTopiaId())) {
                clientDAO.create(client);
            } else {
                clientDAO.update(client);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save client", ex);
        }
        
    }

    public long getPlacesCount() {
        long result = 0;

        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            result = placeDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get places ordered by code.
     * 
     * @return all places
     */
    public List<Place> getPlaces() {
        return getPlaces(null);
    }
    
    /**
     * Get places ordered by code for zone.
     * 
     * @param zone zone filter
     * @return all places
     */
    public List<Place> getPlaces(Zone zone) {
        List<Place> results = null;
        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            results = placeDAO.findAllForZone(zone);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get places", ex);
        }
        return results;
    }

    public Place getPlaceById(String placeId) {
        Place result = null;
        try {
            PlaceDAO placeDAO = daoHelper.getPlaceDAO();
            result = placeDAO.findByTopiaId(placeId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get place", ex);
        }
        return result;
    }

    /**
     * Utile seulement pour les tests.
     * 
     * @param code place code
     * @return place
     */
    protected Place getPlaceByCode(String code) {
        Place result = null;
        try {
            PlaceDAO productDAO = daoHelper.getPlaceDAO();
            result = productDAO.findByCode(code);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get place", ex);
        }
        return result;
    }

    public long getSuppliersCount() {
        long result = 0;

        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get suppliers ordered by code.
     * 
     * @param offset offset
     * @param count count
     * @return count suppliers
     */
    public List<Supplier> getSuppliers(int offset, int count) {
        List<Supplier> results = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            results = supplierDAO.findAllByQueryWithBound("from " + Supplier.class.getName() +
                    " order by " + Supplier.PROPERTY_CODE, offset, offset + count - 1);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return results;
    }
    
    /**
     * Get supplier by topia id.
     * 
     * @param offset offset
     * @param count count
     * @return count suppliers
     */
    public Supplier getSupplierById(String supplierId) {
        Supplier result = null;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.findByTopiaId(supplierId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get supplier", ex);
        }
        return result;
    }

    public long getProductsCount() {
        long result = 0;

        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = productDAO.count();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }

        return result;
    }

    /**
     * Get product ordered by code.
     * 
     * @param search search product search model
     * @param offset offset
     * @param count count
     * @return count products
     */
    public Pair<List<Product>, Long> getProducts(ProductSearchModel search, int offset, int count) {
        Pair<List<Product>, Long> result = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            List<Product> products = productDAO.findAllModel(search, offset, count);
            long totalCount = productDAO.findAllCount(search);
            result = Pair.of(products, totalCount);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        }
        return result;
    }
    
    /**
     * Get product export as csv
     * 
     * @param search search product search model
     */
    public InputStream getProductsAsCsv(ProductSearchModel search) {
        InputStream result = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            List<Product> products = productDAO.findAllModel(search);
            
            File file = File.createTempFile("sgq-search", ".csv");
            file.deleteOnExit();

            CSVWriter csvWriter = new CSVWriter(new FileWriter(file), ';');

            // write headers
            List<String> headers = new ArrayList<String>();
            headers.add("Code");
            headers.add("Nom");
            headers.add("Catégorie");
            headers.add("Nom latin");
            headers.add("Famille");
            headers.add("Statuts");
            headers.add("Analyses");
            headers.add("Contrôle");
            headers.add("Commentaire");
            csvWriter.writeNext(headers.toArray(new String[headers.size()]));

            // write elements
            for (Product product : products) {
                List<String> data = new ArrayList<String>(headers.size());
                // code
                data.add(product.getCode());
                data.add(product.getName());
                data.add(product.getCategory());
                data.add(product.getLatinName());
                data.add(product.getFamily());
                
                StringBuffer statusSB = new StringBuffer();
                String separator = "";
                if (product.getProductStatus() != null) {
                    for (ProductStatus productStatus : product.getProductStatus()) {
                        statusSB.append(separator);
                        statusSB.append(productStatus.getName());
                        separator = ", ";
                    }
                }
                data.add(statusSB.toString());

                StringBuffer analyzeTypeSB = new StringBuffer();
                separator = "";
                if (product.getAnalyzeType() != null) {
                    for (AnalyzeType analyzeType : product.getAnalyzeType()) {
                        analyzeTypeSB.append(separator);
                        analyzeTypeSB.append(analyzeType.getName());
                        separator = ", ";
                    }
                }
                data.add(analyzeTypeSB.toString());

                // this is hard to do with nuiton-csv
                String control = "";
                if (product.isBotanicControl()) {
                    control = "Botanique";
                }
                if (product.isIdentificationControl()) {
                    if (!control.isEmpty()) {
                        control += ", ";
                    }
                    control += "Identification";
                }
                data.add(control);
                data.add(product.getComment());
                
                csvWriter.writeNext(data.toArray(new String[data.size()]));
            }

            csvWriter.close();
            
            result = new FileInputStream(file);
            
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get count", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't output csv file", ex);
        }
        return result;
    }

    public Product getProductById(String productId) {
        Product result = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = productDAO.findByTopiaId(productId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get product", ex);
        }
        return result;
    }

    public void saveProduct(Product product, String productStatusName) {
        try {
            
            String productCode = product.getCode();

            // check empty
            if (StringUtils.isBlank(productCode)) {
                throw new SgqBusinessException(_("Impossible de créer un produit avec un code vide"));
            }

            // Imposition des majuscules pour le code produit
            productCode = productCode.toUpperCase();
            product.setCode(productCode);

            if (StringUtils.isNotEmpty(productStatusName)) {
                ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
                ProductStatus productStatus = productStatusDAO.findByName(productStatusName);
                if (productStatus == null) {
                    productStatus = productStatusDAO.create();
                    productStatus.setName(productStatusName);
                }
                product.addProductStatus(productStatus);
            }

            ProductDAO productDAO = daoHelper.getProductDAO();
            if (StringUtils.isEmpty(product.getTopiaId())) {

                // duplication check
                if (productDAO.findByCode(productCode) != null) {
                    throw new SgqBusinessException(_("Impossible de créer un produit avec le code déjà utilisé %s", productCode));
                }

                productDAO.create(product);
            } else {
                productDAO.update(product);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save product", ex);
        }
    }

    /**
     * Test if current product is not used into 'emplacements' and 'batch' table
     * to allow deletion.
     * 
     * @param product product to test
     * @return {@code true} if product can be delete
     */
    public boolean canBeDeleteProduct(Product product) {
        boolean result;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = !productDAO.isUsedByAnyBatch(product);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete product", ex);
        }
        return result;
    }

    /**
     * Test if product is not currently used by a non expired batch to
     * be archived.
     * 
     * @param product product to test
     * @return {@code true} if product can be delete
     */
    public boolean canBeArchivedProduct(Product product) {
        boolean result;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = !productDAO.isUsedByNonExpiredBatch(product);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete product", ex);
        }
        return result;
    }

    /**
     * Delete product.
     * 
     * @param product product to delete
     */
    public void deleteProduct(Product product) {
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            productDAO.delete(product);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete product", ex);
        }
    }

    /**
     * Archive product.
     * 
     * @param product product to delete
     */
    public void archiveProduct(Product product) {
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            product.setArchived(true);
            productDAO.update(product);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't archive product", ex);
        }
    }

    /**
     * Unarchive product.
     * 
     * @param product product to delete
     */
    public void unArchiveProduct(Product product) {
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            product.setArchived(false);
            productDAO.update(product);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't unarchive product", ex);
        }
    }

    /**
     * Find all analyze types.
     * 
     * @return all analyze types
     */
    public List<AnalyzeType> findAllAnalyzeTypes() {
        List<AnalyzeType> result = null;
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            result = analyzeTypeDAO.findAllWithOrder(AnalyzeType.PROPERTY_ANALYZE_CATEGORY +
                    "." + AnalyzeCategory.PROPERTY_NAME, AnalyzeType.PROPERTY_NAME);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze types", ex);
        }
        return result;
    }

    /**
     * Find all analyze type with usage count.
     * 
     * @return map with count for each type
     */
    public Map<AnalyzeType, Long> findAllAnalyzeTypesWithCount() {
        Map<AnalyzeType, Long> result = null;
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            result = analyzeTypeDAO.findAllWithUsageCount();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze types", ex);
        }
        return result;
    }

    /**
     * Find single analyze types by id.
     * 
     * @return analyze type
     */
    public AnalyzeType findAnalyzeTypeById(String id) {
        AnalyzeType result = null;
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            result = analyzeTypeDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze type", ex);
        }
        return result;
    }

    /**
     * Delete analyze type.
     * 
     * @return analyze type to delete
     */
    public void deleteAnalyzeType(AnalyzeType analyzeType) {
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            analyzeTypeDAO.delete(analyzeType);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete analyze type", ex);
        }
    }

    /**
     * Save (create or update) analyze type.
     */
    public void saveAnalyzeType(AnalyzeType analyzeType) {
        try {
            AnalyzeTypeDAO analyzeTypeDAO = daoHelper.getAnalyzeTypeDAO();
            if (StringUtils.isEmpty(analyzeType.getTopiaId())) {
                analyzeTypeDAO.create(analyzeType);
            } else {
                analyzeTypeDAO.update(analyzeType);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save analyze type", ex);
        }
    }

    /**
     * Save (create or update) analyze category.
     */
    public void saveAnalyzeCategory(AnalyzeCategory analyzeCategory) {
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            if (StringUtils.isEmpty(analyzeCategory.getTopiaId())) {
                analyzeCategoryDAO.create(analyzeCategory);
            } else {
                analyzeCategoryDAO.update(analyzeCategory);
            }
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't save analyze category", ex);
        }
    }

    /**
     * Find all analyze categories.
     * 
     * @return
     */
    public List<AnalyzeCategory> findAllAnalyzeCategories() {
        List<AnalyzeCategory> result = null;
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            result = analyzeCategoryDAO.findAllWithOrder(AnalyzeCategory.PROPERTY_NAME);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze categories", ex);
        }
        return result;
    }

    /**
     * Find all analyze categories with usage count.
     * 
     * @return map with count for each category
     */
    public Map<AnalyzeCategory, Long> findAllAnalyzeCategoriesWithCount() {
        Map<AnalyzeCategory, Long> result = null;
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            result = analyzeCategoryDAO.findAllWithUsageCount();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze categories", ex);
        }
        return result;
    }
    
    public AnalyzeCategory findAnalyzeCategoryById(String id) {
        AnalyzeCategory result = null;
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            result = analyzeCategoryDAO.findByTopiaId(id);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze category", ex);
        }
        return result;
    }
    
    public void deleteAnalyzeCategory(AnalyzeCategory analyzeCategory) {
        try {
            AnalyzeCategoryDAO analyzeCategoryDAO = daoHelper.getAnalyzeCategoryDAO();
            analyzeCategoryDAO.delete(analyzeCategory);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete analyze category", ex);
        }
    }

    /**
     * Return all suppliers (laboratory first).
     * 
     * @return all supplier
     */
    public List<Supplier> getAllSupplierOnlyLabs() {
        List<Supplier> result;
        try {
            SupplierDAO supplierDAO = daoHelper.getSupplierDAO();
            result = supplierDAO.findAllOnlyLabs();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get analyze category", ex);
        }
        return result;
    }

    public List<ProductStatus> findAllProductStatus() {
        List<ProductStatus> result;
        try {
            ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
            result = productStatusDAO.findAllWithOrder(ProductStatus.PROPERTY_NAME +" ASC");
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get product status", ex);
        }
        return result;
    }

    /**
     * Find all product status with product count using this status.
     * 
     * @return product status with count
     */
    public Map<ProductStatus, Long> findAllProductStatusWithCount() {
        Map<ProductStatus, Long> result;
        try {
            ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
            result = productStatusDAO.findAllWithProductCount();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get product status", ex);
        }
        return result;
    }

    public ProductStatus getProductStatusById(String productStatusId) {
        ProductStatus result = null;
        try {
            ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
            result = productStatusDAO.findByTopiaId(productStatusId);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get product status", ex);
        }
        return result;
    }

    public void deleteProductStatus(ProductStatus productStatus) {
        try {
            ProductStatusDAO productStatusDAO = daoHelper.getProductStatusDAO();
            productStatusDAO.delete(productStatus);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't delete product status", ex);
        }
    }

    /**
     * Recherche parmis tous les produits les categories existantes.
     * 
     * @return les categories utilisées par les produits
     */
    public List<String> findAllProductCategories() {
        List<String> result = null;
        try {
            ProductDAO productDAO = daoHelper.getProductDAO();
            result = productDAO.findDistinctCategories();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get product categories", ex);
        }
        return result;
    }
}
