/*
 * #%L
 * SGQ :: Business
 * $Id: ProductionService.java 143 2012-10-08 15:35:13Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.4/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ProductionService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;

import au.com.bytecode.opencsv.CSVReader;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.business.model.ImportLog;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.Expedition;
import com.herbocailleau.sgq.entities.ExpeditionDAO;
import com.herbocailleau.sgq.entities.Presentation;
import com.herbocailleau.sgq.entities.PresentationCode;
import com.herbocailleau.sgq.entities.PresentationDAO;
import com.herbocailleau.sgq.entities.Production;
import com.herbocailleau.sgq.entities.ProductionDAO;
import com.herbocailleau.sgq.entities.Zone;

/**
 * Service gerant principalement:
 * 
 * <ul>
 * <li>Les ventes et transformations des présentations des lots
 * </ul>
 * 
 * @author echatellier
 */
public class ProductionService extends SgqService {

    private static final Log log = LogFactory.getLog(ProductionService.class);

    /**
     * Retourne la configuration pour obtenir les dates de dernière
     * mise à jour (import expedition/fabrication).
     * 
     * @return la configuration (can be null on first import)
     */
    public Date getLastImportDate(Zone zone) {
        Date result = null;
        try {
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();
            result = expeditionDAO.findMaxDateForZone(zone);

            Date result2 = productionDAO.findMaxDateForZone(zone);
            if (result2 != null && (result == null || result.before(result2))) {
                result = result2;
            }
 
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get configuration", ex);
        }
        return result;
    }

    /**
     * Import label file.
     * Use file name to import production file or expedition file.
     * 
     * Le fichier est structuré ainsi:
     * D,20110701,B,147136,C,11520,P,PFEN21,L,20036,Q,40.000
     * 
     * D : Date 
     * B : N° de Bon de commande interne
     * C : Code client
     * P : Code présentation + Code plante
     * L : N° de lot
     * Q : Quantité
     * R : Remplacement lot
     * 
     * La methode fait un grand taille, mais elle n'est pas evident à diviser.
     * 
     * Champs a renseigner :
     * <ul>
     * <li>sur le lot: DMESD = Date de Mise En Service Détaillant : date de la première vente
     *    du lot quelques soit sa présentation mais au niveau de l'atelier des expéditions.
     * <li>sur la presentation : une plante est présente en ZC s'il y a un
     *   mouvement pour le client 99997 : valoraisation de packaging à 'true'
     * </ul>
     * 
     * @param fileFilename file name (FIF_HIST.txt or FIC_HIST.txt allowed)
     * @param file file
     */
    public List<ImportLog> importLabelsFile(String fileFilename, File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing labels file : " + file);
        }

        // test filename to get correct last import date
        // if file in not FIF_HIST.txt or FIC_HIST.txt
        // an exception is thrown
        String productionFileName = config.getLabelProductionFilename();
        Zone sourceZone = null;
        Date lastImport = null;
        if (productionFileName.equalsIgnoreCase(fileFilename)) {
            sourceZone = Zone.ZE;
            lastImport = getLastImportDate(Zone.ZE);
        } else {
            String expeditionFileName = config.getLabelExpeditionFilename();
            if (expeditionFileName.equalsIgnoreCase(fileFilename)) {
                sourceZone = Zone.ZP;
                lastImport = getLastImportDate(Zone.ZP);
            } else {
                throw new SgqBusinessException(_("Nom de fichier '%s' invalide ! (Autorisé %s ou %s)",
                        fileFilename, productionFileName, expeditionFileName));
            }
        }

        // cas du premier import
        if (lastImport == null) {
            lastImport = new Date(0); // pour que l'algo fonctionne
        }
        // date du dernier import + 1 à minuit
        Date importFrom = DateUtils.addDays(lastImport, 1);
        importFrom = DateUtils.truncate(importFrom, Calendar.DAY_OF_MONTH);

        // date du jour a minuit
        Date today = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH);

        // date du dernier commit pour commiter a chaque changement de jour
        // si necessaire
        Date lastCommitDate = null;

        // performing import
        List<ImportLog> importLogs = new ArrayList<ImportLog>();
        boolean errorSpotted = false;
        CSVReader reader = null;
        try {

            reader = new CSVReader(new FileReader(file));
            // le fichier recu n'est pas ordonné sur les dates alors que l'algo
            // est basé sur l'ordre pour commité seulement les jours complets
            // il faut charger le fichier en mémoire et le trier avant
            // d'exploiter les données
            List<String[]> datas = new ArrayList<String[]>();
            String[] csvdata = null;
            while ((csvdata = reader.readNext()) != null) {
                if (csvdata.length <= 1) {
                    continue;
                }
                
                // ajout d'un check pour verifier que le fichier a la forme
                // attendue, notement au niveau des prefix de colonnes
                String columns = "";
                for (int i = 0; i < csvdata.length ; i+=2) {
                    columns += csvdata[i];
                }
                if (!"DBCPLQ".equals(columns)) {
                    throw new SgqBusinessException(_("Nom de colonne invalides; trouvé : %s, attendu : %s",
                            columns, "DBCPLQ"));
                }

                datas.add(csvdata);
            }
            reader.close();

            // sort on date (yyyyMMdd, sort on string is enought)
            Collections.sort(datas, new Comparator<String[]>() {
                @Override
                public int compare(String[] o1, String[] o2) {
                    // data in at index 1
                    return o1[1].compareTo(o2[1]);
                }
            });

            // get date format
            DateFormat dfIn = new SimpleDateFormat("yyyyMMdd");
            DateFormat dfOut = new SimpleDateFormat("dd/MM/yyyy");

            // get dao
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();

            int line = 0;
            for (String[] data : datas) {
                line++;

                String dateddMMyy = data[1];
                String clientId = data[5];
                String productId = data[7];
                String batchId = data[9];
                String quantityStr = data[11];

                Date lineDate = dfIn.parse(dateddMMyy);

                // controle des dates, import entre le last import
                // et la veille de la date du jour
                if (lineDate.before(importFrom)) {
                    if (log.isDebugEnabled()) {
                        log.debug(_("Skipping date %s, before %s", lineDate.toString(),
                                importFrom.toString()));
                    }
                    continue;
                }

                // le commit est un peu particulier il faut commiter toutes
                // les entités créer sur une journée ssi il n'y a pas eu
                // d'erreur d'ici là depuis le debut de l'import
                // il ne faut surtout pas commiter une journée non complete
                // et encore moins les journées suivantes
                // l'import remprendra depuis la date de dernier jour
                // correctement importé
                if (lastCommitDate == null) {
                    lastCommitDate = lineDate;
                }
                if (!errorSpotted) {
                    if (!DateUtils.isSameDay(lineDate, lastCommitDate)) {

                        // unique commit place othersize topia filter
                        // will rollback transaction
                        daoHelper.commit();

                        lastCommitDate = lineDate;
                    }
                }

                // les lignes suivantes donne lieu a une entrée de log
                ImportLog importLog = new ImportLog();
                importLog.setLine(line);
                importLog.setCode(dfOut.format(lineDate));
                importLogs.add(importLog);

                // modification du message d'erreur
                if (errorSpotted) {
                    importLog.setError(errorSpotted);
                    importLog.setMessage(_("Ligne ignorée pour les erreurs précédentes"));
                    continue;
                }

                // check de la date du jour
                if (!lineDate.before(today)) { // pas le meme test que date.after !!!
                    if (log.isDebugEnabled()) {
                        log.debug(_("Skipping date of current day %s", lineDate.toString()));
                    }
                    importLog.setError(true);
                    importLog.setMessage(_("Ligne ignorée (date du jour)"));
                    errorSpotted = true;
                    continue;
                }

                Batch batch = batchDAO.findByNumber(Integer.parseInt(batchId));
                if (batch == null) {
                    importLog.setError(true);
                    importLog.setMessage(_("Impossible de trouver le lot %s", batchId));
                    errorSpotted = true;
                    continue;
                }

                char presentationChar = productId.charAt(0);
                productId = productId.substring(1);
                PresentationCode presentationCode = PresentationCode.getPresentationCodeFor(presentationChar);
                double quantity = Double.parseDouble(quantityStr);
                int client = Integer.parseInt(clientId);

                // check du code produit qui doit correspondre au code
                // produit sur lequel le lot porte
                if (!batch.getProduct().getCode().equals(productId)) {
                    importLog.setError(true);
                    importLog.setMessage(_("Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                            productId, batch.getProduct().getCode()));
                    errorSpotted = true;
                    continue;
                }

                // recherche de la presentation affectée par la production
                // (presentation d'origine) et/ou la vente (presentation dest)
                List<Presentation> presentations = presentationDAO.findAllByBatch(batch);
                Presentation originPresentation = null;
                Presentation destPresentation = null;
                for (Presentation presentation : presentations) {
                    if (presentation.getPresentationCode() == PresentationCode._) {
                        originPresentation = presentation;
                    }

                    // les deux cas peuvent être vrai en même temps
                    if (presentationCode == presentation.getPresentationCode()) {
                        destPresentation = presentation;
                    }
                }

                if (SgqUtils.isInternalClient(config, client)) {
                    
                    // Dans le cas d'une production, on doit diminuer le
                    // stock de la presentation d'origine
                    // mais il ne peut que la presentation d'origine n'existe pas
                    if (originPresentation == null) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find original presentation for row " + Arrays.toString(data));
                        }
                    } else {
                        // Il se peut même que la transformation ne s'applique
                        // pas a la presentation d'origine, mais on ne peut
                        // pas le savoir et on diminue quand meme le stock de
                        // la presentation d'origine. Il peut donc en resulter un
                        // stock négatif
                        double presQuantity = originPresentation.getQuantity();
                        presQuantity -= quantity;
                        originPresentation.setQuantity(presQuantity);
                        presentationDAO.update(originPresentation);
                    }

                    // dans le cas ou la presentation produite n'existe pas,
                    // on doit en creer une nouvelle
                    if (destPresentation == null) {
                        destPresentation = presentationDAO.create();
                        destPresentation.setBatch(batch);
                        destPresentation.setPresentationCode(presentationCode);

                        importLog.setMessage(_("Nouvelle presentation %c pour le lot %s",
                                presentationCode.getCode(), batchId));
                    } else {
                        importLog.setMessage(_("Mise à jour de la presentation %c du lot %s",
                                presentationCode.getCode(), batchId));
                    }

                    double presQuantity = destPresentation.getQuantity();
                    presQuantity += quantity;
                    destPresentation.setQuantity(presQuantity);
                    presentationDAO.update(destPresentation);
                    
                    // sauvegarde des productions
                    Production production = productionDAO.create();
                    production.setDate(lineDate);
                    production.setQuantity(presQuantity);
                    production.setPresentation(destPresentation);
                    production.setSource(sourceZone);

                    // cas ou la destination est différente
                    // si le client est 99997, la plante change de zone
                    if (client == config.getClientInternalZc()) {
                        production.setDestination(Zone.ZC);
                    } else {
                        production.setDestination(sourceZone);
                    }
                    productionDAO.update(production);

                } else {

                    // il peut se produire des cas où une presentation
                    // est expédié directement à un client sans qu'elle
                    // ai été enregistrée en production (manque de tracabilité)
                    // on ne peut donc pas diminuer le stock dans ce cas
                    // mais l'expedition peut-être enregistrer
                    if (destPresentation == null) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find expedition presentation for batch " + batch.getNumber() + ". Create it !");
                        }
                        destPresentation = presentationDAO.create();
                        destPresentation.setBatch(batch);
                        destPresentation.setPresentationCode(presentationCode);
                        destPresentation.setQuantity(0);
                        presentationDAO.update(destPresentation);
                        importLog.setMessage(_("Nouvelle expedition pour le lot %s au client %s",
                                batchId, client));
                    } else {
                        // reduction de la quantité de la presentation
                        double presQuantity = destPresentation.getQuantity();
                        presQuantity -= quantity;
                        destPresentation.setQuantity(presQuantity);
                        presentationDAO.update(destPresentation);
                        
                        importLog.setMessage(_("Nouvelle expedition pour le lot %s au client %s",
                                batchId, client));
                    }

                    // ajout d'une expedition
                    Expedition expedition = expeditionDAO.create();
                    expedition.setDate(lineDate);
                    expedition.setPresentation(destPresentation);
                    expedition.setQuantity(quantity);
                    // les expeditions ne sont pas liées aux entités client
                    // car ils ne sont pas en base. Les clients en base
                    // sont ceux dediés à certains lots
                    expedition.setClient(clientId);
                    expedition.setSource(sourceZone);
                    expeditionDAO.update(expedition);

                    // DMESD : date de la première vente du lot quelques soit
                    // sa présentation mais au niveau de l'atelier des expéditions.
                    if (sourceZone == Zone.ZP) {
                        Date previousDate = batch.getDmesd();
                        if (previousDate == null || lineDate.before(previousDate)) {
                            batch.setDmesd(lineDate);
                            if (log.isDebugEnabled()) {
                                log.debug("Update dmesd for batch to " + lineDate);
                            }
                        }
                    }
                }
            }

            // il faut faire un commit pour la derniere ligne
            // car il n'y a pas de détection de changement de date pour celle ci
            // si lastCommitDate est encore null, c'est que rien n'a été importé
            if (!errorSpotted && lastCommitDate != null) {
                daoHelper.commit();
            } else {
                daoHelper.rollback();
            }

        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (ParseException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }

        return importLogs;
    }

    /**
     * Test si parmis un liste il y a un import log en erreur.
     */
    protected boolean isErrorLog(List<ImportLog> importLogs) {
        boolean result = false;
        Iterator<ImportLog> it = importLogs.iterator();
        while (!result && it.hasNext()) {
            if (it.next().isError()) {
                result = true;
            }
        }
        return result;
    }
}
