package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: ReferentielService.java 161 2014-06-09 17:07:41Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/ReferentielService.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.csv.ImportRuntimeException;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.franciaflex.faxtomail.persistence.entities.Client;
import com.franciaflex.faxtomail.persistence.entities.ClientTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.DemandType;
import com.franciaflex.faxtomail.persistence.entities.DemandTypeImpl;
import com.franciaflex.faxtomail.persistence.entities.DemandTypeTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EmailAccount;
import com.franciaflex.faxtomail.persistence.entities.EmailAccountTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EmailTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EtatAttente;
import com.franciaflex.faxtomail.persistence.entities.EtatAttenteImpl;
import com.franciaflex.faxtomail.persistence.entities.EtatAttenteTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.MailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailFilterTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.Priority;
import com.franciaflex.faxtomail.persistence.entities.PriorityTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.Range;
import com.franciaflex.faxtomail.persistence.entities.RangeTopiaDao;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.franciaflex.faxtomail.services.service.imports.ClientImportModel;
import com.franciaflex.faxtomail.services.service.imports.DemandTypeImportModel;
import com.franciaflex.faxtomail.services.service.imports.EmailAccountImportModel;
import com.franciaflex.faxtomail.services.service.imports.EmailFilterImportModel;
import com.franciaflex.faxtomail.services.service.imports.EtatAttenteImportModel;
import com.franciaflex.faxtomail.services.service.imports.PriorityImportModel;
import com.franciaflex.faxtomail.services.service.imports.RangeImportModel;
import com.google.common.collect.Maps;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since x.x
 */
public class ReferentielService extends FaxToMailServiceSupport {

    private static final Log log = LogFactory.getLog(ReferentielService.class);

    public List<DemandType> getAllDemandType() {
        DemandTypeTopiaDao dao = getPersistenceContext().getDemandTypeDao();
        return new ArrayList<>(dao.findAll());
    }

    public List<Priority> getAllPriority() {
        PriorityTopiaDao dao = getPersistenceContext().getPriorityDao();
        return new ArrayList<>(dao.findAll());
    }

    public List<Range> getAllRange() {
        RangeTopiaDao dao = getPersistenceContext().getRangeDao();
        return new ArrayList<>(dao.findAll());
    }

    public List<EtatAttente> getAllEtatAttente() {
        EtatAttenteTopiaDao dao = getPersistenceContext().getEtatAttenteDao();
        return new ArrayList<>(dao.findAll());
    }

    public Map<String, Long> getEtatAttentesUsage() {
        EtatAttenteTopiaDao dao = getPersistenceContext().getEtatAttenteDao();
        Map<String, Long> result = dao.getEtatAttenteCountByFolder();
        return result;
    }

    public void saveEtatAttente(Collection<EtatAttente> etatAttentes) {

        Binder<EtatAttente, EtatAttente> binderEtatAttente = BinderFactory.newBinder(EtatAttente.class);
        EtatAttenteTopiaDao etatAttenteDAO = getPersistenceContext().getEtatAttenteDao();

        List<EtatAttente> allEtatAttente = getAllEtatAttente();
        Map<String, EtatAttente> allEtatAttenteIndex = new HashMap<>(Maps.uniqueIndex(allEtatAttente, TopiaEntities.getTopiaIdFunction()));
        for (EtatAttente etatAttente : etatAttentes) {
            // get current etat attente
            EtatAttente currentEtatAttente;
            if (StringUtils.isBlank(etatAttente.getTopiaId()) || etatAttente.getTopiaId().startsWith("new_")) {
                currentEtatAttente = new EtatAttenteImpl();
            } else {
                currentEtatAttente = allEtatAttenteIndex.remove(etatAttente.getTopiaId());
            }
            
            // copy
            binderEtatAttente.copyExcluding(etatAttente, currentEtatAttente,
                    EtatAttente.PROPERTY_TOPIA_ID,
                    EtatAttente.PROPERTY_TOPIA_CREATE_DATE,
                    EtatAttente.PROPERTY_TOPIA_VERSION);

            // persist
            if (currentEtatAttente.isPersisted()) {
                etatAttenteDAO.update(currentEtatAttente);
            } else {
                etatAttenteDAO.create(currentEtatAttente);
            }
        }

        // delete remaining
        etatAttenteDAO.deleteAll(allEtatAttenteIndex.values());
        
        getPersistenceContext().commit();
    }

    public void saveDemandTypes(List<DemandType> demandTypes) {
        Binder<DemandType, DemandType> binderDemandType = BinderFactory.newBinder(DemandType.class);
        DemandTypeTopiaDao demandTypeDAO = getPersistenceContext().getDemandTypeDao();

        List<DemandType> allDemandType = getAllDemandType();
        Map<String, DemandType> allDemandTypeIndex = new HashMap<>(Maps.uniqueIndex(allDemandType, TopiaEntities.getTopiaIdFunction()));
        for (DemandType demandType : demandTypes) {
            // get current etat attente
            DemandType currentDemandType;
            if (StringUtils.isBlank(demandType.getTopiaId()) || demandType.getTopiaId().startsWith("new_")) {
                currentDemandType = new DemandTypeImpl();
            } else {
                currentDemandType = allDemandTypeIndex.remove(demandType.getTopiaId());
            }
            
            // copy
            binderDemandType.copyExcluding(demandType, currentDemandType,
                    EtatAttente.PROPERTY_TOPIA_ID,
                    EtatAttente.PROPERTY_TOPIA_CREATE_DATE,
                    EtatAttente.PROPERTY_TOPIA_VERSION);

            // persist
            if (currentDemandType.isPersisted()) {
                demandTypeDAO.update(currentDemandType);
            } else {
                demandTypeDAO.create(currentDemandType);
            }
        }
        
        getPersistenceContext().commit();
    }

    public List<Client> getAllClients() {
        ClientTopiaDao dao = getPersistenceContext().getClientDao();
        return new ArrayList<>(dao.findAll());
    }

    public List<Client> importClients(InputStream inputStream) {
        List<Client> result = new ArrayList<>();
        ClientTopiaDao dao = getPersistenceContext().getClientDao();

        ClientImportModel clientImportModel = new ClientImportModel(';');
        try {
            Import<Client> importer = Import.newImport(clientImportModel, inputStream);
            try {
                for (Client client : importer) {
                    result.add(dao.create(client));
                }
                getPersistenceContext().commit();

            } finally {
                IOUtils.closeQuietly(importer);
            }

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }

    public List<EmailAccount> importEmailAccounts(InputStream inputStream) {
        List<EmailAccount> result = new ArrayList<>();
        EmailAccountTopiaDao dao = getPersistenceContext().getEmailAccountDao();

        EmailAccountImportModel emailAccountImportModel = new EmailAccountImportModel(';');
        try (Import<EmailAccount> importer = Import.newImport(emailAccountImportModel, inputStream)) {
            for (EmailAccount emailAccount : importer) {
                result.add(dao.create(emailAccount));
            }
            getPersistenceContext().commit();
        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }

    public List<MailFilter> importEmailFilters(InputStream inputStream, Map<String, MailFolder> foldersByName) {
        List<MailFilter> result = new ArrayList<>();
        MailFilterTopiaDao dao = getPersistenceContext().getMailFilterDao();

        EmailFilterImportModel emailFilterImportModel = new EmailFilterImportModel(';', foldersByName);
        try (Import<MailFilter> importer = Import.newImport(emailFilterImportModel, inputStream)) {
            for (MailFilter emailFilter : importer) {
                result.add(dao.create(emailFilter));
            }
            getPersistenceContext().commit();

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }

    public List<EtatAttente> importEtatAttentes(InputStream inputStream) {

        List<EtatAttente> result = new ArrayList<>();
        EtatAttenteTopiaDao etatAttenteTopiaDao = getPersistenceContext().getEtatAttenteDao();
        
        EtatAttenteImportModel importModel = new EtatAttenteImportModel(';');
        try (Import<EtatAttente> importer = Import.newImport(importModel, inputStream)) {
            for (EtatAttente etatAttente : importer) {
                
                EtatAttente currentEtatAttente = etatAttenteTopiaDao.forLabelEquals(etatAttente.getLabel()).findAnyOrNull();
                if (currentEtatAttente == null) {
                    currentEtatAttente = etatAttenteTopiaDao.create(etatAttente);
                }
                // else maybe update current ?
                
                result.add(currentEtatAttente);
            }
            getPersistenceContext().commit();

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        
        return result;
    }

    public List<Priority> importPriorities(InputStream inputStream) {
        List<Priority> result = new ArrayList<>();
        PriorityTopiaDao priorityTopiaDao = getPersistenceContext().getPriorityDao();
        
        PriorityImportModel importModel = new PriorityImportModel(';');
        try (Import<Priority> importer = Import.newImport(importModel, inputStream)) {
            for (Priority priority : importer) {
                
                Priority currentPriority = priorityTopiaDao.forLabelEquals(priority.getLabel()).findAnyOrNull();
                if (currentPriority == null) {
                    currentPriority = priorityTopiaDao.create(priority);
                }
                // else maybe update current ?
                
                result.add(currentPriority);
            }
            getPersistenceContext().commit();

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        
        return result;
    }

    public List<Range> importRanges(InputStream inputStream) {
        List<Range> result = new ArrayList<>();
        RangeTopiaDao rangeTopiaDao = getPersistenceContext().getRangeDao();
        
        RangeImportModel importModel = new RangeImportModel(';');
        try (Import<Range> importer = Import.newImport(importModel, inputStream)) {
            for (Range range : importer) {
                
                Range currentRange = rangeTopiaDao.forLabelEquals(range.getLabel()).findAnyOrNull();
                if (currentRange == null) {
                    currentRange = rangeTopiaDao.create(range);
                }
                // else maybe update current ?
                
                result.add(currentRange);
            }
            getPersistenceContext().commit();

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }

    public List<DemandType> importDemandTypes(InputStream inputStream) {
        List<DemandType> result = new ArrayList<>();
        DemandTypeTopiaDao demandTypeTopiaDao = getPersistenceContext().getDemandTypeDao();
        
        DemandTypeImportModel importModel = new DemandTypeImportModel(';');
        try (Import<DemandType> importer = Import.newImport(importModel, inputStream)) {
            for (DemandType demandType : importer) {
                
                DemandType currentDemandType = demandTypeTopiaDao.forLabelEquals(demandType.getLabel()).findAnyOrNull();
                if (currentDemandType == null) {
                    currentDemandType = demandTypeTopiaDao.create(demandType);
                }
                // else maybe update current ?
                
                result.add(currentDemandType);
            }
            getPersistenceContext().commit();

        } catch (ImportRuntimeException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);

        } finally {
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }
}
