/* *##%
 * Copyright (c) 2009 morin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty.jdbc;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Date;
import java.util.Hashtable;
import java.util.Properties;

import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.enhydra.jdbc.pool.StandardXAPoolDataSource;
import org.enhydra.jdbc.standard.StandardXADataSource;
import org.nuiton.wikitty.WikittyException;

/**
 *
 * @author morin
 * @version $Revision: 90 $
 *
 * Last update: $Date: 2010-06-02 21:12:32 +0200 (mer., 02 juin 2010) $
 * by : $Author: bpoussin $
 */
public class WikittyJDBCUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyJDBCUtil.class);

    /** extension list column in the wikitty_admin table */
    static final public String COL_EXTENSION  = "extension_list";
    /** version column in the admin tables */
    static final public String COL_VERSION = "version";
    /** id column in all the tables */
    static final public String COL_ID = "id";
    /** name column in the extension_admin table */
    static final public String COL_NAME = "name";
    /** requires column in the extension_admin table */
    static final public String COL_REQUIRES = "requires";
    /** tagvalues column in the extension_admin table */
    static final public String COL_TAGVALUES = "tagvalues";
    /** field name column in the data tables */
    static final public String COL_FIELDNAME = "fieldName";
    /** field type column in the extension_data table */
    static final public String COL_FIELDTYPE = "fieldType";
    /** boolean value column in the wikitty_data table */
    static final public String COL_BOOLEAN_VALUE = "booleanValue";
    /** number value column in the wikitty_data table */
    static final public String COL_NUMBER_VALUE = "numberValue";
    /** text value column in the wikitty_data table */
    static final public String COL_TEXT_VALUE = "textValue";
    /** date value column in the wikitty_data table */
    static final public String COL_DATE_VALUE = "dateValue";
    /** deletion date column in wikitty_admin table */
    static final public String COL_DELETION_DATE = "deletionDate";

    /** basic selection without where clause query property name */
    static final public String QUERY_SELECT = "jdbc.queries.select";
    /** basic selection without where clause query property name */
    static final public String QUERY_SELECT_NOTDELETED = "jdbc.queries.select.notdeleted";
    /** basic selection with where clause query property name */
    static final public String QUERY_SELECT_WHERE = "jdbc.queries.select.where";
    /** not dleeted data selection with where clause query property name */
    static final public String QUERY_SELECT_WHERE_NOTDELETED = "jdbc.queries.select.where.notdeleted";

    /** wikitty_admin table creation query property name */
    static final public String QUERY_CREATION_WIKITTY_ADMIN_TEST =
            "jdbc.queries.creation.wikitty.admin.test";
    static final public String QUERY_CREATION_WIKITTY_ADMIN =
            "jdbc.queries.creation.wikitty.admin";
    /** wikitty_admin table creation query property name */
    static final public String QUERY_CREATION_WIKITTY_DATA_TEST =
            "jdbc.queries.creation.wikitty.data.test";
    static final public String QUERY_CREATION_WIKITTY_DATA =
            "jdbc.queries.creation.wikitty.data";
    /** insertion in the admin table query property name */
    static final public String QUERY_INSERT_WIKITTY_ADMIN = "jdbc.queries.insert.wikitty.admin";
    /** update in the admin table query property name */
    static final public String QUERY_UPDATE_WIKITTY_ADMIN = "jdbc.queries.update.wikitty.admin";
    /** insertion in the data table query property name */
    static final public String QUERY_INSERT_WIKITTY_DATA = "jdbc.queries.insert.wikitty.data";
    /** deletion in the admin table query property name */
    static final public String QUERY_DELETE_WIKITTY_ADMIN = "jdbc.queries.delete.wikitty.admin";
    /** deletion in the data table query property name */
    static final public String QUERY_DELETE_WIKITTY_DATA = "jdbc.queries.delete.wikitty.data";

    /** clear extension query property name */
    static final public String QUERY_CLEAR_EXTENSION = "jdbc.queries.clear.extension";
    /** clear wikitty query property name */
    static final public String QUERY_CLEAR_WIKITTY = "jdbc.queries.clear.wikitty";

    /** extension_admin table creation query property name */
    static final public String QUERY_CREATION_EXTENSION_ADMIN_TEST =
            "jdbc.queries.creation.extension.admin.test";
    static final public String QUERY_CREATION_EXTENSION_ADMIN =
            "jdbc.queries.creation.extension.admin";
    /** extension_data table creation query property name */
    static final public String QUERY_CREATION_EXTENSION_DATA_TEST =
            "jdbc.queries.creation.extension.data.test";
    static final public String QUERY_CREATION_EXTENSION_DATA =
            "jdbc.queries.creation.extension.data";
    /** insertion in the admin table query property name */
    static final public String QUERY_INSERT_EXTENSION_ADMIN =
            "jdbc.queries.insert.extension.admin";
    /** insertion in the data table query property name */
    static final public String QUERY_INSERT_EXTENSION_DATA =
            "jdbc.queries.insert.extension.data";

    /** JDBC JDBC_DRIVER property name */
    static final public String JDBC_DRIVER = "jdbc.con.driver";
    /** JDBC_HOST property name */
    static protected String JDBC_HOST = "jdbc.con.host";
    /** user name property name */
    static protected String JDBC_USER_NAME = "jdbc.con.userName";
    /** JDBC_PASSWORD property name */
    static protected String JDBC_PASSWORD = "jdbc.con.password";

    /** admin table name */
    static protected String TABLE_WIKITTY_ADMIN = "wikitty_admin";
    /** data table name */
    static protected String TABLE_WIKITTY_DATA = "wikitty_data";
    /** admin table name */
    static protected String TABLE_EXTENSION_ADMIN = "extension_admin";
    /** data table name */
    static protected String TABLE_EXTENSION_DATA = "extension_data";


    /**
     * Loads the properties in the jdbc.properties file.
     *
     * @return the properties for the connection and the queries
     */
    public static synchronized Properties loadProperties(Properties properties) {
        Properties queryConfig = new Properties();
        Properties databaseConfig = new Properties(queryConfig);
        try {
            // FIXME poussin 20100112 perhaps used nuitonutil.ApplicationConfig
            
            // queries
            URL url = ClassLoader.getSystemResource("wikitty-jdbc-query.properties");
            if (url == null) {
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                url = contextClassLoader.getResource("wikitty-jdbc-query.properties");
            }
            log.info("Reading resource from: " + url);
            InputStream stream = url.openStream();
            queryConfig.load(stream);
            stream.close();
        
            // config
            url = ClassLoader.getSystemResource("wikitty-jdbc-config.properties");
            if (url == null) {
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                url = contextClassLoader.getResource("wikitty-jdbc-config.properties");
            }
            log.info("Reading resource from: " + url);
            stream = url.openStream();
            databaseConfig.load(stream);
            stream.close();
        
            // extra config
            if (properties != null) {
                databaseConfig.putAll(properties);
            }
            
        } catch(IOException eee) {
            throw new WikittyException("Unable to load property file", eee);
        }

        return databaseConfig;
    }

    private static Hashtable<String, StandardXAPoolDataSource> dataSources =
        new Hashtable<String, StandardXAPoolDataSource>();

    /**
     * Get a new connection instance (i.e. it opens a new transaction) plug on
     * JTA.
     *
     * @param conf configuration
     * @return a new Connection (db transaction)
     */
    public static synchronized Connection getConnection(Properties conf) {
        try {
            TransactionManager transactionManager = com.arjuna.ats.jta.TransactionManager.transactionManager();

            String jdbcUrl = String.format("%s:%s@%s", conf.getProperty(JDBC_USER_NAME),
                    conf.getProperty(JDBC_PASSWORD), conf.getProperty(JDBC_HOST));
            
            if (!dataSources.containsKey(jdbcUrl)) {
                log.info("Creating PoolDataSource for: " + jdbcUrl);
                StandardXADataSource dataSource = new StandardXADataSource();
                dataSource.setDriverName(conf.getProperty(JDBC_DRIVER));
                dataSource.setUrl(conf.getProperty(JDBC_HOST));
                dataSource.setUser(conf.getProperty(JDBC_USER_NAME));
                dataSource.setPassword(conf.getProperty(JDBC_PASSWORD));
                dataSource.setTransactionManager(transactionManager);
            
                StandardXAPoolDataSource poolDataSource = new StandardXAPoolDataSource(dataSource);
                poolDataSource.setUser(conf.getProperty(JDBC_USER_NAME));
                poolDataSource.setPassword(conf.getProperty(JDBC_PASSWORD));
                poolDataSource.setTransactionManager(transactionManager);
                dataSources.put(jdbcUrl, poolDataSource);
            }
            
            StandardXAPoolDataSource poolDataSource = dataSources.get(jdbcUrl);
            Connection connection = poolDataSource.getConnection();
            return connection;
            
        } catch(SQLException eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * Get a new connection instance (i.e. it opens a new transaction).
     *
     * @return a new Connection (db transaction)
     * @throws SQLException if the connection fails
     */
    public static synchronized Connection getJDBCConnection(Properties conf) {
        try {
            Connection connection = DriverManager.getConnection(conf.getProperty(JDBC_HOST),
                        conf.getProperty(JDBC_USER_NAME), conf.getProperty(JDBC_PASSWORD));
            connection.setAutoCommit(false);
            return connection;
        } catch(SQLException eee) {
            throw new WikittyException(eee);
        }
    }

    public static void closeQuietly(Connection connection) {
        try {
            connection.close();
        } catch (SQLException e) {
            log.error("SQLException while closing connection", e);
        }
    }
    
    /**
     * Closes a connection (i.e. transaction) and commit data.
     *
     * @param connection the connection to close and commit
     */
    public static void commitJDBCConnection(Connection connection) {
        try {
            connection.commit();
        } catch(SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            try {
                connection.close();
            }  catch(SQLException eee) {
                throw new WikittyException(eee);
            }
        }
    }

    /**
     * Closes a connection (i.e. transaction) and rollback data.
     *
     * @param connection the connection to close and rollback
     */
    public static void rollbackJDBCConnection(Connection connection) {
        try {
            connection.rollback();
        } catch(SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            try {
                connection.close();
            }  catch(SQLException eee) {
                throw new WikittyException(eee);
            }
        }
    }

    /**
     * Execute query
     * @param con connexion to use
     * @param query sql query to do
     * @param args arguments for the query
     */
    static public void doQuery(Connection connection, String query, Object ... args) throws SQLException {
        PreparedStatement sta = connection.prepareStatement(query);
        for (int i=0; i<args.length; i++) {
            if (args[i] instanceof Date) {
                // force for java.util.Date to TIMESTAMP, because some driver
                // (Postgresql) don't support it naturaly
                sta.setObject(i + 1, args[i], Types.TIMESTAMP);
            } else {
                sta.setObject(i + 1, args[i]);
            }
        }
        sta.execute();
    }

}
