/*
 * #%L
 * Wikitty :: generators
 * 
 * $Id: WikittyPurifierTransformer.java 650 2010-12-23 11:44:57Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0/wikitty-generators/src/main/java/org/nuiton/wikitty/generator/WikittyPurifierTransformer.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.generator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;

/**
 * this transformer read the original user model and generate the intermediate
 * model needed by generators. 
 * 
 * 
 * 
 */
public class WikittyPurifierTransformer extends ObjectModelTransformerToJava {

    private static final Log log =
            LogFactory.getLog(WikittyPurifierTransformer.class);

    /** for a given class, store all the names used by this class and subClasses */
    Map<ObjectModelClass, List<String>> namesUsedByClass = new HashMap<ObjectModelClass, List<String>>();

    /** class of the original model that are already processed */
    List<ObjectModelClass> processedClasses = new ArrayList<ObjectModelClass>();

    /**
     * for a given, class read all attributes name and try to find conflicts
     * with parent classes attributes names.
     * 
     * If a conflict is found, we try to solve the conflict by adding an
     * tagValue to the conflicting attribute for others transformers
     * to consider it as a new name. To choose a new name, we look for
     * an alternative name provided by the user through a tagValue on this
     * attribute. If no alternative provided, we try to generate an alternative
     * name by adding "From<className>" to the attribute name.
     *  
     * @param clazz the class to process
     */
    protected void processClass(ObjectModelClass clazz) {

        if (processedClasses.contains(clazz)) {
            
            // clazz has already been processed, do nothing
            return;
        }

        if (isVerbose()) {
            log.info("will purify " + clazz.getQualifiedName());
        }
        
        List<String> allUsedNames = new ArrayList<String>();


        // first, we have to get all names used by the super classes and
        // its superclasses. So they must have been processed before
        // current class. Thus, we recursively call processClass on
        // superClass.

        // process superclass first
        for (ObjectModelClass superClasses : clazz.getSuperclasses()) {
            if (WikittyTransformerUtil.isBusinessEntity(superClasses)) {
                if (!processedClasses.contains(superClasses)) {
                    processClass(superClasses);
                    allUsedNames.addAll(namesUsedByClass.get(superClasses));
                }
            }
        }

        // now, allUsedNames contains all names used by superClass, we
        // have to check current class attributes names for conflict.
        // allUsedNames contains name that we
        // can't use without generating a conflict
        for (ObjectModelAttribute attribute : clazz.getAttributes()) {

            // will be null as long as a non-conflicting name is found
            String attributeName = null;

            if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                if (allUsedNames.contains(attributeName)) {
                    // using alternative name lead to a conflict
                    attribute.getTagValues().remove(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                    attributeName = null;
                }
            }

            if (attributeName == null) {
                attributeName = attribute.getName();
                if (allUsedNames.contains(attributeName)) {
                    // using alternative name lead to a conflict
                    attributeName = null;
                }
            }

            if (attributeName == null) {
                attributeName = attribute.getName() + "From" + clazz.getName();
                if (allUsedNames.contains(attributeName)) {
                    // using alternative name lead to a conflict
                    attributeName = null;
                } else {
                    addTagValue(attribute, WikittyTransformerUtil.TAG_ALTERNATIVE_NAME, attributeName);
                }
            }

            // finally
            if (attributeName == null) {
                // still no alternative :-(
                log.error("no way to resolve conflict with attribute" +
                          attribute.getName() + " from class " + clazz
                          + ". You should add or change a tagValue \"" +
                          WikittyTransformerUtil.TAG_ALTERNATIVE_NAME +
                          "\" on this attribute");
            } else {
                allUsedNames.add(attributeName);
            }
        }

        // saving all names we used in current class will permit to sub classes
        // to know what names it must not use
        namesUsedByClass.put(clazz, allUsedNames);
    }
    
    @Override
    public void transformFromModel(ObjectModel model) {
        for (ObjectModelClass clazz : model.getClasses()) {
            if (log.isDebugEnabled()) {
                log.debug("Will clone " + clazz.getQualifiedName());
            }
            ObjectModelClass clone = cloneClass(clazz, true);
            if (WikittyTransformerUtil.isBusinessEntity(clazz)) {
                processClass(clone);
            }
        }
    }
}
