/*
 * #%L
 * Wikitty :: generators
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.generator;

import org.apache.commons.lang.StringUtils;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;





/**
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.wikitty.generator.WikittyHelperGenerator"
 */
public class WikittyHelperGenerator extends ObjectModelTransformerToJava {

    protected static final String META_EXTENSION_SEPARATOR = ":";
    
    @Override
    public void transformFromClass(ObjectModelClass clazz) {
    
        ObjectModelClass helper = createClass(WikittyTransformerUtil.businessEntityToHelperName(clazz),
                                              clazz.getPackageName());
        
        // TODO 20100811 bleny remove unused imports
        addImport(helper, WikittyTransformerUtil.BUSINESS_ENTITY_CLASS_FQN);
        addImport(helper, WikittyTransformerUtil.BUSINESS_ENTITY_WIKITTY_CLASS_FQN);
        addImport(helper, WikittyTransformerUtil.WIKITTY_CLASS_FQN);
        addImport(helper, "org.nuiton.wikitty.WikittyExtension");
        addImport(helper, "org.nuiton.wikitty.WikittyUtil");
        addImport(helper, "org.nuiton.wikitty.WikittyUser");
        addImport(helper, "org.nuiton.wikitty.WikittyUserAbstract");
        addImport(helper, "org.nuiton.wikitty.WikittyUserImpl");
        addImport(helper, "org.nuiton.wikitty.WikittyTreeNode");
        addImport(helper, "org.nuiton.wikitty.WikittyTreeNodeAbstract");
        addImport(helper, "org.nuiton.wikitty.WikittyTreeNodeImpl");
        addImport(helper, java.util.List.class);
        addImport(helper, java.util.ArrayList.class);
        addImport(helper, java.util.Collection.class);
        addImport(helper, java.util.Collections.class);
        addImport(helper, java.util.Set.class);
        addImport(helper, java.util.Date.class);
        addImport(helper, java.util.LinkedHashSet.class);
        
        // making constructor for helper class (empty and private)
        ObjectModelOperation constructor = addConstructor(helper, ObjectModelModifier.PRIVATE);
        setDocumentation(constructor, "utility class all provided methods are accessible the static way");
        setOperationBody(constructor, "// empty\n"); // empty implementation

        if ( WikittyTransformerUtil.isBusinessEntity(clazz) ) {
            createOperationsForBusinessEntity(clazz, helper);
        }

        if ( WikittyTransformerUtil.isMetaExtension(clazz)) {
            createOperationForMetaExtension(clazz, helper);
        }
    }

    /** add operation if input model element has stereotype "entity" */
    protected void createOperationsForBusinessEntity(ObjectModelClass entity,
                                                     ObjectModelClass helper) {

        String extensionVariableName = WikittyTransformerUtil.classToExtensionVariableName(entity, true);
        
        // generating operations with bodies
        for (ObjectModelAttribute attribute : entity.getAttributes()) {
            if (attribute.isNavigable()) {
                // needed below, in templates
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);
                String attributeType = WikittyTransformerUtil.generateResultType(attribute, false);
    
                String attributeName = attribute.getName();
                if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                    // there is a conflict, purifier transformer give as the right name to use
                    attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                }

                if (WikittyTransformerUtil.isAttributeCollection(attribute)) {
                    // attributed is a collection, we will generate operations get, add, remove and clear

                    String attributeTypeSimpleNameInSet = WikittyTransformerUtil.generateResultType(attribute, true);
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);

                    // now, for this attribute, we will generate add, remove and clear methods
                    // adding operations to contract
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(helper, getterName, attributeTypeSimpleNameInSet, ObjectModelModifier.STATIC);
                    addParameter(getter, "Wikitty", "wikitty");
                    String getterBody = ""
+"\n"
+"        "+attributeTypeSimpleNameInSet+" result = wikitty."+getFieldMethodName+"("+extensionVariableName+", "+fieldVariableName+", "+attributeType+".class);\n"
+"        return result;\n"
+"";
                    setOperationBody(getter, getterBody);

                    String addName = "add" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation adder = addOperation(helper, addName, "void", ObjectModelModifier.STATIC);
                    addParameter(adder, "Wikitty", "wikitty");
                    addParameter(adder, attributeType, "element");
                    String adderBody = ""
+"\n"
+"        wikitty.addToField("+extensionVariableName+", "+fieldVariableName+", element);\n"
+"";
                    setOperationBody(adder, adderBody);

                    String removeName = "remove" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation remover = addOperation(helper, removeName, "void", ObjectModelModifier.STATIC);
                    addParameter(remover, "Wikitty", "wikitty");
                    addParameter(remover, attributeType, "element");
                    String removerBody = ""
+"\n"
+"        wikitty.removeFromField("+extensionVariableName+", "+fieldVariableName+", element);\n"
+"";
                    setOperationBody(remover, removerBody);
                    
                    String clearName = "clear" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation clear = addOperation(helper, clearName, "void", ObjectModelModifier.STATIC);
                    addParameter(clear, "Wikitty", "wikitty");
                    String clearBody = ""
+"\n"
+"        wikitty.clearField("+extensionVariableName+", "+fieldVariableName+");\n"
+"";
                    setOperationBody(clear, clearBody);
    
                    
                } else {
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
                    
                    // adding getter and setter to contract
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(helper, getterName, attributeType, ObjectModelModifier.STATIC);
                    addParameter(getter, "Wikitty", "wikitty");
                    setOperationBody(getter, ""
+"\n"
+"        "+attributeType+" value = wikitty."+getFieldMethodName+"("+extensionVariableName+", "+fieldVariableName+");\n"
+"        return value;\n"
+"");
                    
                    String setterName = "set" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation setter = addOperation(helper, setterName, attributeType, ObjectModelModifier.STATIC);
                    addParameter(setter, "Wikitty", "wikitty");
                    addParameter(setter, attributeType, attributeName);
                    setOperationBody(setter, ""
+"\n"
+"        "+attributeType+" oldValue = "+getter.getName()+"(wikitty);\n"
+"        wikitty.setField("+extensionVariableName+", "+fieldVariableName+", "+attributeName+");\n"
+"        return oldValue;\n"
+"");
                }
            }
        }
        
        
        
        // now, adding the equals(w1, w2)

        ObjectModelOperation equals = addOperation(helper, "equals", "boolean", ObjectModelModifier.STATIC);
        addParameter(equals, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "w1");
        addParameter(equals, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "w2");
        
        // the body of the equals method, will be assembled while reading attributes
        String equalsBody = ""
+"\n"
+"    boolean result = true;\n"
+"";

        for(ObjectModelAttribute attribute : entity.getAttributes()) {
            if (attribute.isNavigable()) {
                // two variables needed below
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);

                // considering field in equals body
                equalsBody += ""
+"\n"
+"        if (result) {\n"
+"            Object f1 = w1.getFieldAsObject("+ extensionVariableName +", "+ fieldVariableName +");\n"
+"            Object f2 = w2.getFieldAsObject("+ extensionVariableName +", "+ fieldVariableName +");\n"
+"            result = f1 == f2 || (f1 != null && f1.equals(f2));\n"
+"        };\n"
+"";
            }
        }

        // finishing equals body
        equalsBody += ""
+"\n"
+"    return result;\n"
+"";
        setOperationBody(equals, equalsBody);





        // finally, adding isExtension, hasExtension and addExtension
        
        ObjectModelOperation isExtension = addOperation(helper, "isExtension", "boolean", ObjectModelModifier.STATIC);
        addParameter(isExtension, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        addAnnotation(helper, isExtension, "Deprecated");
        setDocumentation(isExtension, "@deprecated renamed to keep consistency, use hasExtension instead");
        setOperationBody(isExtension, ""
+"\n"
+"        return hasExtension(wikitty);\n"
+"");

        ObjectModelOperation hasExtension = addOperation(helper, "hasExtension", "boolean", ObjectModelModifier.STATIC);
        addParameter(hasExtension, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        setDocumentation(hasExtension, "check if wikitty has current extension");
        setOperationBody(hasExtension, ""
+"\n"
+"        return wikitty.hasExtension("+extensionVariableName+");\n"
+"");
        
        ObjectModelOperation addExtension = addOperation(helper, "addExtension", "void", ObjectModelModifier.STATIC);
        addParameter(addExtension, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        setDocumentation(addExtension, "ajout les extensions static de cette classe au wikitty en argument");
        String contractName = WikittyTransformerUtil.businessEntityToAbstractName(entity);
        setOperationBody(addExtension, ""
+"\n"
+"        for (WikittyExtension ext : "+contractName+".extensions) {\n"
+"            wikitty.addExtension(ext);\n"
+"        }\n"
+"");
    }

    /** add needed operations if input model element has stereotype "meta" */
    protected void createOperationForMetaExtension(ObjectModelClass metaExtension,
                                                   ObjectModelClass helper) {
        
        String abstractName = WikittyTransformerUtil.businessEntityToAbstractName(metaExtension);
        String contractName = WikittyTransformerUtil.businessEntityToContractName(metaExtension);
        String extensionVariableName = WikittyTransformerUtil.classToExtensionVariableName(metaExtension, true);


        ObjectModelOperation addMetaExtension = addOperation(helper, "addMetaExtension", "void", ObjectModelModifier.STATIC);
        addParameter(addMetaExtension, WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN, "extension");
        addParameter(addMetaExtension, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        setDocumentation(addMetaExtension, String.format(
                "add %s meta-extension on given extension to the given wikitty",
                metaExtension.getName()));
        setOperationBody(addMetaExtension, ""
+"\n"
+"        wikitty.addMetaExtension("+abstractName+".extension"+contractName+", extension);\n"
+"");


        ObjectModelOperation hasMetaExtension = addOperation(helper, "hasMetaExtension", "boolean", ObjectModelModifier.STATIC);
        addParameter(hasMetaExtension, WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN, "extension");
        addParameter(hasMetaExtension, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        setDocumentation(hasMetaExtension, String.format(
                "true if given wikitty has %s meta-extension on given extension",
                metaExtension.getName()));
        setOperationBody(hasMetaExtension, ""
+"\n"
+"        return wikitty.hasMetaExtension("+extensionVariableName+", extension.getName());\n"
+"");     
        

        // now adding getMetaFielName, both implementations
        
        ObjectModelOperation getMetaFieldNameFromExtension = addOperation(helper, "getMetaFieldName", "String", ObjectModelModifier.STATIC);
        addParameter(getMetaFieldNameFromExtension, WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN, "extension");
        addParameter(getMetaFieldNameFromExtension, "String", "fieldName");
        setDocumentation(getMetaFieldNameFromExtension, String.format(
                "for extension 'Ext' and field 'f', return 'Ext:%s.f'",
                metaExtension.getName()));
        setOperationBody(getMetaFieldNameFromExtension, ""
+"\n"
+"        String metaFieldName = getMetaFieldName(extension.getName(), fieldName); \n"
+"        return metaFieldName;\n"
+"");
        
        ObjectModelOperation getMetaFieldNameFromExtensionName = addOperation(helper, "getMetaFieldName", "String", ObjectModelModifier.STATIC);
        addParameter(getMetaFieldNameFromExtensionName, "String", "extensionName");
        addParameter(getMetaFieldNameFromExtensionName, "String", "fieldName");
        setDocumentation(getMetaFieldNameFromExtensionName, String.format(
                "for extension 'Ext' and field 'f', return 'Ext:%s.f'",
                metaExtension.getName()));
        setOperationBody(getMetaFieldNameFromExtensionName, ""
+"\n"
+"        String metaFieldName = WikittyUtil.getMetaFieldName(\""+metaExtension.getName()+"\", extensionName, fieldName);\n"
+"        return metaFieldName;\n"
+"");



        // now, adding all
        
        // generating operations with bodies
        for (ObjectModelAttribute attribute : metaExtension.getAttributes()) {
            if (attribute.isNavigable()) {
                // needed below, in templates
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);
                String attributeType = WikittyTransformerUtil.generateResultType(attribute, false);
    
                String attributeName = attribute.getName();
                if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                    // there is a conflict, purifier transformer give as the right name to use
                    attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                }

                if (WikittyTransformerUtil.isAttributeCollection(attribute)) {
                    // attributed is a collection, we will generate operations get, add, remove and clear

                    String attributeTypeSimpleNameInSet = WikittyTransformerUtil.generateResultType(attribute, true);
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);

                    // now, for this attribute, we will generate add, remove and clear methods
                    // adding operations to contract
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(helper, getterName, attributeTypeSimpleNameInSet, ObjectModelModifier.STATIC);
                    addParameter(getter, "String", "extensionName");
                    addParameter(getter, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    String getterBody = ""
+"\n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        "+attributeTypeSimpleNameInSet+" result = ("+attributeTypeSimpleNameInSet+") wikitty.getFqField(fieldName);        \n"
+"        return result;\n"
+"";
                    setOperationBody(getter, getterBody);

                    String addName = "add" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation adder = addOperation(helper, addName, "void", ObjectModelModifier.STATIC);
                    addParameter(adder, "String", "extensionName");
                    addParameter(adder, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    addParameter(adder, attributeType, "element");
                    String adderBody = ""
+"        \n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        wikitty.addToField(fieldName, element);\n"
+"";
                    setOperationBody(adder, adderBody);

                    String removeName = "remove" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation remover = addOperation(helper, removeName, "void", ObjectModelModifier.STATIC);
                    addParameter(remover, "String", "extensionName");
                    addParameter(remover, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    addParameter(remover, attributeType, "element");
                    String removerBody = ""
+"\n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        wikitty.removeFromField(fieldName, element);\n"
+"";
                    setOperationBody(remover, removerBody);
                    
                    String clearName = "clear" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation clear = addOperation(helper, clearName, "void", ObjectModelModifier.STATIC);
                    addParameter(clear, "String", "extensionName");
                    addParameter(clear, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    String clearBody = ""
+"\n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        wikitty.clearField(fieldName);\n"
+"";
                    setOperationBody(clear, clearBody);
    
                    
                } else {
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
                    
                    // adding getter and setter to contract
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(helper, getterName, attributeType, ObjectModelModifier.STATIC);
                    addParameter(getter, "String", "extensionName");
                    addParameter(getter, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    setOperationBody(getter, ""
+"\n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        "+attributeType+" value = ("+attributeType+") wikitty.getFqField(fieldName);        \n"
+"        return value;\n"
+"");
                    
                    String setterName = "set" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation setter = addOperation(helper, setterName, attributeType, ObjectModelModifier.STATIC);
                    addParameter(setter, "String", "extensionName");
                    addParameter(setter, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
                    addParameter(setter, attributeType, attributeName);
                    setOperationBody(setter, ""
+"\n"
+"        "+attributeType+" oldValue = "+getter.getName()+"(extensionName, wikitty);\n"
+"        String fieldName = getMetaFieldName(extensionName, \""+attributeName+"\");\n"
+"        wikitty.setFqField(fieldName, "+attributeName+");\n"
+"        return oldValue;\n"
+"");
                }
            }
        }
    }
}
