/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: InMemoryStorageTest.java 500 2010-11-23 10:38:36Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.3/wikitty-api/src/test/java/org/nuiton/wikitty/memory/InMemoryStorageTest.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.memory;

import static junit.framework.Assert.assertEquals;
import static junit.framework.Assert.assertTrue;

import java.io.File;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;

import org.junit.Before;
import org.junit.Test;
import org.nuiton.wikitty.entities.ExtensionFactory;
import org.nuiton.wikitty.entities.FieldType.TYPE;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyImpl;
import org.nuiton.wikitty.services.WikittyServiceEnhanced;
import org.nuiton.wikitty.services.WikittyServiceInMemory;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.api.AbstractTestApi;

public class InMemoryStorageTest extends AbstractTestApi {
    
    protected WikittyServiceInMemory wikittyService;
    
    @Before
    public void init() throws Exception {
        WikittyConfig config = new WikittyConfig();
        wikittyService = new WikittyServiceInMemory(config);
    }
    
    protected static DateFormat dateFormater = new SimpleDateFormat("dd/MM/yyyy");

    @Test
    public void testFilePersistance() throws Exception {
        // TODO poussin 20101113 hack to create tmp dir, this is necessary
        // actually because new mavenpom move tmp dir to target without create it :(
        File tmp = new File(System.getProperty("java.io.tmpdir"));
        tmp.mkdirs();
        // end of hack

        File tempPersistFile = File.createTempFile(InMemoryStorageTest.class.getName(), ".tmp");
        tempPersistFile.delete();
        tempPersistFile.deleteOnExit();
        
        WikittyConfig config = new WikittyConfig();
        config.setOption(
                WikittyConfig.WikittyOption.WIKITTY_WIKITTYSERVICEINMEMORY_PERSISTENCE.getKey(),
                "true");
        config.setOption(
                WikittyConfig.WikittyOption.WIKITTY_WIKITTYSERVICEINMEMORY_PERSISTENCE_FILE.getKey(),
                tempPersistFile.getPath());
        wikittyService = new WikittyServiceInMemory(config);
        String extName = "MyExtName";
        WikittyExtension ext = ExtensionFactory.create("MyExtName", "1")
            .addField("fieldName0", TYPE.NUMERIC)
            .addField("fieldName1", TYPE.NUMERIC)
            .addField("fieldName2", TYPE.DATE)
            .extension();
        Date date = dateFormater.parse( "30/01/2009" );
        
        Wikitty w = createWikitty(
            "MyExtName.fieldName0 = 123," +
            "MyExtName.fieldName1 = 12.3," +
            "MyExtName.fieldName2 = " + WikittyUtil.formatDate(date),
            ext
        );
        String id = w.getId();
        wikittyService.store(null, Collections.singletonList(w), false);
        ((WikittyServiceInMemory)wikittyService).saveToPersistenceFile(tempPersistFile);
        
        wikittyService = new WikittyServiceInMemory( config );
        w = WikittyServiceEnhanced.restore(wikittyService, null, id);
        assertTrue( w.hasField(extName, "fieldName0") );
        assertEquals( 123, w.getFieldAsInt(extName, "fieldName0") );
        assertTrue( w.hasField(extName, "fieldName1") );
        assertEquals( 12.3f, w.getFieldAsFloat(extName, "fieldName1") );
        assertTrue( w.hasField(extName, "fieldName2") );
        assertEquals( date, w.getFieldAsDate(extName, "fieldName2") );
    }

    private Wikitty createWikitty(String wDef, WikittyExtension... exts) {
        Wikitty w = new WikittyImpl();
        w.addExtension( Arrays.asList(exts) );
        for ( String line : wDef.split(",") ) {
            int eqIdx = line.indexOf('=');
            String value = line.substring(eqIdx+1).trim();
            String key = eqIdx >= 0 ? line.substring(0, eqIdx): "";
            int dotIdx = key.indexOf('.');
            String fieldName = key.substring(dotIdx+1).trim();
            String extName = dotIdx >= 0 ? key.substring(0, dotIdx): "";
            w.setField(extName, fieldName, value);
        }
        return w;
    }

    @Test
    public void testRequire() throws Exception {
        WikittyExtension extension = new WikittyExtension("Extension", "1", null,
                WikittyUtil.buildFieldMapExtension("String value"));
        WikittyExtension extension1 = new WikittyExtension("Extension1", "1", "Extension",
                WikittyUtil.buildFieldMapExtension("String value"));
        WikittyExtension extension11 = new WikittyExtension("Extension11", "1", "Extension1",
                WikittyUtil.buildFieldMapExtension("String value"));
        WikittyExtension extension2 = new WikittyExtension("Extension2", "1", "Extension",
                WikittyUtil.buildFieldMapExtension("String value"));

        Wikitty wikitty = createWikitty("Extension.value = 123", 
                extension, extension1, extension11, extension2);

        Collection<WikittyExtension> result = wikitty.getExtensionDependencies("Extension", true);
        assertEquals(3, result.size());

        result = wikitty.getExtensionDependencies("Extension", false);
        assertEquals(2, result.size());

        result = wikitty.getExtensionDependencies("Extension1", true);
        assertEquals(1, result.size());

        result = wikitty.getExtensionDependencies("Extension2", true);
        assertEquals(0, result.size());
    }
}
