/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: AbstractTopiaMigrationCallback.java 2949 2013-12-20 14:59:42Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-8/topia-service-migration/src/main/java/org/nuiton/topia/migration/AbstractTopiaMigrationCallback.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.migration;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaApplicationContext;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.TopiaPersistenceContext;
import org.nuiton.topia.persistence.support.TopiaSqlSupport;
import org.nuiton.util.StringUtil;
import org.nuiton.util.Version;

import java.util.List;

/**
 * Abstract migration callback.
 *
 * @author tchemit <chemit@codelutin.com>
 * @version $Id: AbstractTopiaMigrationCallback.java 2949 2013-12-20 14:59:42Z tchemit $
 * @since 2.5
 */
public abstract class AbstractTopiaMigrationCallback<PersistenceContext extends TopiaPersistenceContext> {

    /** Logger */
    private static final Log log = LogFactory.getLog(AbstractTopiaMigrationCallback.class);

    /** @return the available versions from the call back */
    public abstract Version[] getAvailableVersions();

    /**
     * Hook to ask user if migration can be performed.
     *
     * @param dbVersion the actual db version
     * @param versions  the versions to update
     * @return {@code false} if migration is canceled, {@code true} otherwise.
     */
    public abstract boolean askUser(Version dbVersion, List<Version> versions);

    /**
     * Get the sql support used to execute sql queries.
     *
     * @param persistenceContext persistence context
     * @return the sql support
     * @since 3.0
     */
    protected abstract TopiaSqlSupport getSqlSupport(PersistenceContext persistenceContext);

    protected abstract void migrateForVersion(Version version,
                                              PersistenceContext tx,
                                              boolean showSql,
                                              boolean showProgression) throws Exception;

    /**
     * Tentative de migration depuis la version de la base version la version
     * souhaitee.
     * <p/>
     * On applique toutes les migrations de version indiquee dans le parametre
     * <code>version</code>.
     * <p/>
     * Pour chaque version, on cherche la methode migrateTo_XXX ou XXX est la
     * version transforme en identifiant java via la methode
     * {@link Version#getValidName()} et on l'execute.
     * <p/>
     * Note: pour chaque version a appliquer, on ouvre une nouvelle transaction.
     *
     * @param applicationContext topia context de la transaction en cours
     * @param dbVersion          database version
     * @param showSql            drapeau pour afficher les requete sql
     * @param showProgression    drapeau pour afficher la progression
     * @param versions           all versions knwon by service  @return migration a
     *                           ggrement
     * @return {@code true} si la migration est accepté, {@code false} autrement.
     */
    public boolean doMigration(TopiaApplicationContext<PersistenceContext> applicationContext,
                               Version dbVersion,
                               boolean showSql,
                               boolean showProgression,
                               List<Version> versions) {

        boolean doMigrate = askUser(dbVersion, versions);
        if (doMigrate) {

            for (Version v : versions) {
                // ouverture d'une connexion direct JDBC sur la base
                try {

                    PersistenceContext persistenceContext =
                            applicationContext.newPersistenceContext();
                    try {

                        log.info(String.format("Start migration to version %s", v));

                        migrateForVersion(v, persistenceContext, showSql, showProgression);

                        // commit des modifs
                        persistenceContext.commit();

                    } catch (Exception eee) {
                        // en cas d'erreur
                        log.error("Could not migrate database", eee);
                        // rollback du travail en cours
                        persistenceContext.rollback();
                        // propagation de l'erreur
                        throw eee;
                    } finally {
                        // close database connexion
                        if (persistenceContext != null) {
                            persistenceContext.closeContext();
                        }
                    }

                } catch (Exception eee) {
                    log.error("Error lors de la tentative de migration", eee);
                    doMigrate = false;
                    // toute erreur arrête la mgration
                    break;
                }
            }
        }
        return doMigrate;
    }

    public void executeSQL(PersistenceContext tx, String... sqls)
            throws TopiaException {
        executeSQL(tx, false, false, sqls);
    }

    /**
     * Executes the given {@code sqls} requests.
     *
     * @param showSql         flag to see sql requests
     * @param showProgression flag to see progession on console
     * @param sqls            requests to execute
     * @throws TopiaException if any pb
     * @since 2.3.0
     */
    public void executeSQL(PersistenceContext tx,
                           final boolean showSql,
                           final boolean showProgression,
                           final String... sqls) throws TopiaException {

        TopiaSqlSupport sqlSupport = getSqlSupport(tx);
        if (log.isInfoEnabled()) {

            log.info(String.format("Will execute %1$s requests...", sqls.length));
        }
        if (showSql) {
            StringBuilder buffer = new StringBuilder();
            for (String s : sqls) {
                buffer.append(s).append("\n");
            }
            log.info("SQL TO EXECUTE :\n" +
                     "--------------------------------------------------------------------------------\n" +
                     "--------------------------------------------------------------------------------\n" +
                     buffer.toString() +
                     "--------------------------------------------------------------------------------\n" +
                     "--------------------------------------------------------------------------------\n"
            );
        }

        int index = 0;
        int max = sqls.length;
        for (String sql : sqls) {
            index++;
            long t0 = System.nanoTime();
            if (log.isInfoEnabled()) {
                String message = "";

                if (showProgression) {
                    message = String.format("Executing request [%1$-4s/%2$-4s]", index, max);
                }
                if (showSql) {
                    message += "\n" + sql;
                }
                if (showProgression || showSql) {

                    log.info(message);
                }
            }

            sqlSupport.executeSql(sql);

            if (log.isDebugEnabled()) {
                String message;
                message = String.format("Request [%1$-4s/%2$-4s] executed in %3$s.", index, max, StringUtil.convertTime(System.nanoTime() - t0));
                log.debug(message);
            }
        }
    }

}