/* *##% ToPIA - Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/

package org.nuiton.topia;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Resource;

/**
 * ContextUtil.java
 *
 * Created: 17 avril 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Revision: 1334 $
 *
 * Outil pour la manipulation de la base de données et des contextes Topia
 * Utilisation recommandé (Patron Singleton) :
 * -------------------------------------------
 *
 * public ContextUtilProject extends ContextUtil {
 *
 *   private static ContextUtil instance = null;
 *
 *   public static ContextUtil getInstance() {
 *       if(instance == null) {
 *           instance = new ContextUtilProject();
 *       }
 *       return instance;
 *   }
 *
 *   private ContextUtilProject() {
 *       super(ContextUtilProject.class);
 *   }
 *
 *   @Override
 *   public void loadProperties(Properties props) throws IOException, URISyntaxException {
 *       loadFileProperties("TopiaContextImpl.properties",props);
 *       props.setProperty("topia.persistence.classes", ProjectModelDAOHelper.getImplementationClassesAsString());
 *   }
 *
 *   @Override // Initialisation/Vérification BD à chaque création du contexte global
 *   protected void initDB() throws TopiaException {
 *
 *   }
 *
 * }
 *
 * Utilisation de la transaction courante :
 * ----------------------------------------
 *
 * // Déclaration du contexte
 * ContextUtil context = ContextUtilProject.getInstance();
 *
 * // Récupération du dao avec transaction courante
 * ObjectDAO dao = ProjectModelDAOHelper.getObjectDAO(context.transaction());
 *
 * ... // traitement avec dao
 *
 * context.commit();
 *
 * ... // traitement avec dao
 *
 * context.commit();
 *
 * context.close();
 *
 * Possibilité de créer plusieurs transactions et de les manipuler indépendamment :
 * --------------------------------------------------------------------------------
 *
 * // Déclaration du contexte
 * ContextUtil context = ContextUtilProject.getInstance();
 *
 * // Création de deux transactions
 * TopiaContext transaction1 = context.getContext().beginTransaction();
 * TopiaContext transaction2 = context.getContext().beginTransaction();
 *
 * ObjectDAO dao1 = ProjectModelDAOHelper.getObjectDAO(transaction1);
 *
 * OtherDAO dao2 = ProjectModelDAOHelper.getOtherDAO(transaction2);
 *
 * try {
 *   ...
 *   transaction1.commitTransaction();
 * } catch (TopiaException ex) {
 *   // Si ereur sur la transaction1 annulation de la transaction2
 *   transaction2.rollbackTransaction();
 * }
 * ...
 *
 * transaction1.closeContext();
 * ...
 *
 *
 * Utilisation dans les tests unitaires :
 * ------------------------------------
 *
 *  @BeforeClass
 *  public static void setUpClass() throws Exception {
 *      ContextUtilProject.getInstance().init();
 *  }
 *
 *  @AfterClass
 *  public static void tearDownClass() throws Exception {
 *      ContextUtilProject.getInstance().disconnect();
 *  }
 */
public abstract class ContextUtil {

    /**
     * Vérifie l'initialisation
     */
    private boolean init;

    private static Log log = LogFactory.getLog(ContextUtil.class);
    
    /**
     * Contexte global
     */
    private TopiaContext globalContext;

    /**
     * Transaction courante
     */
    private TopiaContext currTransaction;

    /**
     * Constructeur avec Class correspondant à la sous-classe qui hérite de ContextUtil
     * @param current
     */
    protected ContextUtil(Class current) {
        ContextUtil.log = LogFactory.getLog(current);
        init();
    }

    protected ContextUtil() {
        init();
    }

    /**
     * Initialisation du TopiaContext global
     * loadProperties est définit par les sous-classes pour
     *   charger le fichier de configuration et les classes pour Hibernate
     * Une initialisation de la base de données est possible via la méthode
     *   initDB() à redéfinir par les sous-classes
     */
    public void init() {
        if (!this.init) {
            if (log.isInfoEnabled()) {
                log.info("Initialize of TopiaContext...");
            }
            try {
                Properties conf = loadProperties();

                this.globalContext = TopiaContextFactory.getContext(conf);
                this.init = true;
                if (log.isInfoEnabled()) {
                    log.info("Context Ready !");
                }
                this.initDB();
            } catch (Exception ex) {
                log.error("Initialize error !!",ex);
            }
        }
    }

    /**
     * Initialisation de la base de données à redéfinir
     * @throws org.nuiton.topia.TopiaException
     */
    protected void initDB() throws TopiaException {

    }

    /**
     * Chargement du ou des fichiers de propriétés nécessaires à Topia
     * @param props
     * @throws java.io.IOException
     * @throws java.net.URISyntaxException
     */
    protected abstract Properties loadProperties() throws IOException, URISyntaxException;

    /**
     * Chargement d'un fichier de propriété
     * @param filename nom du fichier
     * @param props propriétés à charger
     * @throws java.net.URISyntaxException
     * @throws java.io.IOException
     */
    protected Properties loadFileProperties(String filename)
            throws URISyntaxException, IOException {
        Properties props = new Properties();
        URL url = Resource.getURL(filename);
        if (log.isDebugEnabled()) {
            log.debug(url);
        }
        props.load(url.openStream());
        return props;
    }

    public boolean isInit() {
        return this.init;
    }

    /**
     * Retourne le contexte global pour créer les transactions
     * @return Le contexte global
     */
    public TopiaContext getContext() {
        if (!this.init) {
            this.init();
        }
        return this.globalContext;
    }
    
    /**
     * Fermeture de la connexion à la BD
     */
    public void disconnect() {
        try {
            if (this.init) {
                this.globalContext.closeContext();
                if (log.isInfoEnabled()) {
                    log.info("Context closed");
                }
                this.init = false;
            }
        } catch (TopiaException ex) {
            log.error("Disconnect error !!",ex);
        }
    }

    /**
     * Récupération transaction courante (ou création)
     * @return
     * @throws org.nuiton.topia.TopiaException
     */
    public TopiaContext transaction() throws TopiaException {
        if (!this.init) {
            this.init();
        }
        if (this.currTransaction == null || this.currTransaction.isClosed()) {
            this.currTransaction = this.globalContext.beginTransaction();
        }
        return this.currTransaction;
    }

    /**
     * Commit de la transaction courante
     * @throws org.nuiton.topia.TopiaException
     */
    public void commit() throws TopiaException {
        this.transaction().commitTransaction();
    }

    public void rollback() throws TopiaException {
        this.transaction().rollbackTransaction();
    }

    /**
     * Fermeture de la transaction courante
     * @throws org.nuiton.topia.TopiaException
     */
    public void close() throws TopiaException {
        if (this.init && this.currTransaction != null) {
            this.currTransaction.closeContext();
        }
    }
}
