/*
 * *##% 
 * Nuiton Processor :: Api
 * Copyright (C) 2002 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.processor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.processor.filters.Filter;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

/**
 * Utilities methods.
 *
 * @author chemit
 * @since 1.0.0.0
 */
public class ProcessorUtil {

    /** Logger */
    private static final Log log = LogFactory.getLog(ProcessorUtil.class);
    
    /**
     * Instanciate a new {@link Processor} given an array of filters given by
     * their FQN separated by the given separator.
     *
     * @param filters the string representionf of the filters
     * @param separator the separator of filter
     * @return the instanciated processor
     * @throws Exception if any pb
     */
    public static Processor newProcessor(String filters,
                                         String separator) throws Exception {
        Filter[] result = getFilters(filters, separator);
        Processor processor = new Processor(result);
        return processor;
    }
    
    /**
     * Instanciate a array of filters given by thier FQN separated by the
     * given separator.
     *
     * @param filters the list of filters separated by the given separator
     * @param separator filter separator
     * @return the array of instanciated filters.
     *
     * @throws Exception if any pb
     */
    public static Filter[] getFilters(String filters,
                                      String separator) throws Exception {
        String[] filterList = filters.split(separator);
        Filter[] result = new Filter[filterList.length];
        for (int i = 0; i < filterList.length; i++) {

            try {
                // Class.forName semble fonctionner maintenant
                // avant il fallait utiliser getClass().forName
                result[i] = (Filter)
                        Class.forName(filterList[i].trim()).newInstance();
            } catch (Exception eee) {
                throw new IllegalArgumentException(
                        "Error during looking for '" + filterList[i].trim() +
                        "' class", eee);
            }
        }
        return result;
    }

    /**
     * Launch the process of the given {@code processor} to the given files.
     *
     * @param processor the processor to launch
     * @param in        the input file
     * @param out       the output file
     * @throws IOException if any problems while processing file
     * @since 1.0.3
     */
    public static void doProcess(Processor processor,
                                 String in, String out) throws IOException {
        doProcess(processor, new File(in), new File(out));
    }


    /**
     * Launch the process of the given {@code processor} to the given files.
     *
     * @param processor the processor to launch
     * @param in        the input file
     * @param out       the output file
     * @throws IOException if any problems while processing file
     * @since 1.0.3
     */
    public static void doProcess(Processor processor,
                                 File in, File out) throws IOException {
        FileReader input = new FileReader(in);
        try {
            FileWriter output = new FileWriter(out);
            try {
                processor.process(input, output);
            } catch (IOException eee) {
                if (log.isErrorEnabled()) {
                    log.error(
                            "Error while processing file " + in + " to " + out +
                            " with processor " + processor, eee);
                }
                throw eee;
            } finally {
                output.close();
            }
        } finally {
            input.close();
        }
    }
}
