/*
 * #%L
 * Nuiton Utils :: Nuiton Validator
 * 
 * $Id: BeanValidator.java 2059 2011-01-26 10:39:27Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.4.7/nuiton-validator/src/main/java/org/nuiton/validator/bean/BeanValidator.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.validator.bean;

import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;
import org.nuiton.util.converter.ConverterUtil;
import org.nuiton.validator.NuitonValidator;
import org.nuiton.validator.NuitonValidatorModel;
import org.nuiton.validator.NuitonValidatorProvider;
import org.nuiton.validator.NuitonValidatorResult;
import org.nuiton.validator.NuitonValidatorScope;

import javax.swing.event.EventListenerList;
import java.beans.Introspector;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * Validator for a javaBean object.
 * <p/>
 * A such validator is designed to validate to keep the validation of a bean,
 * means the bean is attached to the validator (field {@link #bean}.
 * <p/>
 * A such validator is also a JavaBean and you can listen his states
 * modifications via the classic java bean api.
 * <p/>
 * <strong>Note:</strong> The {@code BeanValidator} should never be used for
 * validation in a service approch since it needs to keep a reference to the
 * bean to validate.
 *
 * @author tchemit <chemit@codelutin.com>
 * @see BeanValidatorListener
 * @since 2.0
 */
public class BeanValidator<O> {

    /**
     * Name of the bounded property {@link #bean}.
     *
     * @see #bean
     * @see #getBean()
     * @see #setBean(Object)
     */
    public static final String BEAN_PROPERTY = "bean";

    /**
     * Name of the bounded property {@code context}.
     *
     * @see #getContext()
     * @see #setContext(String)
     */
    public static final String CONTEXT_PROPERTY = "context";

    /**
     * Name of the bounded property {@code scopes}.
     *
     * @see #getScopes()
     * @see #setScopes(NuitonValidatorScope...)
     */
    public static final String SCOPES_PROPERTY = "scopes";

    /**
     * Name of the bounded property {@link #valid}.
     *
     * @see #valid
     * @see #isValid()
     * @see #setValid(boolean)
     */
    public static final String VALID_PROPERTY = "valid";

    /**
     * Name of the bounded property {@link #changed}.
     *
     * @see #changed
     * @see #isChanged()
     * @see #setChanged(boolean)
     */
    public static final String CHANGED_PROPERTY = "changed";

    /** Logger. */
    protected static final Log log = LogFactory.getLog(BeanValidator.class);

    /** The bean to validate. */
    protected O bean;

    /** To chain to another validator (acting as parent of this one). */
    protected BeanValidator<?> parentValidator;

    /** The delegate validator used to validate the bean. */
    protected NuitonValidator<O> delegate;

    /**
     * State of validation (keep all messages of validation for the filled
     * bean).
     */
    protected NuitonValidatorResult messages;

    /**
     * State to indicate that validator has changed since the last time bean was
     * setted.
     */
    protected boolean changed;

    /** State of the validator (is true if no errors of error scope is found). */
    protected boolean valid = true;

    /**
     * State to know if the validator can be used (we keep this state for
     * performance reasons : do not want to compute this value each time a
     * validation is asked...).
     */
    protected boolean canValidate = true;

    /** map of conversion errors detected by this validator */
    protected Map<String, String> conversionErrors;

    /** Listener that listens on bean modification. */
    protected PropertyChangeListener l;

    /** delegate property change support */
    protected PropertyChangeSupport pcs;

    /** A list of event listeners for this validators */
    protected EventListenerList listenerList = new EventListenerList();

    /**
     * The provider of delegate validators.
     * <p/>
     * It will also produce validator model.
     *
     * @see NuitonValidatorProvider
     */
    protected final NuitonValidatorProvider validatorProvider;

    public BeanValidator(NuitonValidatorProvider validatorProvider,
                         Class<O> beanClass,
                         String context) {

        this(validatorProvider, beanClass,
             context,
             NuitonValidatorScope.values()
        );
    }

    public BeanValidator(NuitonValidatorProvider validatorProvider,
                         Class<O> beanClass,
                         String context,
                         NuitonValidatorScope... scopes) {

        // check if given bean class is Javabean compiliant
        boolean javaBeanCompiliant = BeanUtil.isJavaBeanCompiliant(beanClass);

        if (!javaBeanCompiliant) {

            throw new IllegalArgumentException(
                    beanClass.getName() + " is not JavaBean compiliant (" +
                    BeanUtil.ADD_PROPERTY_CHANGE_LISTENER + ", or " +
                    BeanUtil.REMOVE_PROPERTY_CHANGE_LISTENER +
                    " method not found).");
        }

        this.validatorProvider = validatorProvider;
        pcs = new PropertyChangeSupport(this);
        conversionErrors = new TreeMap<String, String>();

        // build delegate validator
        rebuildDelegateValidator(
                beanClass,
                context,
                scopes
        );

        // context has changed
        firePropertyChange(CONTEXT_PROPERTY,
                           null,
                           context
        );

        // scopes has changed
        firePropertyChange(SCOPES_PROPERTY,
                           null,
                           scopes
        );

        l = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                // the bean has changed, replay validation
                doValidate();
            }
        };
    }

    /**
     * Obtain the {@link #changed} property value.
     * <p/>
     * Returns {@code true} if bean was modified since last
     * time a bean was attached.
     *
     * @return {@code true} if bean was modified since last attachement of
     *         a bean.
     */
    public boolean isChanged() {
        return changed;
    }

    /**
     * To force the value of the property {@link #changed}.
     *
     * @param changed flag to force reset of property {@link #changed}
     */
    public void setChanged(boolean changed) {
        this.changed = changed;

        // force the property to be fired (never pass the older value)
        firePropertyChange(CHANGED_PROPERTY, null, changed);
    }

    public boolean isCanValidate() {
        return canValidate;
    }

    public void setCanValidate(boolean canValidate) {
        this.canValidate = canValidate;
    }

    /**
     * Obtain the {@link #valid} property value.
     *
     * @return {@code true} if attached bean is valid (no error or fatal messages)
     */
    public boolean isValid() {
        return valid;
    }

    /**
     * Change the value of the {@link #valid} property.
     *
     * @param valid the new value of the property
     */
    public void setValid(boolean valid) {
        this.valid = valid;

        // force the property to be fired (never pass the older value)
        firePropertyChange(VALID_PROPERTY, null, valid);
    }

    /**
     * Obtain the actual bean attached to the validator.
     *
     * @return the bean attached to the validor or {@code null} if no bean
     *         is attached
     */
    public O getBean() {
        return bean;
    }

    /**
     * Change the attached bean.
     * <p/>
     * As a side effect, the internal {@link #messages} will be reset.
     *
     * @param bean the bean to attach (can be {@code null} to reset the
     *             validator).
     */
    public void setBean(O bean) {
        O oldBean = this.bean;
        if (log.isDebugEnabled()) {
            log.debug(this + " : " + bean);
        }

        // clean conversions of previous bean
        conversionErrors.clear();

        if (oldBean != null) {
            try {
                BeanUtil.removePropertyChangeListener(l, oldBean);
            } catch (Exception eee) {
                if (log.isInfoEnabled()) {
                    log.info("Can't register as listener for bean " + oldBean.getClass() +
                             " for reason " + eee.getMessage(), eee);
                }
            }
        }
        this.bean = bean;

        setCanValidate(!getDelegate().getEffectiveFields().isEmpty() && bean != null);

        if (bean == null) {

            // remove all messages for all fields of the validator

            mergeMessages(null);

        } else {
            try {

                BeanUtil.addPropertyChangeListener(l, bean);
            } catch (Exception eee) {
                if (log.isInfoEnabled()) {
                    log.info("Can't register as listener for bean " + bean.getClass() +
                             " for reason " + eee.getMessage(), eee);
                }
            }
            validate();
        }
        setChanged(false);
        setValid(messages == null || messages.isValid());
        firePropertyChange(BEAN_PROPERTY, oldBean, bean);
    }

    public String getContext() {
        return delegate.getModel().getContext();
    }

    public void setContext(String context) {

        String oldContext = getContext();

        if (context == null && oldContext == null || context != null && context.equals(oldContext)) {

            // same context do nothing
            return;
        }
        NuitonValidatorModel<O> validatorModel = delegate.getModel();

        // compute the new validator model
        NuitonValidatorScope[] scopes = validatorModel.getScopes().toArray(
                new NuitonValidatorScope[validatorModel.getScopes().size()]);

        rebuildDelegateValidator(
                validatorModel.getType(),
                context,
                scopes
        );

        firePropertyChange(CONTEXT_PROPERTY,
                           oldContext,
                           context
        );
    }

    public Set<NuitonValidatorScope> getScopes() {
        return delegate.getModel().getScopes();
    }

    public Set<NuitonValidatorScope> getEffectiveScopes() {
        return delegate.getEffectiveScopes();
    }

    public Set<String> getEffectiveFields() {
        return delegate.getEffectiveFields();
    }

    public Set<String> getEffectiveFields(NuitonValidatorScope scope) {
        return delegate.getEffectiveFields(scope);
    }

    public void setScopes(NuitonValidatorScope... scopes) {

        Set<NuitonValidatorScope> oldScopes = getScopes();

        NuitonValidatorModel<O> validatorModel = delegate.getModel();

        rebuildDelegateValidator(
                validatorModel.getType(),
                validatorModel.getContext(),
                scopes
        );

        firePropertyChange(SCOPES_PROPERTY,
                           oldScopes,
                           scopes
        );
    }

    public Class<O> getType() {
        return delegate.getModel().getType();
    }

    protected void rebuildDelegateValidator(Class<O> beanType,
                                            String context,
                                            NuitonValidatorScope... scopes) {

        // changing context could change fields definition
        // so dettach bean, must rebuild the fields

        // Dettach the bean before any thing, because with the new delegate
        // validator some old fields could not be used any longer, and then
        // listeners will never have the full reset of their model...
        if (bean != null) {
            setBean(null);
        }

        if (scopes==null || scopes.length == 0) {
            scopes = NuitonValidatorScope.values();
        }

        // compute the new validator model
        NuitonValidatorModel<O> validatorModel = validatorProvider.getModel(beanType,
                                                                            context,
                                                                            scopes
        );

        // remove old delegate validator
        delegate = validatorProvider.newValidator(validatorModel);
    }

    public BeanValidator<?> getParentValidator() {
        return parentValidator;
    }

    public void setParentValidator(BeanValidator<?> parentValidator) {
        this.parentValidator = parentValidator;
    }

    public boolean hasFatalErrors() {
        boolean result = messages != null && messages.hasFatalMessages();
        return result;
    }

    public boolean hasErrors() {
        boolean result = messages != null && messages.hasErrorMessagess();
        return result;
    }

    public boolean hasWarnings() {
        boolean result = messages != null && messages.hasWarningMessages();
        return result;
    }

    public boolean hasInfos() {
        boolean result = messages != null && messages.hasInfoMessages();
        return result;
    }

    /**
     * Test a the validator contains the field given his name
     *
     * @param fieldName the name of the searched field
     * @return <code>true</code> if validator contaisn this field,
     *         <code>false</code> otherwise
     */
    public boolean containsField(String fieldName) {
        Set<String> effectiveFields = getDelegate().getEffectiveFields();
        boolean result = effectiveFields.contains(fieldName);
        return result;
    }

    public boolean isValid(String fieldName) {
        if (messages == null) {

            // no message, so this is valid
            return true;
        }

        // field is valid if no fatal messages nor error messages
        boolean result = !(
                messages.hasMessagesForScope(fieldName, NuitonValidatorScope.FATAL) ||
                messages.hasMessagesForScope(fieldName, NuitonValidatorScope.ERROR));

        return result;
    }

    public NuitonValidatorScope getHighestScope(String field) {
        if (messages == null) {

            // no messages
            return null;
        }

        NuitonValidatorScope scope = messages.getFieldHighestScope(field);
        return scope;
    }

    /**
     * Convert a value.
     * <p/>
     * If an error occurs, then add an error in validator.
     *
     * @param <T>        the type of conversion
     * @param fieldName  the name of the bean property
     * @param value      the value to convert
     * @param valueClass the type of converted value
     * @return the converted value, or null if conversion was not ok
     */
    @SuppressWarnings({"unchecked"})
    public <T> T convert(String fieldName, String value, Class<T> valueClass) {
        if (fieldName == null) {
            throw new IllegalArgumentException("fieldName can not be null");
        }
        if (valueClass == null) {
            throw new IllegalArgumentException("valueClass can not be null");
        }

        // on ne convertit pas si il y a un bean et que le resultat de la
        // validation pourra etre affiche quelque part
        if (!isCanValidate() || value == null) {
            return null;
        }

        // remove the previous conversion error for the field
        conversionErrors.remove(fieldName);

        T result;
        try {
            Converter converter = ConverterUtil.getConverter(valueClass);
            if (converter == null) {
                throw new RuntimeException(
                        "could not find converter for the type " + valueClass);
            }
            result = (T) converter.convert(valueClass, value);
            /* Why this test ? if (result != null && !value.equals(result.toString())) {
            conversionErrors.put(fieldName, "error.convertor." + Introspector.decapitalize(valueClass.getSimpleName()));
            result = null;
            validate();
            }*/
        } catch (ConversionException e) {
            // get
            String s = Introspector.decapitalize(valueClass.getSimpleName());
            conversionErrors.put(fieldName, "error.convertor." + s);
            result = null;
            validate();
        }
        return result;
    }

    public void doValidate() {
        validate();
        setValid(messages == null || messages.isValid());
        setChanged(true);
    }

    /**
     * il faut eviter le code re-intrant (durant une validation, une autre est
     * demandee). Pour cela on fait la validation dans un thread, et tant que la
     * premiere validation n'est pas fini, on ne repond pas aux solicitations.
     * Cette method est public pour permettre de force une validation par
     * programmation, ce qui est utile par exemple si le bean ne supporte pas
     * les {@link PropertyChangeListener}
     * <p/>
     * <b>Note:</b> la methode est protected et on utilise la methode
     * {@link #doValidate()} car la méthode ne modifie pas les etats
     * internes et cela en rend son utilisation delicate (le validateur entre
     * dans un etat incoherent par rapport aux messages envoyés).
     */
    public void validate() {

        // on ne valide que si il y a un bean et que le resultat de la validation
        // pourra etre affiche quelque part
        if (!isCanValidate()) {
            return;
        }

        NuitonValidatorResult result = getDelegate().validate(bean);

        // treate conversion errors
        // reinject them
        for (Map.Entry<String, String> entry : conversionErrors.entrySet()) {


            // remove from validation, errors occurs on this field
            String field = entry.getKey();


            List<String> errors = result.getErrorMessages(field);

            String conversionError = entry.getValue();
            if (errors != null) {
                errors.clear();
                errors.add(conversionError);
            } else {
                errors = Collections.singletonList(conversionError);
            }

            result.setMessagesForScope(NuitonValidatorScope.ERROR, field, errors);
        }

        mergeMessages(result);

        if (parentValidator != null) {
            // chained validation
            // the parent validator should not be changed from this validation
            boolean wasModified = parentValidator.isChanged();
            parentValidator.doValidate();
            if (!wasModified) {
                // push back old state
                parentValidator.setChanged(false);
            }
        }
    }

    @Override
    public String toString() {
        return super.toString() + "<beanClass:" + delegate.getModel().getType() +
               ", context:" + getContext() + ">";
    }

    public void addBeanValidatorListener(BeanValidatorListener listener) {
        listenerList.add(BeanValidatorListener.class, listener);
    }

    public void removeBeanValidatorListener(BeanValidatorListener listener) {
        listenerList.remove(BeanValidatorListener.class, listener);
    }

    public BeanValidatorListener[] getBeanValidatorListeners() {
        return listenerList.getListeners(BeanValidatorListener.class);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
                                          PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName,
                                             PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(propertyName, listener);
    }

    public void firePropertyChange(String propertyName,
                                   Object oldValue,
                                   Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }

    protected BeanValidatorEvent createEvent(String field,
                                             NuitonValidatorScope scope,
                                             String[] toAdd,
                                             String[] toDelete) {
        BeanValidatorEvent evt = new BeanValidatorEvent(
                this,
                field,
                scope,
                toAdd,
                toDelete
        );
        return evt;
    }

    protected void fireFieldChanged(String field,
                                    NuitonValidatorScope scope,
                                    String[] toAdd,
                                    String[] toDelete) {

        BeanValidatorEvent evt = createEvent(
                field,
                scope,
                toAdd,
                toDelete
        );

        fireFieldChanged(evt);
    }

    protected void fireFieldChanged(BeanValidatorEvent evt) {

        for (BeanValidatorListener listener :
                listenerList.getListeners(BeanValidatorListener.class)) {
            listener.onFieldChanged(evt);
        }
    }

    protected void mergeMessages(NuitonValidatorResult newMessages) {

        if (newMessages == null && messages == null) {

            // no messages ever registred and ask to delete them, so nothing
            // to do
            return;
        }

        Set<NuitonValidatorScope> scopes = getDelegate().getEffectiveScopes();

        // list of events to send after the merge of messages
        List<BeanValidatorEvent> events = new ArrayList<BeanValidatorEvent>();

        for (NuitonValidatorScope scope : scopes) {

            // do the merge at scope level
            mergeMessages(scope, newMessages, events);

        }

        if (newMessages != null) {

            //TODO tchemit 2011-01-23 Perharps it will necessary to clear the messages for memory performance ?

            // finally keep the new messages as the current messages
            messages = newMessages;
        }

        if (CollectionUtils.isNotEmpty(events)) {

            // send all messages
            for (BeanValidatorEvent event : events) {
                fireFieldChanged(event);
            }
        }
    }

    protected void mergeMessages(NuitonValidatorScope scope,
                                 NuitonValidatorResult newMessages,
                                 List<BeanValidatorEvent> events) {

        if (newMessages == null) {

            // special case to empty all messages

            List<String> fieldsForScope = messages.getFieldsForScope(scope);

            for (String field : fieldsForScope) {
                List<String> messagesForScope = messages.getMessagesForScope(field, scope);
                events.add(createEvent(field, scope, null, messagesForScope.toArray(new String[messagesForScope.size()])));
            }

            // suppress all messages for this scope
            messages.clearMessagesForScope(scope);


        } else {

            List<String> newFields = newMessages.getFieldsForScope(scope);

            if (messages == null) {

                // first time of a merge, just add new messages

                for (String field : newFields) {
                    List<String> messagesForScope = newMessages.getMessagesForScope(field, scope);
                    events.add(createEvent(field, scope, messagesForScope.toArray(new String[messagesForScope.size()]), null));
                }

                // nothing else to do
                return;
            }

            List<String> oldFields = messages.getFieldsForScope(scope);

            Iterator<String> itr;

            // detects field with only new messages
            itr = newFields.iterator();
            while (itr.hasNext()) {
                String newField = itr.next();

                if (!oldFields.contains(newField)) {

                    // this fields has now messages but not before : new messages
                    List<String> messagesForScope = newMessages.getMessagesForScope(newField, scope);
                    events.add(createEvent(newField, scope, messagesForScope.toArray(new String[messagesForScope.size()]), null));

                    // treated field
                    itr.remove();
                }
            }

            // detects fields with only obsolete messages
            itr = oldFields.iterator();
            while (itr.hasNext()) {
                String oldField = itr.next();

                if (!newFields.contains(oldField)) {

                    // this fields has no more messages
                    List<String> messagesForScope = messages.getMessagesForScope(oldField, scope);
                    events.add(createEvent(oldField, scope, null, messagesForScope.toArray(new String[messagesForScope.size()])));

                    // treated field
                    itr.remove();
                }
            }

            // now deal with mixte field (toAdd and toDelete)
            for (String field : newFields) {

                List<String> newMessagesForScope = newMessages.getMessagesForScope(field, scope);
                List<String> oldMessagesForScope = messages.getMessagesForScope(field, scope);

                // get old obsoletes messages to delete
                Set<String> toDelete = new HashSet<String>(oldMessagesForScope);
                toDelete.removeAll(newMessagesForScope);

                // get new messages to add
                Set<String> toAdd = new HashSet<String>(newMessagesForScope);
                toAdd.removeAll(oldMessagesForScope);

                events.add(createEvent(
                        field,
                        scope,
                        toAdd.isEmpty() ? null : toAdd.toArray(new String[toAdd.size()]),
                        toDelete.isEmpty() ? null : toDelete.toArray(new String[toDelete.size()])
                ));

            }
        }
    }

    protected NuitonValidator<O> getDelegate() {
        return delegate;
    }

}
