/*
 * *##% 
 * JRedmine client
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.model.io.xpp3;

import java.util.Set;
import org.nuiton.jredmine.TestHelper;
import java.beans.Introspector;
import org.nuiton.jredmine.model.*;
import java.io.File;
import java.io.FileInputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.plexus.util.IOUtil;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 *
 * @author chemit
 * @since 1.0.0
 */
public class RedmineXpp3HelperTest {

    protected static final Log log = LogFactory.getLog(RedmineXpp3HelperTest.class);
    //TODO use the RedmineModelEnum and finish the model mock (miss some types)
    private static final RedmineModelEnum[] TYPE_ORDER = new RedmineModelEnum[]{
        RedmineModelEnum.attachment,
        RedmineModelEnum.issue,
        RedmineModelEnum.project,
        RedmineModelEnum.tracker,
        RedmineModelEnum.user,
        RedmineModelEnum.version,
        RedmineModelEnum.issueStatus,
        RedmineModelEnum.issuePriority,
        RedmineModelEnum.issueCategory,
        RedmineModelEnum.news
    };
    protected static File testDir;
    protected static RedmineXpp3Helper builder;
    File file;
    protected boolean verbose;
    protected static int universeSize;

    @BeforeClass
    public static void setUpClass() throws Exception {

        testDir = TestHelper.getTestDir(RedmineXpp3HelperTest.class.getPackage().getName(), "target" + File.separator + "test-classes");

        builder = new RedmineXpp3Helper();

        log.info("test dir : " + TestHelper.getRelativePath(TestHelper.getBasedir(), testDir));

        // setup memory model
        TestHelper.loadMemoryModel();

        universeSize = TYPE_ORDER.length;

        if (universeSize < RedmineModelEnum.values().length) {
            log.warn("There is some types of the model which are not tested :");
            Set<RedmineModelEnum> keySet = TestHelper.getMemoryModel().keySet();
            for (RedmineModelEnum t : RedmineModelEnum.values()) {
                if (!keySet.contains(t)) {
                    log.warn(t);
                }
            }
        }

    }

    @AfterClass
    public static void tearDownClass() throws Exception {
        testDir = null;
        builder = null;
    }

    @Before
    public void setUp() {
        verbose = TestHelper.isVerbose();
    }

    @After
    public void tearDown() {
        file = null;
    }

    /**
     * Test the method {@code read} of class ModelBuilder.
     *
     * @throws Exception
     */
    @Test
    public void testRead() throws Exception {

        TestAction action = new TestAction() {

            @Override
            protected <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception {

                File file = new File(rootDir, Introspector.decapitalize(type.getSimpleName()) + ".xml");

                if (verbose) {
                    log.info("will test " + TestHelper.getRelativePath(TestHelper.getBasedir(), file));
                }

                // test from file

                T actual = builder.readObject(type, file);

                TestHelper.assertMyEquals(type, expected.get(0), actual);

                // test from string

                String txt = IOUtil.toString(new FileInputStream(file));

                actual = builder.readObject(type, txt);

                TestHelper.assertMyEquals(type, expected.get(0), actual);
            }
        };

        action.run("single",
                Arrays.asList(TestHelper.getModel(Attachment.class).get(0)),
                Arrays.asList(TestHelper.getModel(Issue.class).get(0)),
                Arrays.asList(TestHelper.getModel(Project.class).get(0)),
                Arrays.asList(TestHelper.getModel(Tracker.class).get(0)),
                Arrays.asList(TestHelper.getModel(User.class).get(0)),
                Arrays.asList(TestHelper.getModel(Version.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssueStatus.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssuePriority.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssueCategory.class).get(0)),
                Arrays.asList(TestHelper.getModel(News.class).get(0)));
    }

    /**
     * Test the method {@code readArray} of class ModelBuilder.
     *
     * @throws Exception
     */
    @Test
    @SuppressWarnings("unchecked")
    public void testReadArray() throws Exception {

        TestAction action = new TestAction() {

            @Override
            protected <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception {

                File file = new File(rootDir, Introspector.decapitalize(type.getSimpleName()) + "s.xml");

                if (verbose) {
                    log.info("will test " + TestHelper.getRelativePath(TestHelper.getBasedir(), file));
                }

                // test from file

                T[] actual = builder.readObjects(type, file);

                TestHelper.assertMyListEquals(type, (List<T>) expected, actual);

                // test from text
                String txt = IOUtil.toString(new FileInputStream(file));

                actual = builder.readObjects(type, txt);

                TestHelper.assertMyListEquals(type, (List<T>) expected, actual);
            }
        };

        action.run("array-empty",
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST,
                Collections.EMPTY_LIST);


        action.run("array-singleton",
                Arrays.asList(TestHelper.getModel(Attachment.class).get(0)),
                Arrays.asList(TestHelper.getModel(Issue.class).get(0)),
                Arrays.asList(TestHelper.getModel(Project.class).get(0)),
                Arrays.asList(TestHelper.getModel(Tracker.class).get(0)),
                Arrays.asList(TestHelper.getModel(User.class).get(0)),
                Arrays.asList(TestHelper.getModel(Version.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssueStatus.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssuePriority.class).get(0)),
                Arrays.asList(TestHelper.getModel(IssueCategory.class).get(0)),
                Arrays.asList(TestHelper.getModel(News.class).get(0)));

        action.run("array-multi",
                TestHelper.getModel(Attachment.class),
                TestHelper.getModel(Issue.class),
                TestHelper.getModel(Project.class),
                TestHelper.getModel(Tracker.class),
                TestHelper.getModel(User.class),
                TestHelper.getModel(Version.class),
                TestHelper.getModel(IssueStatus.class),
                TestHelper.getModel(IssuePriority.class),
                TestHelper.getModel(IssueCategory.class),
                TestHelper.getModel(News.class));

    }

    /**
     * A little action wrapper to simplify this test class :)
     *
     */
    public static abstract class TestAction {

        /**
         * Implements the logic of action for the given {@code type} and
         * the {@code expected} datas.
         *
         * @param <T> the type of data used
         * @param rootDir the rootdir file of the local cache
         * @param type the type of data used
         * @param expected the list of expected datas
         * @throws Exception if any pb
         */
        protected abstract <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception;

        public void run(String name, List<?>... expected) throws Exception {
            Assert.assertEquals(universeSize, expected.length);
            File rootDir = new File(testDir, name);

            for (int i = 0; i < universeSize; i++) {

                RedmineModelEnum next = TYPE_ORDER[i];

                runForType(rootDir, next.getModelType(), expected[i]);
            }
        }
    }
}
