package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * %%
 * Copyright (C) 2012 - 2015 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */




import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Generates a java bean and a utility class around it. This transformer acts like {@link SimpleJavaBeanTransformer}
 * but with no interface generation (plus there is not factory generation at all).
 *
 * For example:
 * <pre>
 *     AbstractBoat
 *     Boat (extends AbstractBoat)
 *     AbstractBoats
 *     Boats (extends AbstractBoats)
 * </pre>
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.SimpleJavaBeanWithNoInterfaceTransformer"
 * @since 3.0
 */
public class SimpleJavaBeanWithNoInterfaceTransformer extends AbstractJavaBeanTransformer {

    /** Logger. */
    private static final Log log = LogFactory.getLog(SimpleJavaBeanWithNoInterfaceTransformer.class);

    @Override
    public void transformFromClass(ObjectModelClass input) {

        ObjectModelPackage aPackage = getPackage(input);

        if (JavaTemplatesStereoTypes.hasBeanStereotype(input, aPackage)) {

            String className = getBeanClassName(aPackage, input);
            String abstractClassName = "Abstract" + className;
            String defaultClassName = getBeanDefaultsClassName(aPackage, input);
            String abstractDefaultClassName = "Abstract" + defaultClassName;

            boolean generateDefaults = canGenerateBeanDefaults(aPackage, input);
            boolean generateAbstractClass = canGenerateAbstractBean(input, abstractClassName);
            boolean generateClass = canGenerateBean(input, className);

            String prefix = getConstantPrefix(input);
            setConstantPrefix(prefix);

            if (generateAbstractClass) {
                generateAbstractBeanClass(aPackage, input, abstractClassName);
            }

            if (generateClass) {
                generateBeanClass(input, className, abstractClassName);
            }

            if (generateDefaults) {
                generateBeanDefaults(aPackage, input, className, abstractDefaultClassName, defaultClassName);
            }
        }

    }

    protected boolean canGenerateAbstractBean(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean canGenerate = !isInClassPath(fqn);
        return canGenerate;
    }

    protected boolean canGenerateBean(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean canGenerate = !isInClassPath(fqn);

        if (canGenerate) {

            // check there is no operation on input
            if (!input.getOperations().isEmpty()) {

                canGenerate = false;
//                throw new GeneratorException(
//                        "Can't generate a simple bean as class " +
//                        fqn +
//                        " contains so operations." +
//                        "\nUse instead the JavaBeanTransformer.");
            }
        }
        return canGenerate;
    }

    protected boolean canGenerateBeanDefaults(ObjectModelPackage aPackage, ObjectModelClass input) {

        boolean withInput = input != null;

        boolean canGenerate = getJavaTemplatesTagValues().isSimpleBeanWithNoInterfaceGenerateDefaults(input, aPackage, model);

        if (canGenerate) {

            if (withInput) {

                // class in not abstract
                // class is a bean

                canGenerate = !input.isAbstract() && JavaTemplatesStereoTypes.hasBeanStereotype(input, aPackage);

            }
        }

        return canGenerate;
    }

    protected ObjectModelClass generateBeanClass(ObjectModelClass input,
                                                 String className,
                                                 String abstractClassName) {

        ObjectModelClass output;

        if (input.isAbstract()) {
            output = createAbstractClass(className, input.getPackageName());
        } else {
            output = createClass(className, input.getPackageName());
        }

        setSuperClass(output, abstractClassName);

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        addSerializable(input, output, true);

        return output;
    }

    protected ObjectModelClass generateAbstractBeanClass(ObjectModelPackage aPackage,
                                                         ObjectModelClass input,
                                                         String className) {

        String superClass = null;

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesStereoTypes.hasBeanStereotype(superclass, aPackage)) {
                    superClassIsBean = true;
                    superClass = superclass.getPackageName() + "." + getBeanClassName(aPackage, superclass);
                    break;
                } else {
                    superClass = superclass.getQualifiedName();
                }
            }
        }

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClass = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceSuperClassTagValue(input, aPackage, model);
            if (superClass != null) {

                // will act as if super class is a bean
                superClassIsBean = true;
            }
        }

        ObjectModelClass output;

        output = createAbstractClass(className, input.getPackageName());

        if (superClass != null) {
            setSuperClass(output, superClass);
        }
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean serializableFound;

        serializableFound = addInterfaces(input, output, null);

        generateI18nBlockAndConstants(aPackage, input, output);

        addSerializable(input, output, serializableFound || superClassIsBean);

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        boolean usePCS = getJavaTemplatesTagValues().isGeneratePropertyChangeSupport(input, aPackage, model);

        boolean generateBooleanGetMethods = getEugeneTagValues().isGenerateBooleanGetMethods(input, aPackage, model);
        boolean generateNotEmptyCollections = getJavaTemplatesTagValues().isGenerateNotEmptyCollections(input, aPackage, model);

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output,
                           attr,
                           usePCS,
                           generateBooleanGetMethods,
                           generateNotEmptyCollections);
        }

        if (!superClassIsBean) {
            addDefaultMethodForNoneBeanSuperClass(output, usePCS, properties);
        }
        return output;
    }

    protected void generateBeanDefaults(ObjectModelPackage aPackage,
                                        ObjectModelClass aClass,
                                        String typeName,
                                        String abstractClassName,
                                        String defaultClassName) {

        boolean generateDefault = canGenerateBeanDefaults(aPackage, aClass);

        if (!generateDefault) {
            return;
        }

        String packageName = aClass.getPackageName();
//        String typeName = getBeanInterfaceName(aPackage, aClass);
//        String typeBeanName = getBeanClassName(aPackage, aClass);

        if (!isInClassPath(packageName, defaultClassName)) {

            // generate defaults class

            ObjectModelClass output = createClass(defaultClassName, packageName);
            setSuperClass(output, packageName + "." + abstractClassName);
            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }
        }

        if (!isInClassPath(packageName, abstractClassName)) {

            // generate abstract defaults class

            // try to find a super class by tag-value
            String superClassName =
                    getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceDefaultsSuperClassTagValue(
                            aClass, aPackage, model);

            ObjectModelClass output = createAbstractClass(abstractClassName, packageName);
            if (StringUtils.isNotBlank(superClassName)) {
                setSuperClass(output, superClassName);
            }

            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }

            ObjectModelOperation operation = addOperation(
                    output,
                    "typeOf" + typeName,
                    "<BeanType extends " + typeName + "> Class<BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeName+".class;\n"
+"    "
            );
            operation = addOperation(
                    output,
                    "new" + typeName,
                    typeName,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return new "+typeName+"();\n"
+"    "
            );

            addImport(output, Binder.class);
            addImport(output, BinderFactory.class);
            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType,BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        BeanType result = new"+typeName+"(source, binder);\n"
+"        return result;\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            addParameter(operation, "Binder<BeanType, BeanType>", "binder");
            setOperationBody(operation, ""
    +"\n"
+"        BeanType result = (BeanType) new"+typeName+"();\n"
+"        binder.copy(source, result);\n"
+"        return result;\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "copy" + typeName,
                    "<BeanType extends " + typeName + "> void",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            addParameter(operation, "BeanType", "target");
            setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType,BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        binder.copy(source, target);\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "copy" + typeName,
                    "<BeanType extends " + typeName + "> void",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            addParameter(operation, "BeanType", "target");
            addParameter(operation, "Binder<BeanType, BeanType>", "binder");
            setOperationBody(operation, ""
    +"\n"
+"        binder.copy(source, target);\n"
+"    "
            );
        }
    }

    protected Collection<ObjectModelOperation> getPublicOperations(ObjectModelClass clazz) {

        Collection<ObjectModelOperation> result = new ArrayList<ObjectModelOperation>();
        for (ObjectModelOperation operation : clazz.getOperations()) {

            ObjectModelJavaModifier visibility = ObjectModelJavaModifier.fromVisibility(operation.getVisibility());
            if (ObjectModelJavaModifier.PUBLIC == visibility) {
                result.add(operation);
            }
        }
        return result;
    }

    protected String getBeanClassName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String classNamePrefix = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceClassNamePrefixTagValue(input, aPackage, model);
        String classNameSuffix = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceClassNameSuffixTagValue(input, aPackage, model);

        return generateName(classNamePrefix, input.getName(), classNameSuffix);
    }

    protected String getBeanDefaultsClassName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String classNamePrefix = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceDefaultsClassNamePrefixTagValue(input, aPackage, model);
        String classNameSuffix = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceDefaultsClassNameSuffixTagValue(input, aPackage, model);
        return generateName(classNamePrefix, input.getName(), classNameSuffix);
    }

    protected String generateName(String prefix, String name, String suffix) {
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            sb.append(prefix);
        }
        sb.append(name);
        if (StringUtils.isNotEmpty(suffix)) {
            sb.append(suffix);
        }
        return sb.toString();
    }
}

