/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: SampleRowForm.java 1570 2012-03-19 17:01:27Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.3.2/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/SampleRowForm.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.SamplingFilterImpl;
import fr.ifremer.wao.bean.SamplingStrategy;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.bean.ValidationResult;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.DCF5Code;
import fr.ifremer.wao.entity.FishingGearDCF;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.Profession;
import fr.ifremer.wao.entity.ProfessionImpl;
import fr.ifremer.wao.entity.SampleMonth;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.SampleRowLog;
import fr.ifremer.wao.entity.TargetSpeciesDCF;
import fr.ifremer.wao.entity.TerrestrialDivision;
import fr.ifremer.wao.entity.TerrestrialLocation;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.data.WaoActivationContext;
import fr.ifremer.wao.ui.data.WaoPropertyChangeListener;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.Field;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.BeanModelSource;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.PeriodDates;
import org.nuiton.util.StringUtil;
import org.slf4j.Logger;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import static org.nuiton.i18n.I18n.n_;

/**
 * SampleRowForm
 *
 * Created: 26 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication(allowedRoles = { UserRole.ADMIN,
                                         UserRole.COORDINATOR // for ObsDeb only
                                       },
                        readOnlyAllowed = false)
@Import(stylesheet = "context:css/sampling.css")
public class SampleRowForm {

    @Inject
    private Logger log;
    
    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private Messages messages;

    @SessionState
    @Property
    private ConnectedUser user;
    
    @Inject
    private PropertyAccess propertyAccess;

    private boolean edited;

    private WaoActivationContext waoActivationContext;

    void onActivate(EventContext ec) {
        waoActivationContext = WaoActivationContext.newContextFromTapestry(ec);
        sampleRowId = waoActivationContext.getSampleRowId();
    }

    String[] onPassivate() {
        waoActivationContext = WaoActivationContext.newEmptyContext();
        waoActivationContext.setSampleRowId(sampleRowId);
        return waoActivationContext.toStrings();
    }

    void setupRender() throws WaoException {
        // Reset data
        sampleRow = null;
        getSampleRow();
        sampleMonths = null;
        getSampleMonths();
        sampleRowLog = null;
        getSampleRowLog();

        // Reset persist select
        professionSelectModel = null;
        getProfessionSelectModel();
        programSelectModel = null;
        getProgramSelectModel();
        fishingZoneSelectModel = null;
        getFishingZoneSelectModel();
        companySelectModel = null;
        getCompanySelectModel();
        
        professionId = null;
        sampleRowForm.clearErrors();

        if (!isCreateMode()) {
            programId = getSampleRow().getProgramName();
        } else {
            programId = null;
        }

        // initialize companyId if exist for companySelect
        Company company = getSampleRow().getCompany();
        if (company != null) {
            companyId = company.getTopiaId();
        }
        
        immatriculations = getSampleRow().getMainElligibleBoatsAsString();

        if (getSampleRow().getTerrestrialLocation() != null) {
            terrestrialLocationId = getSampleRow().getTerrestrialLocation().getTopiaId();
        }
    }

    public boolean showObsVente() {
        return user.getProfile().getObsProgram() == ObsProgram.OBSVENTE;
    }

    @Inject
    private Block obsDebBlock;

    @Inject
    private Block obsMerVenteBlock;

    public Block getProgramBlock() {
        Block block;
        // what we show depends on the program
        if (ObsProgram.OBSDEB.equals(sampleRow.getObsProgram())) {
            block = obsDebBlock;
        } else {
            // form suitable for both ObsMer and ObsVente
            block = obsMerVenteBlock;
        }
        return block;
    }
    /*************************** PROFESSION ***********************************/

    /** Select model which contains all professions **/
    @Persist
    private GenericSelectModel<SampleRow> professionSelectModel;

    /** The current selected professionId, this is actually a sample row topia Id **/
    @Property
    private String professionId;

    /** Bean model source to create the BeanModel for profession **/
    @Inject
    private BeanModelSource beanModelSource;

    /** Component resources needed to create the BeanModel for profession **/
    @Inject
    private ComponentResources resources;

    /**
     * Get SelectModel for profession input Select. The select contains all existing professions.
     *
     * @return a GenericSelectModel<Profession>
     * @throws WaoException
     */
    public GenericSelectModel<SampleRow> getProfessionSelectModel() throws WaoException {
        if (professionSelectModel == null) {
            List<SampleRow> professions = serviceSampling.getSampleRowsByFilter(new SamplingFilterImpl());
            professionSelectModel = new GenericSelectModel<SampleRow>(professions, SampleRow.class,
                    "professionDescription", SampleRow.TOPIA_ID, propertyAccess);
        }
        return professionSelectModel;
    }

    /**
     * Get the Profession model for BeanDisplay component.
     *
     * @return a BeanModel<Profession>
     */
    public BeanModel<Profession> getProfessionModel() {
        BeanModel<Profession> professionModel = beanModelSource.createDisplayModel(Profession.class, resources.getMessages());
        professionModel.exclude(TopiaEntity.TOPIA_ID);
        professionModel.exclude(TopiaEntity.TOPIA_CREATE_DATE);
        professionModel.exclude(TopiaEntity.TOPIA_VERSION);
        return professionModel;
    }

    /**
     * Action on submit : addNewProfession. Used to set a new profession from the selected one.
     *
     * @throws WaoException
     */
    void onSelectedFromAddNewProfession() throws WaoException {
        Profession profession = new ProfessionImpl();
        if ( ! StringUtils.isEmpty(professionId)) {
            SampleRow professionSelected = getProfessionSelectModel().findObject(professionId);
            if (log.isDebugEnabled()) {
                log.debug("add new profession from : " + professionSelected);
            }
            profession = serviceSampling.getNewProfession(professionSelected.getProfession());

            getSampleRow().setdCF5Code(professionSelected.getdCF5Code());
            dcf5codes = null;
        }
        getSampleRow().setProfession(profession);
        edited = true;
    }

    private String dcf5codes;

    public String getDcf5codes() {
        if (dcf5codes == null) {
            Collection<DCF5Code> codes = getSampleRow().getdCF5Code();
            if (CollectionUtils.isEmpty(codes)) {
                dcf5codes = "";
            } else {
                dcf5codes = StringUtil.join(codes, ", ", true);
            }
        }
        return dcf5codes;
    }

    public void setDcf5codes(String dcf5codes) {
        this.dcf5codes = dcf5codes;
    }

    /*************************** FISHING ZONE *********************************/

    /** Select model for the FishingZone list **/
    @Persist
    private GenericSelectModel<FishingZone> fishingZoneSelectModel;

    /** Current fishingZone id selected **/
    @Property
    private String fishingZoneId;

    /** Current fishingZone for loop **/
    @Property
    private FishingZone fishingZone;

    /** Current index in the loop **/
    @Property
    private int indexFishingZone;

    /** Field fishingZone for validation : must have at least one fishingZone **/
    @InjectComponent
    private Field fishingZoneSelect;

    /**
     * Get the select model for FishingZone. The select contains all existing fishingZones.
     *
     * @return a GenericSelectModel<FishingZone>
     * @throws WaoException
     */
    public GenericSelectModel<FishingZone> getFishingZoneSelectModel() throws WaoException {
        if (fishingZoneSelectModel == null) {
            List<FishingZone> zones = serviceReferential.getFishingZones();
            fishingZoneSelectModel = new GenericSelectModel<FishingZone>(zones, FishingZone.class,
                    "code", "topiaId", propertyAccess);
        }
        return fishingZoneSelectModel;
    }

    /**
     * Get the list of FishingZones from the SampleRow.
     *
     * @return a List of FishingZone linked with the SampleRow
     * @throws WaoException
     */
    public List<FishingZone> getFishingZones() throws WaoException {
        return getSampleRow().getFishingZone();
    }

    /**
     * Action on submit : addFishingZone. Used to add the selected fishingZone to the SampleRow.
     *
     * @throws WaoException
     */
    void onSelectedFromAddFishingZone() throws WaoException {
        fishingZone = getFishingZoneSelectModel().findObject(fishingZoneId);
        if (!getFishingZones().contains(fishingZone)) {
            if (log.isDebugEnabled()) {
                log.debug("Add new fishingZone");
            }
            getFishingZones().add(fishingZone);
        }
        edited = true;
    }

    /**
     * Action on submit : removeFishingZone. Used to remove the fishingZone from the SampleRow.
     *
     * @param index of the FishingZone in the list (from loop component)
     * @throws WaoException
     */
    void onSelectedFromRemoveFishingZone(int index) throws WaoException {
        if (log.isDebugEnabled()) {
            log.debug("Remove fishingZone at position : " + index);
        }
        // getFishingZones().remove(index); throw a ConcurrentModificationException,
        // a copy to prevent it :
        List<FishingZone> fishingZones = new ArrayList<FishingZone>(getSampleRow().getFishingZone());
        fishingZones.remove(index);
        getSampleRow().setFishingZone(fishingZones);
        edited = true;
    }

    /*************************** TERRESTRIAL LOCATION *************************/

    private GenericSelectModel<TerrestrialLocation> terrestrialLocationSelectModel;

    @Property
    private String terrestrialLocationId;

    public GenericSelectModel<TerrestrialLocation> getTerrestrialLocationSelectModel() {
        if (terrestrialLocationSelectModel == null) {
            List<TerrestrialLocation> locations = serviceReferential.getAllTerrestrialDistricts(new SamplingFilterImpl());
            terrestrialLocationSelectModel = new GenericSelectModel<TerrestrialLocation>(locations, TerrestrialLocation.class,
                        TerrestrialLocation.PROPERTY_DISTRICT_NAME, TerrestrialLocation.TOPIA_ID, propertyAccess);
        }
        return terrestrialLocationSelectModel;
    }

    /*************************** PROGRAM & MONTHS *****************************/

    @Inject
    private WaoManager manager;

    @Persist
    private SelectModel programSelectModel;

    @Property
    private String programName;

    @Property
    private String programId;

//    @Property
//    private Date month;

    private PeriodDates period;

//    @Persist
//    private List<Date> months;

    @Persist
    private List<SampleMonth> sampleMonths;

    @Property
    private int monthIndex;

    @Property
    private SampleMonth sampleMonth;

    @InjectComponent
    private Field programPeriodBegin;

    @InjectComponent
    private Field programPeriodEnd;

    @InjectComponent
    private Field program;

    @InjectComponent
    private Field code;

    private boolean periodChanged;

    public SelectModel getProgramSelectModel() throws WaoException {
        if (programSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String name : serviceSampling.getPrograms(null)) {
                options.add(new OptionModelImpl(name, name));
            }
            programSelectModel = new SelectModelImpl(null, options);
        }
        return programSelectModel;
    }

//    public List<Date> getMonths() throws WaoException {
//        if (months == null) {
//            if (log.isDebugEnabled()) {
//                log.debug("Reset months list");
//            }
//            months = getPeriod().getMonths();
//            sampleMonths = null;
//            getSampleMonths();
//        }
//        return months;
//    }

//    public Program getProgram() throws WaoException {
//        return getSampleRow().getProgram();
//    }

    public PeriodDates getPeriod() throws WaoException {
        if (period == null) {
            Date begin = getSampleRow().getPeriodBegin();
            Date end = getSampleRow().getPeriodEnd();
            period = new PeriodDates(begin, end);
        }
        return period;
    }

    public List<SampleMonth> getSampleMonths() throws WaoException {
        if (sampleMonths == null) {
            sampleMonths = new ArrayList<SampleMonth>();
            for (Date currMonth : getPeriod().getMonths()) {
                SampleMonth curr = getSampleRow().getSampleMonth(currMonth);
                if (curr != null) {
                    sampleMonths.add(curr);
                // If an other period is set
                } else if (periodChanged) {
                    if (curr == null) {
                        curr = serviceSampling.getNewSampleMonth(currMonth,
                                                        getSampleRow());
                        curr.addPropertyChangeListener(
                            SampleMonth.PROPERTY_EXPECTED_TIDES_VALUE, propertyChange);
                    }
                    sampleMonths.add(curr);
                }
            }
        }
        return sampleMonths;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    public String getMonthStyle() {
        String style = " ";
        
        Date current = manager.getCurrentDate();
        if (sampleMonth.isCurrentMonth()) {
            style += "selected";
        } else if (current.after(sampleMonth.getPeriodDate())) {
            style += "even";
        }
        return style;
    }

    /**
     * Search a program from the list.
     * The program will be set from the one selected. If no selection, a new one will be set.
     * @throws WaoException
     */
//    @Log
//    void onSelectedFromSearchProgram() throws WaoException {
//        if (!StringUtils.isEmpty(programId)) {
//            programName = null;
////            period = null;
////            months = null;
//            getSampleRow().setProgramName(programId);
//
//            // No period in this case ?!?
//        }
//        edited = true;
//    }

    /**
     * Refresh months from program period.
     * If the programName is not empty, a new program will be set.
     * The period dates will be saved in program (validation will be done in onSuccess).
     * @throws WaoException
     */
    @Log
    void onSelectedFromRefreshMonths() throws WaoException {
        getSampleRow().setPeriod(period);
        periodChanged = true;
        // Reset sampleMonths for new period
        sampleMonths = null;
        getSampleMonths();
        saveProgram();
    }

    protected void saveProgram() throws WaoException {
        // Save programName
        if (StringUtils.isNotEmpty(programName)) {
            getSampleRow().setProgramName(programName);
            programId = null;
        } else if (StringUtils.isNotEmpty(programId)) {
            getSampleRow().setProgramName(programId);
        }
        if (log.isInfoEnabled()) {
            log.info("Program : " + getSampleRow().getProgramName());
        }
    }

    /************** OBSDEB ***********/

    /** get the format of date needed to fill the value of SampleRow#expectedDate */
    public String getExpectedDateFormat() {
        if (getSampleRow().getObservationType() == null) {
            throw new NullPointerException();
        }
        String expectedDateFormat;
        if (getSampleRow().isPhoneCall()) {
            expectedDateFormat = "'semaine' w, yyyy";
        } else {
            expectedDateFormat = "dd/MM/yyyy";
        }
        return expectedDateFormat;
    }

    public List<WaoUser> getObservers() {
        List<WaoUser> observers = serviceUser.getObservers(getSampleRow().getCompany(), true);
        return observers;
    }

    public List<TerrestrialDivision> getObservationUnits() {
        List<TerrestrialDivision> observationUnits = serviceReferential.getAllObservationUnits();
        return observationUnits;
    }

    public List<Boat> getBoats() {
        List<Boat> boats = serviceBoat.getAllActiveBoats();
        return boats;
    }

    public List<TerrestrialLocation> getRegionIfremers() {
        List<TerrestrialLocation> regionIfremers =
                serviceReferential.getAllRegionIfremers();
        return regionIfremers;
    }
    /*************************** SAMPLE ROW ***********************************/

    private String sampleRowId;

    @Persist
    private SampleRow sampleRow;

    private BeanModel<SampleRow> sampleRowModel;

    @InjectComponent
    private Form sampleRowForm;

    @InjectComponent
    private Field updateComment;

    @Persist
    private SampleRowLog sampleRowLog;

    @Property
    private String immatriculations;

    @Property
    private String companyId;

    private List<Boat> boats;

    private GenericSelectModel<Company> companySelectModel;

    private boolean cancel;
    
    @Persist
    private WaoPropertyChangeListener propertyChange;

    public SampleRow getSampleRow() throws WaoException {
        if (sampleRow == null) {
            if (StringUtils.isEmpty(sampleRowId)) {
                sampleRow = serviceSampling.newSampleRow(user);
                if (sampleRow.getObsProgram() == ObsProgram.OBSDEB) {
                    sampleRow.setObservationType(waoActivationContext.getObservationType());
                }
            } else {
                sampleRow = serviceSampling.getSampleRow(sampleRowId);
                for (SampleMonth month : sampleRow.getSampleMonth()) {
                    month.addPropertyChangeListener(
                            SampleMonth.PROPERTY_EXPECTED_TIDES_VALUE, propertyChange);
                }
            }
            propertyChange = new WaoPropertyChangeListener();
        }
        return sampleRow;
    }

    public SampleRowLog getSampleRowLog() throws WaoException {
        if (sampleRowLog == null) {
            sampleRowLog = serviceSampling.getNewSampleRowLog(getSampleRow(), user);
        }
        return sampleRowLog;
    }

    public BeanModel<SampleRow> getSampleRowModel() {
        if (sampleRowModel == null) {
            sampleRowModel = beanModelSource.createEditModel(SampleRow.class, resources.getMessages());
            if (user.isObsDeb()) {
                if (getSampleRow().isFieldWorkObservation()) {
                    sampleRowModel.include(SampleRow.PROPERTY_COMMENT,
                                           SampleRow.PROPERTY_BLANK_ROW);
                } else {
                    sampleRowModel.include(SampleRow.PROPERTY_COMMENT);
                }
            } else {
                sampleRowModel.add(SampleRow.PROPERTY_COMPANY, null);
                sampleRowModel.add("boats", null);
                if (user.getProfile().getObsProgram() == ObsProgram.OBSMER) {
                    sampleRowModel.include(SampleRow.PROPERTY_CODE, SampleRow.PROPERTY_COMPANY,
                            SampleRow.PROPERTY_NB_OBSERVANTS, SampleRow.PROPERTY_AVERAGE_TIDE_TIME, "boats",
                            SampleRow.PROPERTY_COMMENT, SampleRow.PROPERTY_APPLIED_COVERAGE_RATE);
                }
                if (user.getProfile().getObsProgram() == ObsProgram.OBSVENTE) {
                    sampleRowModel.include(SampleRow.PROPERTY_CODE, SampleRow.PROPERTY_COMPANY, "boats",
                            SampleRow.PROPERTY_COMMENT);
                    sampleRowModel.add("samplingStrategy");
                }
            }
        }
        return sampleRowModel;
    }

    public GenericSelectModel<Company> getCompanySelectModel() throws WaoException {
        if (companySelectModel == null) {
            List<Company> companies = serviceUser.getCompanies(true);
            companySelectModel = new GenericSelectModel<Company>(companies, Company.class,
                    "name", "topiaId", propertyAccess);
        }
        return companySelectModel;
    }

    public List<Company> getCompanies() throws WaoException {
        List<Company> companies = serviceUser.getCompanies(true);
        return companies;
    }

    public boolean isCreateMode() throws WaoException {
        return StringUtils.isEmpty(getSampleRow().getTopiaId());
    }

    void onSelectedFromCancelSave() {
        cancel = true;
    }

    void onSelectedFromSaveData() throws WaoException {
        saveProgram();
    }

    @Log
    void onValidateFormFromSampleRowForm() throws WaoException {
        sampleRowForm.clearErrors();

        if (ObsProgram.OBSDEB == getSampleRow().getObsProgram()) {

            ValidationResult validationResult = serviceSampling.validateSampleRow(getSampleRow());

            boolean commentLogIsExpected = ! isCreateMode();
            boolean commentLogIsEmpty =
                          StringUtils.isBlank(getSampleRowLog().getComment());
            if (commentLogIsExpected && commentLogIsEmpty) {
                sampleRowForm.recordError(messages.get(n_("wao.ui.form.SampleRow.error.missingLogComment")));
            }

            if ( ! validationResult.isSuccess()) {
                sampleRowForm.recordError(validationResult.getMessage());
            }

        } else {

            // TODO 20110406 bleny move validation of ObsMer/ObsVente in ServiceSampling#validateSampleRow()

            if (StringUtils.isNotEmpty(terrestrialLocationId)) {
                getSampleRow().setTerrestrialLocation(
                        getTerrestrialLocationSelectModel().findObject(terrestrialLocationId));
            }

            // remove sampleMonths with "-1" as value, it should be removed
            // to mean that no observation should be done this month
            List<SampleMonth> sampleMonths = new ArrayList<SampleMonth>();
            for (SampleMonth sampleMonth : getSampleMonths()) {
                if (sampleMonth.getExpectedTidesValue() != -1) {
                    sampleMonths.add(sampleMonth);
                }
            }
            getSampleRow().clearSampleMonth();
            getSampleRow().addAllSampleMonth(sampleMonths);

            try {
                // Only for save or periodChanged (refreshMonth)
                if (!cancel && !edited) {
                    if (StringUtils.isEmpty(getSampleRow().getProgramName())) {
                        sampleRowForm.recordError(program,
                                "Vous devez choisir un programme existant ou en créer un nouveau");
                    }

                    // validate DCF5 codes
                    getSampleRow().setDCF5Code(getDcf5codes(), ", ");
                    if (CollectionUtils.isEmpty(getSampleRow().getdCF5Code())) {
                        sampleRowForm.recordError("Vous devez associer au moins un code DCF5");
                    } else {
                        // check that codes exist in the reference
                        SamplingFilter noFilter = new SamplingFilterImpl();
                        List<String> existingFishingGearCodes = new LinkedList<String>();
                        // filling existingFishingGearCodes
                        for (FishingGearDCF dcfGear : serviceSampling.getDCFGears(noFilter)) {
                            existingFishingGearCodes.add(dcfGear.getCode());
                        }
                        List<String> existingTargetSpeciesCodes = new LinkedList<String>();
                        for (TargetSpeciesDCF dcfSpecies : serviceSampling.getDCFSpecies(noFilter)) {
                            existingTargetSpeciesCodes.add(dcfSpecies.getCode());
                        }


                        for (DCF5Code code : getSampleRow().getdCF5Code()) {
                            if ( ! existingFishingGearCodes.contains(code.getFishingGearCode())) {
                                sampleRowForm.recordError(code.getFishingGearCode() + " n'est pas un code d'engin valide");
                            }

                            if (StringUtils.isEmpty(code.getTargetSpeciesCode())) {
                                if (StringUtils.isEmpty(getSampleRow().getProfession().getSpecies())) {
                                    sampleRowForm.recordError("Le code DCF d'espèces cibles peut ne pas être renseigné, mais dans ce cas il faut préciser une description des espèces cibles");
                                }
                            } else {
                                if ( ! existingTargetSpeciesCodes.contains(code.getTargetSpeciesCode())) {
                                    sampleRowForm.recordError(code.getTargetSpeciesCode() + " n'est pas un code d'espèces cibles valide");
                                }
                            }
                        }
                    }

                    Date periodBegin = getSampleRow().getPeriodBegin();
                    Date periodEnd = getSampleRow().getPeriodEnd();

                    // Validation for period dates
                    if (periodBegin == null) {
                        sampleRowForm.recordError(programPeriodBegin,
                                "Vous ne pouvez pas rafraîchir les mois avec une date de début de programme vide");
                    }
                    if (periodEnd == null) {
                        sampleRowForm.recordError(programPeriodEnd,
                                "Vous ne pouvez pas rafraîchir les mois avec une date de fin de programme vide");
                    }

                    if (periodBegin != null && periodEnd != null) {

                        SampleMonth firstMonth = getSampleMonths().get(0);

                        if (firstMonth != null) {
                            Date firstDate = firstMonth.getPeriodDate();

                            if (periodBegin.after(firstDate) && firstMonth.getRealTidesValue() != 0) {
                                sampleRowForm.recordError(programPeriodBegin,
                                    "Vous ne pouvez pas réduire la période du programme. " +
                                    "Il existe des enregistrements de marées réels.");
                            }
                        }

                        for (SampleMonth month : getSampleMonths()) {
                            if (log.isTraceEnabled()) {
                                log.trace("month : " + month.formatMonth() + " _ " +
                                        propertyChange.hasChanged(month.getTopiaId()));
                            }

                            if (propertyChange.hasChanged(month.getTopiaId()) &&
                                    month.getExpectedTidesValue() < month.getRealTidesValue()) {
                                sampleRowForm.recordError(
                                    "Il n'est pas possible de mettre moins de marées (" + month.getExpectedTidesValue() + ") " +
                                    "pour le mois " + getDateFormat().format(month.getPeriodDate()) +
                                    " qu'il y en a de réels (" + month.getRealTidesValue() + ")");
                            }
                        }
                    }

                    // Validate profession, fishingZone and boats (from immatriculations) if no programChanged
                    if (!periodChanged) {

                        if (!isCreateMode() && StringUtils.isEmpty(sampleRowLog.getComment())) {
                            sampleRowForm.recordError(updateComment,
                                    "Un commentaire est obligatoire lors d'une modification de la ligne");
                        }

                        if (CollectionUtils.isEmpty(getSampleRow().getdCF5Code())) {
                            sampleRowForm.recordError(
                                    "Le métier doit obligatoirement avoir un Code DCF niveau 5");
                        }
                        int nbZones = getSampleRow().sizeFishingZone();
                        if (nbZones == 0) {
                            sampleRowForm.recordError(fishingZoneSelect,
                                    "Vous devez ajouter au moins une zone de pêche à la ligne");
                        }

                        boats = serviceBoat.getBoatsByImmatriculations(immatriculations);
                    // Update sampleRowCode from program only if changed
                    } else if (isCreateMode()) {
                        if ( ! WaoUtils.getSampleRowCodePattern(getSampleRow().getObsProgram()).matcher(getSampleRow().getCode()).matches()) {
                            sampleRowForm.recordError(code, String.format(
                                    messages.get("wao.import.sampleRow.failure.wrongSampleRowCodeFormat"),
                                    getSampleRow().getCode()));
                        }
                    }

                    if (sampleRow.getSamplingStrategy() == SamplingStrategy.SPECIFIC_STOCK) {
                        if (StringUtils.isEmpty(getSampleRow().getProfession().getSpecies())) {
                            sampleRowForm.recordError("Pour ce choix de stratégie d'échantillonnage, il faut préciser les espèces cibles");
                        }
                    }
                }
            } catch (WaoBusinessException eee) {
                sampleRowForm.recordError(eee.getMessage());
                log.error("BUSINESS ERROR : " + eee.getType() + " depuis " + eee.getServiceName() +
                        " : " + eee.getMessage());
            }

            if (log.isDebugEnabled()) {
                log.debug("SampleMonths : " + getSampleMonths());
            }
        }
    }

    @InjectPage
    private SamplingPlan samplingPlan;

    @InjectPage
    private ObsDebSamplingPlan obsDebSamplingPlan;

    @Log
    Object onSuccessFromSampleRowForm() throws WaoException {
        if (cancel) {
            return this;
        }
        if (!edited && !periodChanged) {

            if (user.isAdmin() && ! user.isObsDeb()) { // in obsDeb, coordinator may not change company
                // No company change possible when sampleRow has real tide time
                if (!getSampleRow().hasSampleMonthRealTideTime()) {
                    // Save company
                    Company company = getCompanySelectModel().findObject(companyId);
                    getSampleRow().setCompany(company);
                }
            }

            if (getSampleRow().getObsProgram() != ObsProgram.OBSDEB) {

                // Save sampleMonths
                getSampleRow().setSampleMonth(getSampleMonths());

                // save dcf5 codes
                getSampleRow().setDCF5Code(getDcf5codes(), ", ");

            }

            serviceSampling.createUpdateSampleRow(sampleRow, boats, getSampleRowLog(), user);

            if (user.isObsDeb()) {
                obsDebSamplingPlan.setObservationType(sampleRow.getObservationType());
                return obsDebSamplingPlan;
            } else {
                samplingPlan.setSelectedRowId(sampleRow.getTopiaId());
                return samplingPlan;
            }
        }  else {
            // Reinitialize professionId for select to avoid selection from existing Profession
            professionId = null;
        }
        return sampleRowForm;
    }

    @Log
    Object onFailureFromSampleRowForm() {
        return sampleRowForm;
    }

        
}
