/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.components;

import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.SamplingFilter;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.nuiton.util.PeriodDates;
import org.slf4j.Logger;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Factorisation des comportements entre les différents composants filtres.
 *
 * Notamment, le triplet de boutons reset/search/refesh
 *
 * @author sletellier <letellier@codelutin.com>
 */
public abstract class FilterComponent<E extends SamplingFilter> {

    @Inject
    private Logger log;

    @Inject
    private ComponentResources resources;

    @Parameter(required = true)
    private ConnectedUser user;

    @Persist
    private E filter;

    public ConnectedUser getUser() {
        return user;
    }

    public E getFilter() {
        if (filter == null) {
            resetFilter();
        }
        return filter;
    }

    public void setFilter(E filter) {
        this.filter = filter;
    }

    /**
     * Method to initialize filter, is call on constructor
     */
    public abstract void resetFilter();

    /**
     * Used to show only active company and observers.
     *
     * @return true if only active users will be showed, false otherwise
     */
    protected abstract boolean isAvailableDataForFiltersOnly();

    /**
     * Used to update search fields
     */
    public abstract void updateSearchFields();

    @Parameter(required = false)
    private Boolean visibleOnStartup;

    @InjectComponent
    private Zone filtersZone;

    @Persist
    private Boolean filtersVisible;

    public ComponentResources getResources() {
        return resources;
    }

    // True by default
    public Boolean getVisibleOnStartup() {
        if (visibleOnStartup == null) {
            visibleOnStartup = true;
        }
        return visibleOnStartup;
    }

    public Boolean getFiltersVisible() {
        if (filtersVisible == null) {
            filtersVisible = getVisibleOnStartup();
        }
        return filtersVisible;
    }

    public void setFiltersVisible(Boolean filtersVisible) {
        this.filtersVisible = filtersVisible;
    }

    public void switchFiltersVisible() {
        filtersVisible = ! getFiltersVisible();
    }

    public Zone getFiltersZone() {
        return filtersZone;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    /**
     * Activate reset filter
     */
    private boolean reset;
    private boolean refresh;

    public boolean isReset() {
        return reset;
    }

    public boolean isRefresh() {
        return refresh;
    }

    public void onSelectedFromReset() {
        reset = true;
    }

    public void onSelectedFromRefresh() {
        refresh = true;
    }

    public Object onSuccessFromFiltersForm() {
        if (isReset()) {
            resetFilter();
        }

        // if user entered a non-chronological time interval, invert it
        PeriodDates period = getFilter().getPeriod();
        if (period != null) {
            Date fromDate = period.getFromDate();
            Date thruDate = period.getThruDate();
            if (thruDate.before(fromDate)) {
                period.setFromDate(thruDate);
                period.setThruDate(fromDate);
            }

            // Evo#344 limitation de la vue aux années n-1, n et n+1 pour le
            // rôle professionnel
            if (user.isProfessional()) {
                Date currentYear = DateUtils.truncate(new Date(), Calendar.YEAR);
                Date minBound = DateUtils.addYears(currentYear, -1);
                Date maxBound = DateUtils.addYears(currentYear, 2);
                maxBound = DateUtils.addMonths(maxBound, -1);
                if (period.getFromDate() != null) {
                    if (period.getFromDate().before(minBound)) {
                        period.setFromDate(minBound);
                    }
                    if (period.getFromDate().after(maxBound)) {
                        period.setThruDate(maxBound);
                    }
                }
                if (period.getThruDate() != null) {
                    if (period.getThruDate().before(minBound)) {
                        period.setThruDate(minBound);
                    }
                    if (period.getThruDate().after(maxBound)) {
                        period.setThruDate(maxBound);
                    }
                }
            }
        }

        // EVO#556 on ne rafraichit pas les valeurs des champs lorsque l'utilisateur
        // lance la recherche, on ne le fait que s'il le demande explicitement
        // (utilisation des boutons reset ou refresh
        if (isReset() || isRefresh()) {
            updateSearchFields();
        }

        if (isRefresh()) {
            // Refresh search panel
            return this;
        } else {
            // Refresh owner
            return getResources().getContainer();
        }
    }
}
