/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: SamplingPlan.java 1425 2011-12-16 14:16:26Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.1/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/SamplingPlan.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.FacadeRow;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.entity.DCF5Code;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.SampleMonth;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.components.SamplingFilterComponent;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.data.WaoActivationContext;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.nuiton.util.PeriodDates;
import org.nuiton.util.StringUtil;
import org.nuiton.util.StringUtil.ToString;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * SampingPlan
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication(allowedPrograms = {ObsProgram.OBSMER, ObsProgram.OBSVENTE})
@Import(stylesheet = "context:css/sampling.css")
public class SamplingPlan {

    @Inject
    private Logger log;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private WaoManager manager;

    @Inject
    private Messages messages;

    @InjectComponent
    private SamplingFilterComponent filterComponent;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser user;

    private long nbTidesExpectedTime;

    private long nbTidesRealTime;

    private long totalTidesExpectedTime;

    private long totalTidesRealTime;

    public ServiceSampling getServiceSampling() {
        return serviceSampling;
    }

    /**
     * Page initialization
     */
    void setupRender() {
        // Initialize fullView depends on user admin role
        if (fullView == null) {
            fullView = user.isAdminOrProfessional();
        }
    }

    /**************************** EXPORT **************************************/

    StreamResponse onActionFromExportSamplingPlan() {

        if (log.isInfoEnabled()) {
            log.info("user " + user + " is asking export with " + getFilter());
        }

        return new ExportStreamResponse("wao-echantillonnage") {

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceSampling.exportSamplingPlanCsv(
                                                    user, getFilter());
                } catch (WaoException eee) {
                    throw new IOException(eee);
                }
                return result;
            }
        };
    }

    /**************************** FILTERS *************************************/

    public boolean isFiltersVisible() {
        return filterComponent.getFiltersVisible();
    }

    Object onActionFromShowFilters() {

        filterComponent.switchFiltersVisible();

        //return filterComponent.getFiltersZone();
        return filterComponent;
    }

    public PeriodDates getPeriod() {
        return getFilter().getPeriod();
    }

    public SamplingFilter getFilter() {
        return filterComponent.getFilter();
    }


    /**************************** MAIN ACTIONS ********************************/

    @Persist
    @Property
    private Boolean fullView;

    /**
     * ACTION:: Used to change the display mode of the samplingPlan table.
     * This change affect the loading of the css style over the main table.
     *
     * @see #getMainClass()
     */
    void onActionFromToggleDisplayMode() {
        fullView = !fullView;
    }

    void onActionFromChangeFilterEstimatedTides() {
        boolean oldValue = getFilter().getEstimatedTides();
        if (log.isDebugEnabled()) {
            log.debug("Change estimatedTides in filter to : " + !oldValue);
        }
        getFilter().setEstimatedTides(!oldValue);
    }

    /**************************** MAIN TABLE **********************************/

    /** ------------------------- DATA ------------------------------------- **/

    /**
     * Main data for samplingPlan : List of SampleRow ordered by FishingZone.
     */
//    @Persist
    private List<SampleRow> data;

    private List<Date> months;

    @Property
    private Date month;

    /* variable used in template */
    @Property
    private DCF5Code dcf5code;

    public String getDcf5CodeDescription() {
        String description = dcf5code.getFishingGearCode() + " - " + dcf5code.getFishingGearDescription();
        if (dcf5code.getTargetSpeciesCode() != null) {
            description += " ; " + dcf5code.getTargetSpeciesCode() + " - " + dcf5code.getTargetSpeciesDescription();
        }
        return description;
    }

    /**
     * Current SampleRow from loop
     */
    @Property
    private SampleRow row;

    @Property
    private String currentFacadeName;

    @Property
    private String currentSectorName;

    /**
     * Return List of SampleRow from suiviobsmer-business
     * @return List of SampleRow
     * @throws WaoException
     */
    public List<SampleRow> getData() throws WaoException {
        if (data == null) {
            FacadeRow facade =
                    serviceSampling.getSampleRowsOrderedByFishingZone(getFilter());
            data = facade.getValues();
        }
        return data;
    }

    public List<Date> getMonths() {
        if (months == null) {
            months = getPeriod().getMonths();
        }
        return months;
    }

    @Log
    public boolean isFacadeChanged() {
        boolean facadeChanged = false;
        if (!row.getFacade().equals(currentFacadeName)) {
            currentFacadeName = row.getFacade();
            facadeChanged = true;
        }

        if (facadeChanged) {
            // reset currentSectorName to force re-print even if
            // it's the same
            currentSectorName = null;
        }

        return facadeChanged;
    }

    @Log
    public boolean isSectorChanged() {
        boolean sectorChanged = false;
        if (!row.getSectors().equals(currentSectorName)) {
            currentSectorName = row.getSectors();
            sectorChanged = true;
        }
        return sectorChanged;
    }

    public String getCompanyName() {
        return row.getCompany() != null ? row.getCompany().getName() : "";
    }

    public boolean isCurrentMonth() {
        String currentStr = getDateFormat().format(new Date());
        String monthStr = getDateFormat().format(month);
        return currentStr.equals(monthStr);
    }

    public Integer getNbTidesExpected() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = sampleMonth.getExpectedTidesValue();
        }
        return result;
    }

    private static final ToString<FishingZone> FISHING_ZONE_TO_STRING_DISTRICT =
            new ToString<FishingZone>() {

        @Override
        public String toString(FishingZone o) {
            return o.getDistrictCode();
        }
    };

    public String getFishingZones() {
        return StringUtil.join(row.getFishingZone(),
                FISHING_ZONE_TO_STRING_DISTRICT, ", ", false);
//        String result = "";
//        for (FishingZone zone : row.getFishingZone()) {
//            result += zone.getDistrictCode() + ", ";
//        }
//        return result.substring(0, result.length()-2);

    }

    public Integer getNbTidesReal() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = getFilter().getEstimatedTides() ?
                sampleMonth.getEstimatedTidesValue() :
                sampleMonth.getRealTidesValue();
        }
        return result;
    }

    public boolean hasNbTidesReal() {
        Date current = new Date();
        boolean validMonth = month.before(current) || isCurrentMonth();
        return validMonth && getNbTidesReal() != null;
    }

    public boolean canDisplayTidesReal() {
        // Evo #2227 : Guest users can't have access to real tides values
        boolean result = hasNbTidesReal() && !user.isGuest();
        return result;
    }

    public String getTotalPercentage() {
        String result;
        if (row.getTotalTidesExpected() > 0) {
            double percentage = (double)row.getTotalTidesReal() / (double)row.getTotalTidesExpected();
            result = NumberFormat.getPercentInstance().format(percentage);
        } else {
            result = "- %";
        }
        return result;
    }

    public NumberFormat getNumberFormat() {
        return NumberFormat.getNumberInstance();
    }

    private Map<Date, Integer> totalExpectedForMonths;
    private Map<Date, Integer> totalRealForMonths;
    private Integer highTotalExpected;
    private Integer highTotalReal;

    /** Set totalExpectedForMonths, totalRealForMonths, highTotalExpected and
     * highTotalReal values.
     */
    protected void setTotalsForMonths() {

        // to prevent a side-effect, keep month value
        Date keptMonth = month;

        totalExpectedForMonths = new HashMap<Date, Integer>();
        totalRealForMonths = new HashMap<Date, Integer>();

        // high totals is the total... of the totals, it will appear
        // in the bottom right, as the last element of the "totals" line
        highTotalExpected = 0;
        highTotalReal = 0;

        for (Date month : getMonths()) {
            this.month = month;

            Integer totalExpected = null, totalReal = null;
            for (SampleRow row : getData()) {
                this.row = row;

                Integer expected = getNbTidesExpected(), real = getNbTidesReal();
                if (log.isTraceEnabled()) {
                    log.trace(String.format("for month %s and row %s : expected = %s, real = %s",
                                  getDateFormat().format(month), row.getCode(), expected, real));
                }

                if (expected != null) {
                    if (totalExpected == null) {
                        totalExpected = 0;
                    }
                    totalExpected += expected;
                }
                if (real != null && canDisplayTidesReal()) {
                    if (totalReal == null) {
                        totalReal = 0;
                    }
                    totalReal += real;
                }
            }

            totalExpectedForMonths.put(month, totalExpected);
            totalRealForMonths.put(month, totalReal);

            if (log.isTraceEnabled()) {
                log.trace(String.format("totals for month %s : expected = %s, actual = %s",
                              getDateFormat().format(month), totalExpected, totalReal));
            }

            if (totalExpected != null) {
                highTotalExpected += totalExpected;
            }
            if (totalReal != null) {
                highTotalReal += totalReal;
            }

        }
        
        if (log.isDebugEnabled()) {
            for (Map.Entry<Date, Integer> expected : totalExpectedForMonths.entrySet()) {
                log.debug( String.format("totals for month %s : expected = %s, actual = %s",
                        getDateFormat().format(expected.getKey()), expected.getValue(),
                        totalRealForMonths.get(expected.getKey())));
            }
        }

        // to prevent side-effect, restore month value
        month = keptMonth;
    }

    public Integer getTotalExpectedForMonth() {
        if (totalExpectedForMonths == null) {
            setTotalsForMonths();
        }
        Integer total = totalExpectedForMonths.get(month);
        if (log.isDebugEnabled()) {
            log.debug(String.format("total for month %s is %s", getDateFormat().format(month), total));
        }
        return total;
    }

    public Integer getTotalRealForMonth() {
        if (totalExpectedForMonths == null) {
            setTotalsForMonths();
        }
        return totalRealForMonths.get(month);
    }

    public String getRatioForMonth() {
        String ratio = "-";
        if (getTotalExpectedForMonth() == null || getTotalRealForMonth() == null) {
            // 
        } else if (getTotalExpectedForMonth() > 0 && getTotalRealForMonth() > 0) {
            double percent = ((double) getTotalRealForMonth() / getTotalExpectedForMonth());
            ratio = NumberFormat.getPercentInstance().format(percent);
        }
        return ratio;
    }

    public Integer getHighTotalExpected() {
        if (highTotalExpected == null) {
            setTotalsForMonths();
        }
        return highTotalExpected;
    }

    public Integer getHighTotalReal() {
        if (highTotalReal == null) {
            setTotalsForMonths();
        }
        return highTotalReal;
    }

    public String getHighTotalRatio() {
        String ratio = "";
        if (getHighTotalExpected() > 0) {
            double percent = ((double) getHighTotalReal() / getHighTotalExpected());
            ratio = NumberFormat.getPercentInstance().format(percent);
        }
        return ratio;
    }

    /** ------------------------- HTML & STYLE ----------------------------- **/

    @Property
    private int rowIndex;

    public String getRowComment() {
        String rowComment = null;
        if (row.getComment() != null) {
            rowComment = row.getComment();
            // Problem with " in chenilleKit ToolTip component
            rowComment = rowComment.replaceAll("\n", "<br />");
        }
        return rowComment;
    }

    public int getNbColumnsForProfession() {
        // Code ligne |	Programme ou règlement rattachement	| Zone
        // | METIER Code | METIER Libellé
        int fixed = 5;
        if (fullView) {
            // company, program.periodBegin, program.periodEnd, fishingZonesInfos
            // profession.meshSize, profession.size, profession.other, profession.species
            fixed += 8;
        }
        if (user.isObsVente()) {
            fixed += 3; // add sampling strategy, terrestrial location, terrestrial location infos
        }
        return fixed;
    }

    public int getNbColumnsForMonths() {
        return getMonths().size() + 1;
    }

    public int getNbColumnsForOther() {
        int nbColumnsForOther = 2; // comment, actions
        if (user.isObsMer()) {
            nbColumnsForOther += 2; // nbObservants, averageTideTime
        }
        return nbColumnsForOther;
    }

    public int getNbColumnsTotal() {
        return getNbColumnsForProfession() +
                getNbColumnsForMonths() + getNbColumnsForOther();
    }

    public String getMainClass() {
        return fullView ? "admin" : "user";
    }

    public String getActionsClass() {
        return fullView ? "width100" : "width50";
    }

    public String getRealTidesClass() {
        String result = "real-warning";
        if (getNbTidesReal() < getNbTidesExpected()) {
            result += "-inf";
        } else if (getNbTidesReal() > getNbTidesExpected()) {
            result += "-sup";
        }
        return result;
    }

    public String getRealTidesClassForTotal() {
        String result = "real-warning";
        if (getTotalRealForMonth() == null || getTotalExpectedForMonth() == null) {
            // 
        } else if (getTotalRealForMonth() < getTotalExpectedForMonth()) {
            result += "-inf";
        } else if (getTotalExpectedForMonth() > getTotalRealForMonth()) {
            result += "-sup";
        }
        return result;
    }

    public String getMonthCurrentClass() {
        return isCurrentMonth() ? "selected" : "";
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    @Persist
    private String selectedRowId;

    public void setSelectedRowId(String selectedRowId) {
        this.selectedRowId = selectedRowId;
    }

    public String getRowClasses() {
        String classes = rowIndex % 2 == 0 ? "even" : "odd";
        if (row.getTopiaId().equals(selectedRowId)) {
            classes += " selected";
        }
        return classes;
    }

    /** ------------------------- ACTIONS ---------------------------------- **/

    /**
     * Can edit the sampleRow. Only for admin with no readOnly rights.
     * 
     * @return true if the sampleRow can be edited.
     */
    public boolean canEditSampleRow() {
        return user.isAdmin() && !user.isReadOnly();
    }

    /**
     * Used to filter period using dates from the program selected in table.
     * @param rowIndex index of the row in the table
     * @throws WaoException for a data problem
     */
    void onActionFromFilterPeriodDates(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        getPeriod().setFromDate(row.getPeriodBegin());
        getPeriod().setThruDate(row.getPeriodEnd());
    }

    void onActionFromDeleteSampleRow(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        try {
            serviceSampling.deleteSampleRow(row);
        } catch (WaoBusinessException eee) {
            String error = manager.getErrorMessage(eee, messages, log);
            layout.addError(error);
        }
    }

    public boolean isRowNotFinished() {
        // Test if the row isFinished with a gap of 1 month (today - 1 month)
        return !row.isFinished(-1);
    }

    public boolean isRowModified() {
        boolean rowHasHistoric = CollectionUtils.isNotEmpty(row.getSampleRowLog());
        return rowHasHistoric;
    }

    public String[] getContextForEditingSampleRow() {
        WaoActivationContext waoActivationContext = WaoActivationContext.newEmptyContext();
        waoActivationContext.setSampleRowId(row.getTopiaId());
        return waoActivationContext.toStrings();
    }

    public String[] getContextForBoats() {
        WaoActivationContext contextForBoats =
                                    WaoActivationContext.newEmptyContext();
        contextForBoats.setSampleRowCode(row.getCode());
        return contextForBoats.toStrings();
    }

}
