/*
 * #%L
 * Wao :: Web Interface
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.data;

import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.UserFilter;
import fr.ifremer.wao.bean.WaoFilter;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceUser;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Created: 24 juil. 2010
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: FiltersModel.java 809 2010-11-26 16:00:26Z tchemit $
 */
public class FiltersModel<F extends WaoFilter> {

    private static final Logger logger = LoggerFactory.getLogger(FiltersModel.class);

    /** SERVICES */
    private PropertyAccess propertyAccess;

    private ServiceUser serviceUser;

    private ServiceSampling serviceSampling;

    /** MODELS */
    private GenericSelectModel<Company> companySelectModel;

    private SelectModel programSelectModel;

    private GenericSelectModel<SampleRow> sampleRowSelectModel;

    /** DATA */
    private F filter;

    private boolean companyActive;

    public FiltersModel(PropertyAccess propertyAccess) {
        this.propertyAccess = propertyAccess;
    }

    public FiltersModel(PropertyAccess propertyAccess, F filter) {
        this(propertyAccess);
        this.filter = filter;
    }

    /**
     * ServiceUser is used for filters : COMPANY.
     *
     * @param serviceUser ServiceUser to set
     */
    public void setServiceUser(ServiceUser serviceUser) {
        this.serviceUser = serviceUser;
    }

    /**
     * ServiceSampling is used for filters : PROGRAM_NAME, SAMPLE_ROW.
     *
     * @param serviceSampling ServiceSampling to set
     */
    public void setServiceSampling(ServiceSampling serviceSampling) {
        this.serviceSampling = serviceSampling;
    }

    public void setCompanyActive(boolean companyActive) {
        this.companyActive = companyActive;
    }

    public void update(F filter) {
        this.filter = filter;
    }

    /**
     * Retrieve Company SelectModel to used in page.
     *
     * @return the GenericSelectModel for Company
     * @throws IllegalStateException if ServiceUser is not defined
     * @see ServiceUser#getCompanies(boolean)
     */
    public GenericSelectModel<Company> getCompanySelectModel()
            throws IllegalStateException {

        if (serviceUser == null) {
            throw new IllegalStateException("ServiceUser not exists to retrieve companies");
        }
        
        if (companySelectModel == null) {
            List<Company> companies = serviceUser.getCompanies(companyActive);
            companySelectModel = new GenericSelectModel<Company>(
                    companies,
                    Company.class,
                    Company.PROPERTY_NAME,
                    Company.TOPIA_ID,
                    propertyAccess
            );
        }
        return companySelectModel;
    }

    /**
     * Set the company in filter from its {@code companyId}. The company
     * object will be retrieve from the Company SelectModel. If you want
     * to set directly the company, do it directly on the filter using
     * {@link #getFilter()} method.
     *
     * @param companyId Id of the company to set
     * @throws IllegalStateException if Company SelectModel is not defined
     */
    public void setCompanyId(String companyId) throws IllegalStateException {
        if (companySelectModel == null) {
            throw new IllegalStateException("Company selectModel not found !");
        }
        Company company = companySelectModel.findObject(companyId);
        getUserFilter().setCompany(company);
    }

    public String getCompanyId() {
        String companyId = null;
        if (getUserFilter().getCompany() != null) {
            companyId = getUserFilter().getCompany().getTopiaId();
        }
        return companyId;
    }

    /**
     * Update company with {@code companyId}. Will reset dependent selects
     * (PROGRAM_NAME, SAMPLE_ROW).
     *
     * @param companyId Company id to set
     */
    public void updateCompany(String companyId) {
        setCompanyId(companyId);
        
        // Reset filter dependencies on company, on next load company filtered will be used
        programSelectModel = null;
        sampleRowSelectModel = null;
    }

    /**
     * Get select model for sampleRow program. Depends on company set in
     * filter.
     *
     * @return the SelectModel for program names
     */
    public SelectModel getProgramSelectModel() {

        if (serviceSampling == null) {
            throw new IllegalStateException("ServiceSampling not exists to retrieve programs");
        }

        if (programSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String programName : serviceSampling.getPrograms(getUserFilter().getCompany())) {
                options.add(new OptionModelImpl(programName, programName));
            }
            programSelectModel = new SelectModelImpl(null, options);
        }
        return programSelectModel;
    }

    /**
     * Update program with {@code programName}. Will reset dependent selects
     * (SAMPLE_ROW).
     *
     * @param programName Name of the program to set
     */
    public void updateProgram(String programName) {
        getSamplingFilter().setProgramName(programName);

        // Reset filter dependencies on program
        sampleRowSelectModel = null;
    }

    /**
     * Get select model for sampleRows. Depends on filter (Sampling part only).
     * The sampleRow set from filter is temporarly extracted to avoid returning
     * only one element in the select.
     *
     * @return the GenericSelectModel<SampleRow> for sampleRows
     * @throws WaoException
     */
    public GenericSelectModel<SampleRow> getSampleRowSelectModel() {

        if (serviceSampling == null) {
            throw new IllegalStateException("ServiceSampling not exists to retrieve sampleRows");
        }

        if (sampleRowSelectModel == null) {
            // WARN : no use of sampleRow in the filter to retrieve sampleRows
            SampleRow rowFiltered = getSamplingFilter().getSampleRow();
            getSamplingFilter().setSampleRow(null);

            List<SampleRow> sampleRows = serviceSampling.getSampleRowsByFilter(getSamplingFilter());

            getSamplingFilter().setSampleRow(rowFiltered);

            sampleRowSelectModel = new GenericSelectModel<SampleRow>(
                    sampleRows,
                    SampleRow.class,
                    SampleRow.PROPERTY_CODE,
                    SampleRow.TOPIA_ID,
                    propertyAccess
            );
        }
        return sampleRowSelectModel;
    }

    public void setSampleRowId(String sampleRowId) {
        if (sampleRowSelectModel == null) {
            throw new IllegalStateException("SampleRow selectModel not found !");
        }
        SampleRow sampleRow = sampleRowSelectModel.findObject(sampleRowId);
        getSamplingFilter().setSampleRow(sampleRow);
    }

    public String getSampleRowId() {
        String sampleRowId = null;
        if (getSamplingFilter().getSampleRow() != null) {
            sampleRowId = getSamplingFilter().getSampleRow().getTopiaId();
        }
        return sampleRowId;
    }

    /**
     * Get the filter.
     *
     * @return the current filter associated to this model.
     */
    public F getFilter() {
        return filter;
    }

    protected UserFilter getUserFilter() {
        checkAssignableFrom(UserFilter.class);
        return (UserFilter)filter;
    } 

    protected SamplingFilter getSamplingFilter() {
        checkAssignableFrom(SamplingFilter.class);
        return (SamplingFilter)filter;
    }

    private void checkAssignableFrom(Class filterClass) {
        if (!filterClass.isAssignableFrom(filter.getClass())) {
            throw new ClassCastException("SamplingFilter not assignable from " +
                    filter.getClass().getName());
        }
    }
}
