/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Synthesis.java 550 2010-06-18 13:39:46Z fdesbois $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.5/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Synthesis.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.BoardingResult;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactAverageReactivity;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.ContactStateStatistics;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.SamplingFilterImpl;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.service.ServiceSynthesis;
import fr.ifremer.wao.ui.base.AbstractFilteredPage;
import fr.ifremer.wao.ui.data.ChartUtils;
import fr.ifremer.wao.ui.data.ChartUtils.ChartType;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.data.SynthesisId;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.jfree.chart.JFreeChart;
import org.nuiton.util.DateUtils;
import org.nuiton.util.PeriodDates;
import org.slf4j.Logger;

import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;

/**
 * Synthesis
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication({UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER})
@IncludeStylesheet("context:css/synthesis.css")
public class Synthesis extends AbstractFilteredPage {

    @Inject
    private WaoManager manager;
    
    @Inject
    private ServiceSynthesis serviceSynthesis;

    @Inject
    private Logger log;

    @SessionState
    @Property
    private ConnectedUser user;

    @Persist
    private SynthesisId activeSynthesis;

    @Inject
    private ComponentResources resources;

    @InjectComponent
    private Zone delegator;

    void setupRender() {
        // The company of connected user will be contributed to abstractFilteredPage
        initCompanyFilter();
    }

    public Block getActiveBlock() throws WaoException {
        if (activeSynthesis == null) {
            // Initialize default synthesis by Data Sampling
            activeSynthesis = SynthesisId.GRAPH_SAMPLING;
        }
        switch (activeSynthesis) {
            case GRAPH_SAMPLING : initSamplingBlock(); break;
            case GRAPH_BOARDING : initBoardingBlock();
        }
        return resources.getBlock(activeSynthesis.getBlockId());
    }

    public String getActionSynthesisId() {
        return SynthesisId.class.getName();
    }

    public String getActionChartType() {
        return ChartType.class.getName();
    }

    public NumberFormat getPercentFormat() {
        return NumberFormat.getPercentInstance();
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public NumberFormat getNumberFormat() {
        NumberFormat result = NumberFormat.getNumberInstance();
        result.setMaximumFractionDigits(1);
        return result;
    }

    @Log
    Object onAction(String actionType, String actionValue) {
        // Delegator action (menu)
        if (actionType.equals(SynthesisId.class.getName())) {
            activeSynthesis = SynthesisId.valueOf(actionValue);
            return delegator.getBody();
        // Change chart type for dataSampling
        } else if (actionType.equals(ChartType.class.getName())) {
            dataSamplingChartType = ChartType.valueOf(actionValue);            
        }
        return this;
    }

    /********************* DYNAMICAL GRAPH : DATASAMPLING *********************/

    @Persist
    private SamplingFilter dataSamplingFilter;
    
    @InjectComponent
    private Zone filtersZone;

    @Persist
    private ChartType dataSamplingChartType;

    private boolean reset;

    private void initSamplingBlock() throws WaoException {
        initSelectFilters(true, false, true);
    }

    @Override
    public SamplingFilter getFilter() throws WaoException {
        if (dataSamplingFilter == null) {
            dataSamplingFilter = new SamplingFilterImpl();
            if (!user.isAdmin()) {
                dataSamplingFilter.setCompany(user.getCompany());
            }            
        }
        return dataSamplingFilter;
    }

    public PeriodDates getPeriod() throws WaoException {
        if (getFilter().getPeriod() == null) {
            PeriodDates period = PeriodDates.createMonthsPeriodFromToday(-12);
            if (log.isDebugEnabled()) {
                log.debug("Period : " + period);
            }
            getFilter().setPeriod(period);
        }
        return getFilter().getPeriod();
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return false;
    }

    public JFreeChart getDataSamplingChart() throws WaoException {   
        List<SortedMap<Date, Integer>> res = serviceSynthesis.getDataSampling(getFilter());
        Map<String, Map<?, Integer>> data = new HashMap<String, Map<?, Integer>>();
        data.put("Planifié", res.get(0));
        String realText =
                getFilter().getEstimatedTides() ? "Estimé" : "Réalisé";
        data.put(realText, res.get(1));
        String title = manager.getDataSamplingTitle(getFilter());
        return ChartUtils.createCategoryChart(title, "Nb marées", "Mois", getDataSamplingChartType(), data);
    }

    public ChartType getDataSamplingChartType() {
        // Default type is BAR
        if (dataSamplingChartType == null) {
            dataSamplingChartType = ChartType.BAR;
        }
        if (log.isDebugEnabled()) {
            log.debug("ChartType : " + dataSamplingChartType);
        }
        return dataSamplingChartType;
    }

    void onSelectedFromReset() {
        reset = true;
    }

    Object onSuccessFromFiltersForm() {
        if (isEdited()) {
            return filtersZone.getBody();
        }
        if (reset) {
            dataSamplingFilter = null;
        }
        return this;
    }

    /********************* STATIC GRAPH : BOARDINGBOAT ************************/

    private BoardingResult boardingResult;

    @Persist
    private Company companyForBoarding;

    @Property
    private String companyIdForBoarding;

    private void initBoardingBlock() throws WaoException {
        resetCompanySelect();
        // Set the companyIdForBoarding is only necessary for an admin which
        // have access to a select with all active companies (from Filters :
        // getCompanySelectModel())
        if (companyForBoarding != null && user.isAdmin()) {
            companyIdForBoarding = companyForBoarding.getTopiaId();
            if (log.isDebugEnabled()) {
                log.debug("companyId init : " + companyIdForBoarding);
            }
        }
    }

    public JFreeChart getBoardingBoatsChart() throws WaoException {
        Map<String, Map<?, Integer>> data =
                new HashMap<String, Map<?, Integer>>();
        data.put("Navires", getBoardingResult().getData());

        String title = "Sollicitations des navires depuis le " +
                getDateFormat().format(getFromDate());

        if (companyForBoarding != null) {
            title += "\nSociété " + companyForBoarding.getName();
        }
        
        return ChartUtils.createCategoryChart(title, "Nb navires",
                "Nb embarquements", ChartType.BAR, data);
    }

    public BoardingResult getBoardingResult() throws WaoException {
        if (boardingResult == null) {
            boardingResult = serviceSynthesis.getBoardingBoats(
                                    getCompanyForBoarding(), getFromDate());
        }
        return boardingResult;
    }

    public Company getCompanyForBoarding() {
        // Initialize only for an observer user. For admin, the company can
        // be null to search boardings for all companies.
        if (companyForBoarding == null && !user.isAdmin()) {
            companyForBoarding = user.getCompany();
        }
        if (log.isDebugEnabled()) {
            log.debug("company : " + companyForBoarding);
        }
        return companyForBoarding;
    }

    public Date getFromDate() {
        return DateUtils.createDateAfterToday(0, -12, 0);
    }

    @Log
    void onSuccessFromFilterCompanyForBoarding() throws WaoException {
        companyForBoarding = getCompanySelectModel().findObject(companyIdForBoarding);        
    }

    /********************* INDICATOR : NONCOMPLIANCEBOARDING ******************/

    private Map<String, Double> nonComplianceBoarding;

    @Property
    private Map.Entry<String, Double> nonComplianceBoardingEntry;

    /**
     * Data for nonComplianceBoarding indicator. The key is the company 
     * considered and the value is the indicator value. The map will contains
     * only one row, if the current user is an observer.
     *
     * @return a Map with companies and there values for nonComplianceBoarding
     * @throws WaoException 
     */
    public Map<String, Double> getNonComplianceBoarding()
            throws WaoException {
        if (nonComplianceBoarding == null) {
            Company company = !user.isAdmin() ? user.getCompany() : null;
            nonComplianceBoarding = 
                    serviceSynthesis.getNonComplianceBoardingIndicator(company);
        }
        return nonComplianceBoarding;
    }

    /**
     * Return the value of nonComplianceBoarding indicator. Only useful for
     * observer user.
     *
     * @return the value of the indicator
     * @throws WaoException
     */
    public Double getNonComplianceBoardingValue() throws WaoException {
        return getNonComplianceBoarding().get(user.getCompany().getName());
    }

    /********************* INDICATOR : NONCOMPLIANCEBOARDING ******************/

    private Collection<ContactStateStatistics> contactStateStatistics;

    @Persist
    private PeriodDates periodForContactStates;

    @Property
    private ContactStateStatistics contactStateStats;

    @Property
    private ContactState contactState;

    /**
     * Get all existing contactStates.
     *
     * @return ContactState[]
     */
    public ContactState[] getContactStates() {
        return ContactState.values();
    }

    /**
     * Get all contactStateStatistics from serviceSynthesis. There will be
     * only one result if the current user is an observer.
     *
     * @return all contactStateStatistics depends on company and selected period
     * @throws WaoException
     */
    public Collection<ContactStateStatistics> getContactStateStatistics()
            throws WaoException {
        if (contactStateStatistics == null) {
            Company company = !user.isAdmin() ? user.getCompany() : null;
            contactStateStatistics = 
                    serviceSynthesis.getContactStateStatistics(company, getPeriodForContactStates());
        }
        return contactStateStatistics;
    }

    public PeriodDates getPeriodForContactStates() {
        if (periodForContactStates == null) {
            periodForContactStates = PeriodDates.createMonthsPeriodFromToday(-12);
        }
        return periodForContactStates;
    }

    /**
     * Get the value for current contactStateStats in table row (by company)
     * and current contactState in table column.
     *
     * @return the number of contact states for the current contactState
     */
    public int getNbContactStates() {
        return contactStateStats.getData().get(contactState);
    }

    /**
     * Get the percent value for the current contactStateStats in table row
     * (by company) and current contactState in table column.
     *
     * @return the percent value of contact states for the current contactState
     */
    public double getPercentContactStates() {
        if (log.isDebugEnabled())  {
            log.debug("nbContactStates : " + getNbContactStates());
            log.debug("total : " + contactStateStats.getTotal());
        }
        return (double)getNbContactStates() / (double)contactStateStats.getTotal();
    }

    /********************* INDICATOR : NONCOMPLIANCEBOARDING ******************/

    private Collection<ContactAverageReactivity> allegroReactivity;

    @Persist
    private PeriodDates periodForAllegroReactivity;

    @Property
    private ContactAverageReactivity allegroReactivityEntry;

    public Collection<ContactAverageReactivity> getAllegroReactivity()
            throws WaoException {
        if (allegroReactivity == null) {
            Company company = !user.isAdmin() ? user.getCompany() : null;
            allegroReactivity =
                    serviceSynthesis.getContactDataInputDateReactivity(company,
                            getPeriodForAllegroReactivity());
        }
        return allegroReactivity;
    }

    public PeriodDates getPeriodForAllegroReactivity() {
        if (periodForAllegroReactivity == null) {
            periodForAllegroReactivity = PeriodDates.createMonthsPeriodFromToday(-3);
        }
        return periodForAllegroReactivity;
    }

    public Double getAllegroReactivityValue() throws WaoException {
        if (!getAllegroReactivity().isEmpty()) {
            ContactAverageReactivity entry = (ContactAverageReactivity)
                    CollectionUtils.get(getAllegroReactivity(), 0);
            return entry.getResult();
        } else {
            return 0.;
        }
    }

    public String getAllegroReactivityStyle() throws WaoException {
        double value = user.isAdmin() ?
            allegroReactivityEntry.getResult() : getAllegroReactivityValue();
        if (value >= 15.) {
            return " warn";
        }
        return "";
    }

}
