package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.persistence.entities.referential.TaxonCache;
import fr.ifremer.tutti.persistence.entities.referential.TaxonCaches;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.FrequencyConfigurationMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.actions.ApplyBenthosFrequencyRafaleAction;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiNumberTickUnitSource;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.jdesktop.swingx.table.TableColumnExt;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberTickUnitSource;
import org.jfree.chart.axis.ValueAxis;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.awt.BorderLayout;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.t;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyUIHandler extends AbstractTuttiTableUIHandler<BenthosFrequencyRowModel, BenthosFrequencyUIModel, BenthosFrequencyUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosFrequencyUIHandler.class);

    private BenthosFrequencyCellComponent.FrequencyCellEditor frequencyEditor;

    private Map<String, Caracteristic> lengthStepCaracteristics;

    private JFreeChart chart;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected WeightUnit weightUnit;

    protected ApplyBenthosFrequencyRafaleAction applyBenthosFrequencyRafaleAction;

    private TaxonCache taxonCache;

    public BenthosFrequencyUIHandler() {
        super(BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
              BenthosFrequencyRowModel.PROPERTY_NUMBER,
              BenthosFrequencyRowModel.PROPERTY_WEIGHT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosFrequencyTableModel getTableModel() {
        return (BenthosFrequencyTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    public boolean isRowValid(BenthosFrequencyRowModel row) {

        BenthosFrequencyUIModel model = getModel();
        boolean valid = model.isRowValid(row);
        return valid;

    }

    @Override
    protected void onModelRowsChanged(List<BenthosFrequencyRowModel> rows) {

        BenthosFrequencyUIModel model = getModel();

        model.reloadRows();

        getTableModel().setRows(rows);

        // clean log table
        BenthosFrequencyLogsTableModel logsTableModel = (BenthosFrequencyLogsTableModel) ui.getLogsTable().getModel();
        logsTableModel.setRows(Lists.<BenthosFrequencyLogRowModel>newArrayList());

    }

    @Override
    protected void onRowModified(int rowIndex,
                                 BenthosFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        // We do nothing here. This API works only on the selected row.
        // On this screen, we can interacts with not selected row, so won't come here.
        // Better then to work directly on rows in the table model

    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<BenthosFrequencyRowModel> rowMonitor,
                                             BenthosFrequencyRowModel row) {
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          BenthosFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);
        ui.getValidator().doValidate();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<BenthosFrequencyUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(BenthosFrequencyUI ui) {

        this.applyBenthosFrequencyRafaleAction = new ApplyBenthosFrequencyRafaleAction(ui);

        super.beforeInit(ui);

        this.weightUnit = getConfig().getBenthosWeightUnit();
        SampleCategoryModel sampleCategoryModel = getDataContext().getSampleCategoryModel();

        BenthosFrequencyUIModel model = new BenthosFrequencyUIModel(weightUnit, sampleCategoryModel);

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(BenthosFrequencyUI ui) {

        initUI(ui);

        List<Caracteristic> lengthStepCaracterics =
                Lists.newArrayList(getDataContext().getLengthStepCaracteristics());

        lengthStepCaracteristics = TuttiEntities.splitById(lengthStepCaracterics);

        BenthosFrequencyUIModel model = getModel();

        taxonCache = TaxonCaches.createBenthosCacheWithoutVernacularCode(getPersistenceService(), getDataContext().getProtocol());

        Caracteristic modelCaracteristic = model.getLengthStepCaracteristic();
        initBeanFilterableComboBox(ui.getLengthStepCaracteristicComboBox(),
                                   lengthStepCaracterics,
                                   modelCaracteristic);

        // get step from the pmfm
        float step = getStep(modelCaracteristic);
        model.setStep(step);

        model.setMinStep(null);
        model.setMaxStep(null);

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) BenthosFrequencyUIHandler.this.ui.getRafaleStepField().getModel().getNumberValue();

                    applyBenthosFrequencyRafaleAction.applyRafaleStep(step, false);

                    //select text
                    JTextField field = (JTextField) e.getSource();
                    field.selectAll();
                }
            }
        });

        // when lengthStepCaracteristic changed, let's updates all row with the new value
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_LENGTH_STEP_CARACTERISTIC, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Caracteristic newValue = (Caracteristic) evt.getNewValue();
                // get step from the pmfm
                float step = getStep(newValue);
                getModel().setStep(step);
                getModel().setDataSetIntervalWidth(step);
                chart.getXYPlot().getDomainAxis().setStandardTickUnits(new TuttiNumberTickUnitSource(step == 1f));
                if (CollectionUtils.isNotEmpty(getModel().getRows())) {
                    for (BenthosFrequencyRowModel rowModel : getModel().getRows()) {
                        rowModel.setLengthStepCaracteristic(newValue);
                        recomputeRowValidState(rowModel);
                    }
                }
                BenthosFrequencyUIHandler.this.ui.getValidator().doValidate();
            }
        });

        // when lengthStepCaracteristicUnit changed, let's updates the label of some fields
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_LENGTH_STEP_CARACTERISTIC_UNIT, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                String unit = (String) evt.getNewValue();

                if (unit == null) {

                    unit = t("tutti.editBenthosFrequencies.unkownStepUnit");
                }

                getUI().getMinStepLabel().setText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.minStep"), unit));
                getUI().getMinStepLabel().setToolTipText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.minStep.tip"), unit));

                getUI().getMaxStepLabel().setText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.maxStep"), unit));
                getUI().getMaxStepLabel().setToolTipText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.maxStep.tip"), unit));

                getUI().getRafaleStepLabel().setText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.rafaleStep"), unit));
                getUI().getRafaleStepLabel().setToolTipText(getLabelWithUnit(t("tutti.editBenthosFrequencies.field.rafaleStep.tip"), unit));

                TableColumnExt column = (TableColumnExt) getUI().getTable().getColumn(BenthosFrequencyTableModel.LENGTH_STEP);
                column.setHeaderValue(getLabelWithUnit(t("tutti.editSpeciesFrequencies.table.header.lengthStep"), unit));
                column.setToolTipText(getLabelWithUnit(t("tutti.editSpeciesFrequencies.table.header.lengthStep"), unit));
            }
        });

        // when configuration mode change, let's focus the best component (see http://forge.codelutin.com/issues/4035)
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_CONFIGURATION_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                final FrequencyConfigurationMode newValue = (FrequencyConfigurationMode) evt.getNewValue();
                SwingUtilities.invokeLater(
                        new Runnable() {
                            @Override
                            public void run() {
                                JComponent componentToFocus = getComponentToFocus(newValue);
                                if (componentToFocus != null) {
                                    componentToFocus.grabFocus();
                                }
                                updateLogVisibility();
                            }
                        }
                );
            }
        });

        // init histogram

        chart = ChartFactory.createXYBarChart(null,
                                              t("tutti.editSpeciesFrequencies.table.header.lengthStep"),
                                              false,
                                              t("tutti.editSpeciesFrequencies.table.header.number"),
                                              model.dataset);
        chart.clearSubtitles();

        ValueAxis rangeAxis = chart.getXYPlot().getRangeAxis();
        rangeAxis.setAutoRange(true);
        rangeAxis.setStandardTickUnits(new NumberTickUnitSource(true));

        ValueAxis domainAxis = chart.getXYPlot().getDomainAxis();
        domainAxis.setAutoRange(true);
        domainAxis.setStandardTickUnits(new TuttiNumberTickUnitSource(true));
        domainAxis.setMinorTickMarksVisible(true);

        chart.getXYPlot().getRenderer().setSeriesPaint(0, getConfig().getColorComputedWeights());

        final ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setDomainZoomable(false);
        chartPanel.setMouseZoomable(false);
        chartPanel.setPopupMenu(null);

        JPanel histogramPanel = ui.getHistogramPanel();
        histogramPanel.add(chartPanel, BorderLayout.CENTER);

        // init data table
        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN,
                                  table);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    BenthosFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN,
                                    table);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.WEIGHT,
                                  weightUnit,
                                  table);
        }

        // create table model
        BenthosFrequencyTableModel tableModel =
                new BenthosFrequencyTableModel(weightUnit, columnModel, model);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        installTableKeyListener(columnModel, table);

        // init log table
        JXTable logTable = ui.getLogsTable();

        // create log table column model
        DefaultTableColumnModelExt logColumnModel = new DefaultTableColumnModelExt();

        { // Date
            addColumnToModel(logColumnModel,
                             BenthosFrequencyLogCellComponent.newEditor(ui),
                             BenthosFrequencyLogCellComponent.newRender(),
                             BenthosFrequencyLogsTableModel.LABEL);
        }

        // create log table model
        BenthosFrequencyLogsTableModel logTableModel = new BenthosFrequencyLogsTableModel(logColumnModel);
        logTableModel.setRows(new ArrayList<BenthosFrequencyLogRowModel>());

        logTable.setModel(logTableModel);
        logTable.setColumnModel(logColumnModel);

        // by default do not authorize to change column orders
        logTable.getTableHeader().setReorderingAllowed(false);
        Highlighter evenHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.ODD,
                getConfig().getColorAlternateRow());
        logTable.addHighlighter(evenHighlighter);

        listenValidatorValid(ui.getValidator(), model);

    }

    @Override
    protected JComponent getComponentToFocus() {
        FrequencyConfigurationMode configurationMode = getModel().getConfigurationMode();
        JComponent componentToFocus = getComponentToFocus(configurationMode);
        if (componentToFocus == null) {
            componentToFocus = getUI().getLengthStepCaracteristicComboBox();
        }
        return componentToFocus;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        frequencyEditor = null;

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        getModel().setSimpleCount(null);

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    public void editBatch(BenthosFrequencyCellComponent.FrequencyCellEditor editor) {

        BenthosBatchRowModel speciesBatch = editor.getEditRow();

        BenthosFrequencyUIModel model = getModel();
        model.setNextEditableRowIndex(editor.getNextEditableRowIndex());
        model.setTotalNumber(null);
        model.setTotalComputedWeight(null);
        model.setTotalWeight(null);
        model.setSimpleCount(null);
        model.setMinStep(null);
        model.setMaxStep(null);

        frequencyEditor = editor;

        Caracteristic lengthStepCaracteristic = null;
        Float lengthStep;

        List<BenthosFrequencyRowModel> rows = Lists.newArrayList();

        if (speciesBatch != null) {

            model.setTotalWeight(speciesBatch.getWeight());

            List<BenthosFrequencyRowModel> frequency = speciesBatch.getFrequency();

            // try to load existing frequency

            if (CollectionUtils.isNotEmpty(frequency)) {

                BenthosFrequencyTableModel tableModel = getTableModel();

                for (BenthosFrequencyRowModel rowModel : frequency) {

                    BenthosFrequencyRowModel newRow = tableModel.createNewRow(false);
                    newRow.setLengthStepCaracteristic(rowModel.getLengthStepCaracteristic());
                    newRow.setLengthStep(rowModel.getLengthStep());
                    newRow.setNumber(rowModel.getNumber());
                    newRow.setWeight(rowModel.getWeight());
                    rows.add(newRow);
                }

                // use first frequency row length step caracteristics

                BenthosFrequencyRowModel rowModel = frequency.get(0);
                lengthStepCaracteristic = rowModel.getLengthStepCaracteristic();
                lengthStep = rowModel.getLengthStep();

                if (log.isInfoEnabled()) {
                    log.info("Use existing lengthStep " +
                             "caracteristic / step " +
                             decorate(lengthStepCaracteristic) + " / " +
                             lengthStep);
                }
            }

            BenthosBatchRowModel previousSiblingRow = frequencyEditor.getPreviousSiblingRow();

            if (lengthStepCaracteristic == null && previousSiblingRow != null) {

                // try to get it from his previous brother row
                List<BenthosFrequencyRowModel> previousFrequency = previousSiblingRow.getFrequency();

                if (CollectionUtils.isNotEmpty(previousFrequency)) {

                    // use the first frequency length step caracteristic / step
                    BenthosFrequencyRowModel rowModel = previousFrequency.get(0);
                    lengthStepCaracteristic =
                            rowModel.getLengthStepCaracteristic();
                    lengthStep = rowModel.getLengthStep();
                    if (log.isInfoEnabled()) {
                        log.info("Use previous sibling existing lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }

            if (lengthStepCaracteristic == null) {

                Species species = speciesBatch.getSpecies();

                String lengthStepPmfmId = taxonCache.getLengthStepPmfmId(species);

                if (lengthStepPmfmId != null) {

                    lengthStepCaracteristic = lengthStepCaracteristics.get(lengthStepPmfmId);
                    lengthStep = taxonCache.getLengthStep(species);

                    if (log.isInfoEnabled()) {
                        log.info("Use existing from protocol lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will edit batch row: " + speciesBatch + " with " + rows.size() + " frequency");
        }

        FrequencyConfigurationMode mode = FrequencyConfigurationMode.AUTO_GEN;
        if (lengthStepCaracteristic == null) {
            String lengthStepPmfmId = taxonCache.getLengthStepPmfmId(speciesBatch.getSpecies());
            if (lengthStepPmfmId == null) {
                mode = FrequencyConfigurationMode.SIMPLE_COUNTING;
            }
        }
        Integer number = speciesBatch.getNumber();
        if (number != null && rows.isEmpty()) {
            mode = FrequencyConfigurationMode.SIMPLE_COUNTING;
            model.setSimpleCount(number);
        }
        // make sure configuration mode will be rebind
        model.setConfigurationMode(null);
        model.setConfigurationMode(mode);

        // connect model to validator
        ui.getValidator().setBean(model);

        // always sort row by their length
        // see http://forge.codelutin.com/issues/2482
        Collections.sort(rows);

        model.setLengthStepCaracteristic(lengthStepCaracteristic);
        model.setRows(rows);

        // keep batch (will be used to push back editing entry)
        model.setBatch(speciesBatch);

    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected JComponent getComponentToFocus(FrequencyConfigurationMode mode) {
        JComponent componentToFocus = null;
        if (mode != null) {
            boolean withLengthStepCaracteristic =
                    getModel().getLengthStepCaracteristic() != null;
            switch (mode) {
                case AUTO_GEN:
                    if (withLengthStepCaracteristic) {

                        componentToFocus = ui.getMinStepField();
                    } else {
                        componentToFocus = ui.getLengthStepCaracteristicComboBox();
                    }
                    break;
                case RAFALE:

                    if (withLengthStepCaracteristic) {

                        componentToFocus = ui.getRafaleStepField();
                    } else {
                        componentToFocus = ui.getLengthStepCaracteristicComboBox();
                    }

                    break;
                case SIMPLE_COUNTING:
                    componentToFocus = ui.getSimpleCountingField();
                    break;
                default:
                    componentToFocus = null;
            }
        }
        return componentToFocus;
    }

    protected float getStep(Caracteristic caracteristic) {
        Float precision = null;
        if (caracteristic != null) {
            precision = caracteristic.getPrecision();
        }
        if (precision == null) {
            precision = 1f;
        }
        return precision;
    }

    protected void updateLogVisibility() {

        boolean logVisible = getModel().isRafaleMode();
        BenthosFrequencyUI ui = getUI();
        JSplitPane firstSplitPane = ui.getFirstSplitPane();
        JSplitPane secondSplitPane = ui.getSecondSplitPane();

        int lastDividerLocation = secondSplitPane.getLastDividerLocation();
        if (lastDividerLocation == 0) {
            lastDividerLocation = 200;
        }
        secondSplitPane.setDividerLocation(logVisible ? lastDividerLocation : 0);
        secondSplitPane.setDividerSize(logVisible ? firstSplitPane.getDividerSize() : 0);

        ui.getLogsScrollPane().setVisible(logVisible);
    }

    protected String getLabelWithUnit(String label, String unit) {
        return label + " (" + unit + ")";
    }

    @Override
    public <E> void initBeanFilterableComboBox(BeanFilterableComboBox<E> comboBox, List<E> data, E selectedData) {
        super.initBeanFilterableComboBox(comboBox, data, selectedData);
    }

    public BenthosFrequencyCellComponent.FrequencyCellEditor getFrequencyEditor() {
        return frequencyEditor;
    }
}
