package fr.ifremer.tutti.ui.swing.content.operation.catches.actions;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatch;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.MarineLitterBatch;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.catches.TuttiWeightComputingException;
import fr.ifremer.tutti.service.catches.WeightCleaningService;
import fr.ifremer.tutti.service.catches.WeightComputingService;
import fr.ifremer.tutti.ui.swing.util.actions.LongActionSupport;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryAble;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryColumnIdentifier;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SpeciesBatchNaturalOrderComparator;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SpeciesSortMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchTableModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchTableModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.util.Numbers;
import jaxx.runtime.swing.JTables;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;

import javax.swing.JOptionPane;
import javax.swing.table.TableColumn;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputeBatchWeightsAction extends LongActionSupport<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    protected final WeightCleaningService cleaningService;

    protected List<SpeciesBatchRowModel> speciesBatchRows;

    protected List<BenthosBatchRowModel> benthosBatchRows;

    protected CatchBatch catchBatch;

    protected Boolean modified;

    protected Integer tabInError;

    public ComputeBatchWeightsAction(EditCatchesUIHandler handler) {
        super(handler, true);
        this.cleaningService = getContext().getWeightCleaningService();
    }

    @Override
    public boolean prepareAction() throws Exception {

        // Clean internal states

        speciesBatchRows = null;
        benthosBatchRows = null;
        catchBatch = null;
        modified = null;
        tabInError = null;

        boolean doAction = super.prepareAction();
        if (doAction) {

            // do a check of double weights
            FishingOperation fishingOperation = getModel().getFishingOperation();
            Multimap<String, String> errors = cleaningService.checkFishingOperation(fishingOperation.getId());

            if (errors.isEmpty()) {

                // no errors

                sendMessage(t("tutti.editCatchBatch.action.computeWeights.no.double.weight.detected"));
            } else {

                // show errors to user as warning

                String errorsStr = cleaningService.errorsToString(errors);
                String htmlMessage = t("tutti.editCatchBatch.action.computeWeights.double.weight.detected", errorsStr);

                JOptionPane.showMessageDialog(getContext().getActionUI(),
                                              htmlMessage,
                                              t("tutti.editCatchBatch.action.computeWeights.double.weight.detected.title"),
                                              JOptionPane.WARNING_MESSAGE);


            }
        }
        return doAction;
    }

    @Override
    public void doAction() throws Exception {
        PersistenceService persistenceService = getContext().getPersistenceService();

        WeightComputingService weightComputingService = getContext().getWeightComputingService();

        EditCatchesUIModel model = getModel();
        String operationId = model.getFishingOperation().getId();

        // ---------
        // Compute species batches
        // ---------

        Float totalSpeciesSortedWeight;
        BatchContainer<SpeciesBatch> computedSpeciesBatches;

        try {
            computedSpeciesBatches = weightComputingService.getComputedSpeciesBatches(operationId);
            totalSpeciesSortedWeight = computeSpeciesBatches(persistenceService, computedSpeciesBatches);

        } catch (TuttiWeightComputingException e) {

            tabInError = 1;
            throw e;

        }

        // ---------
        // Compute benthos batches
        // ---------

        Float totalBenthosSortedWeight;
        BatchContainer<BenthosBatch> computedBenthosBatches;

        try {
            computedBenthosBatches = weightComputingService.getComputedBenthosBatches(operationId);
            totalBenthosSortedWeight = computeBenthosBatches(persistenceService, computedBenthosBatches);

        } catch (TuttiWeightComputingException e) {

            tabInError = 2;
            throw e;

        }

        // ---------
        // Compute marine litter batches
        // ---------

        BatchContainer<MarineLitterBatch> computedMarineLitterBatches;
        try {
            computedMarineLitterBatches =
                    weightComputingService.getComputedMarineLitterBatches(operationId, model.getMarineLitterTotalWeight());

        } catch (TuttiWeightComputingException e) {

            tabInError = 3;
            throw e;

        }

        modified = model.isModify();
        catchBatch = model.toEntity();

        // ---------
        // Check species rates
        // ---------

        Float rate = getConfig().getDifferenceRateBetweenSortedAndTotalWeights();
        if (model.getSpeciesTotalSortedWeight() != null
            && model.getSpeciesTotalSortedWeight() >= totalSpeciesSortedWeight
            && model.getSpeciesTotalSortedWeight() < (1 + rate / 100) * totalSpeciesSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"

            getUI().getTabPane().setSelectedIndex(1);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message.species", rate),
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {

                catchBatch.setSpeciesTotalSortedWeight(null);
                catchBatch.setSpeciesTotalSortedComputedWeight(totalSpeciesSortedWeight);
                modified = true;

            }

        }

        // ---------
        // Check benthos rates
        // ---------

        if (model.getBenthosTotalSortedWeight() != null
            && model.getBenthosTotalSortedWeight() >= totalBenthosSortedWeight
            && model.getBenthosTotalSortedWeight() < (1 + rate / 100) * totalBenthosSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"

            getUI().getTabPane().setSelectedIndex(2);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message.benthos", rate),
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {

                catchBatch.setBenthosTotalSortedWeight(null);
                catchBatch.setBenthosTotalSortedComputedWeight(totalBenthosSortedWeight);
                modified = true;

            }
        }

        weightComputingService.computeCatchBatchWeights(catchBatch,
                                                        computedSpeciesBatches,
                                                        computedBenthosBatches,
                                                        computedMarineLitterBatches);

    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        afterAction();

        getUI().repaint();
        //TCHEMIT-2015-04-27 Plus besoin car le champs qui posait problème n'existe plus
//        // see http://forge.codelutin.com/issues/3853
//        SwingUtilities.invokeLater(
//                new Runnable() {
//                    @Override
//                    public void run() {
//                        getUI().getComputeSpeciesBatchButton().requestFocus();
//                    }
//                }
//        );
    }

    protected void afterAction() {

        EditCatchesUIModel editCatchesUIModel = getModel();

        editCatchesUIModel.setLoadingData(true);

        try {

            if (speciesBatchRows != null) {

                getUI().getSpeciesTabContent().getModel().setRows(speciesBatchRows);

            }

            if (benthosBatchRows != null) {

                getUI().getBenthosTabContent().getModel().setRows(benthosBatchRows);

            }

            if (catchBatch != null) {

                editCatchesUIModel.fromEntity(catchBatch);

            }

            if (modified != null) {

                editCatchesUIModel.setModify(modified);

            }

        } finally {

            editCatchesUIModel.setLoadingData(false);

        }

        // keep sortMode
        // see https://forge.codelutin.com/issues/5699

        {
            SpeciesBatchUIModel model = getUI().getSpeciesTabContent().getModel();
            SpeciesSortMode sortMode = model.getSpeciesSortMode();
            if (sortMode != SpeciesSortMode.NONE) {

                // must reload order
                model.setSpeciesSortMode(SpeciesSortMode.NONE);
                model.setSpeciesSortMode(sortMode);

            }
        }

        {
            BenthosBatchUIModel model = getUI().getBenthosTabContent().getModel();
            SpeciesSortMode sortMode = model.getSpeciesSortMode();
            if (sortMode != SpeciesSortMode.NONE) {

                // must reload order
                model.setSpeciesSortMode(SpeciesSortMode.NONE);
                model.setSpeciesSortMode(sortMode);

            }
        }

    }

    @Override
    public void postFailedAction(Throwable error) {

        afterAction();

        if (error instanceof TuttiWeightComputingException) {

            // elevation error
            TuttiWeightComputingException e = (TuttiWeightComputingException) error;

            getUI().getTabPane().setSelectedIndex(tabInError);

            switch (tabInError) {

                case 1:
                    // species error
                    treatSpeciesBatchError(e);

                    break;

                case 2:

                    // benthos error
                    treatBenthosBatchError(e);

                    break;

                case 3:
                    // marine litter error
                    JTables.doSelectCell(getUI().getMarineLitterTabContent().getTable(), e.getIndex(), 3);

            }

        }

    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected Float computeSpeciesBatches(PersistenceService persistenceService, BatchContainer<SpeciesBatch> computedSpeciesBatches) {
        Float totalSortedWeight = 0f;

        if (computedSpeciesBatches != null) {
            speciesBatchRows = Lists.newArrayList();
            List<SpeciesBatch> children = computedSpeciesBatches.getChildren();
            for (SpeciesBatch batch : children) {
                SpeciesBatchRowModel row = getUI().getSpeciesTabContent().getHandler().loadBatch(batch, null, speciesBatchRows);
                if (persistenceService.isVracBatch(row)) {
                    SampleCategory<?> sampleCategory = row.getFirstSampleCategory();
                    Float weight = Numbers.getValueOrComputedValue(
                            sampleCategory.getCategoryWeight(),
                            sampleCategory.getComputedWeight());
                    totalSortedWeight += weight;
                }
            }
//            getUI().getSpeciesTabContent().getModel().setRows(speciesBatchRows);
        }

        return totalSortedWeight;
    }

    protected Float computeBenthosBatches(PersistenceService persistenceService, BatchContainer<BenthosBatch> computedBenthosBatches) {
        Float totalSortedWeight = 0f;

        if (computedBenthosBatches != null) {
            benthosBatchRows = Lists.newArrayList();
            List<BenthosBatch> children = computedBenthosBatches.getChildren();
            for (BenthosBatch batch : children) {
                BenthosBatchRowModel row = getUI().getBenthosTabContent().getHandler().loadBatch(batch, null, benthosBatchRows);

                if (persistenceService.isVracBatch(row)) {
                    SampleCategory<?> sampleCategory = row.getFirstSampleCategory();
                    Float weight = Numbers.getValueOrComputedValue(
                            sampleCategory.getCategoryWeight(),
                            sampleCategory.getComputedWeight());
                    totalSortedWeight += weight;
                }
            }
//            getUI().getBenthosTabContent().getModel().setRows(benthosBatchRows);

        }
        return totalSortedWeight;
    }

    protected void treatSpeciesBatchError(TuttiWeightComputingException e) {

        int index = e.getIndex();
        SpeciesBatchUIModel speciesBatchUIModel = getUI().getSpeciesTabContent().getModel();

        SpeciesBatchRowModel row;

        if (speciesBatchUIModel.getSpeciesSortMode() != SpeciesSortMode.NONE) {

            // must resort rows in natural order (in service we use this order)
            List<SpeciesBatchRowModel> rows = Lists.newArrayList(speciesBatchUIModel.getRows());
            SpeciesBatchNaturalOrderComparator.sort(rows);
            // get the row
            row = rows.get(index);
            // get the correct rowIndex in the sorted list
            index = speciesBatchUIModel.getRows().indexOf(row);

        } else {

            // correct order can directly get row from list
            List<SpeciesBatchRowModel> rows = speciesBatchUIModel.getRows();
            row = rows.get(index);

        }

        JXTable table = getUI().getSpeciesTabContent().getTable();
        SpeciesBatchTableModel tableModel = (SpeciesBatchTableModel) table.getModel();
        Set<SampleCategoryColumnIdentifier<SpeciesBatchRowModel>> sampleCols = tableModel.getSampleCols();
        selectBatchCell(table, index, row, e.getProperty(), sampleCols, SpeciesBatchTableModel.WEIGHT);

    }

    protected void treatBenthosBatchError(TuttiWeightComputingException e) {

        int index = e.getIndex();

        BenthosBatchUIModel benthosBatchUIModel = getUI().getBenthosTabContent().getModel();

        BenthosBatchRowModel row;
        if (benthosBatchUIModel.getSpeciesSortMode() != SpeciesSortMode.NONE) {

            // must resort rows in natural order (in service we use this order)
            List<BenthosBatchRowModel> rows = Lists.newArrayList(benthosBatchUIModel.getRows());
            SpeciesBatchNaturalOrderComparator.sort(rows);
            // get the row
            row = rows.get(index);
            // get the correct rowIndex in the sorted list
            index = benthosBatchUIModel.getRows().indexOf(row);

        } else {

            // correct order can directly get row from list
            List<BenthosBatchRowModel> rows = benthosBatchUIModel.getRows();
            row = rows.get(index);

        }

        JXTable table = getUI().getBenthosTabContent().getTable();
        BenthosBatchTableModel tableModel = (BenthosBatchTableModel) table.getModel();
        Set<SampleCategoryColumnIdentifier<BenthosBatchRowModel>> sampleCols = tableModel.getSampleCols();

        selectBatchCell(table, index, row, e.getProperty(), sampleCols, BenthosBatchTableModel.WEIGHT);

    }

    protected <R extends SampleCategoryAble> void selectBatchCell(JXTable table,
                                                                  int index,
                                                                  SampleCategoryAble row,
                                                                  String property,
                                                                  Set<SampleCategoryColumnIdentifier<R>> sampleCols,
                                                                  ColumnIdentifier<R> weightColumn) {

        ColumnIdentifier columnIdentifier = weightColumn;

        if (SpeciesBatch.PROPERTY_SAMPLE_CATEGORY_WEIGHT.equals(property)) {

            // get category column for category
            Integer categoryId = row.getFinestCategory().getCategoryId();
            for (SampleCategoryColumnIdentifier<R> sampleCol : sampleCols) {
                if (categoryId.equals(sampleCol.getSampleCategoryId())) {
                    columnIdentifier = sampleCol;
                    break;
                }
            }

        }

        int column = 0;
        TableColumnExt columnExt = table.getColumnExt(columnIdentifier);
        int modelIndex = columnExt.getModelIndex();
        for (TableColumn tableColumn : table.getColumns(false)) {

            if (columnExt.equals(tableColumn) || tableColumn.getModelIndex() > modelIndex) {
                break;
            }
            column++;
        }

        JTables.doSelectCell(table, index, column);

    }

}
