package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiReportUpdaterCallBack.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiReportUpdaterCallBack.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.jaxx.application.swing.action.ApplicationActionException;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.ui.swing.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.auth.AuthenticationInfo;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationInfo;
import org.nuiton.updater.ApplicationUpdaterCallback;

import java.util.Map;

import static org.nuiton.i18n.I18n.t;
import static org.nuiton.i18n.I18n.n;

/**
 * CallBack to update db.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.6
 */
public class TuttiReportUpdaterCallBack implements ApplicationUpdaterCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiReportUpdaterCallBack.class);

    public static final String REPORT_UPDATE_NAME = "report";

    static {
        n("tutti.update.report");
    }

    protected final TuttiUIContext context;

    protected ProgressionModel progressionModel;

    protected boolean reportUpdated;

    protected final AbstractTuttiAction action;

    public TuttiReportUpdaterCallBack(AbstractTuttiAction action,
                                      ProgressionModel progressionModel) {
        this.action = action;
        this.context = action.getContext();
        this.progressionModel = progressionModel;
    }

    public boolean isReportUpdated() {
        return reportUpdated;
    }

    @Override
    public Map<String, ApplicationInfo> updateToDo(Map<String, ApplicationInfo> appToUpdate) {
        Map<String, ApplicationInfo> result = Maps.newHashMap();

        ApplicationInfo info = appToUpdate.get(REPORT_UPDATE_NAME);
        if (info != null) {
            result.put(info.name, info);

            if (info.needAuthentication) {
                // ask auth
                AuthenticationInfo authenticationInfo = context.getAuthenticationInfo(info.url);
                if (authenticationInfo != null) {
                    info.setAuthentication(authenticationInfo.getLogin(), authenticationInfo.getPassword());
                }
            }
        }

        return result;
    }

    @Override
    public void startUpdate(ApplicationInfo info) {

        progressionModel.setMessage(t("tutti.applicationUpdater.startUpdate.report", info.newVersion));
    }

    @Override
    public void updateDone(Map<String, ApplicationInfo> appToUpdate,
                           Map<String, Exception> appUpdateError) {

        boolean doRestart = updateDoneReport(appToUpdate, appUpdateError);

        if (doRestart) {

            reportUpdated = true;
        }
    }

    @Override
    public void aborted(String propertiesURL, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error("Could not update from " + propertiesURL, eee);
        }
        throw ApplicationActionException.propagateError(action, eee);
    }

    protected boolean updateDoneReport(Map<String, ApplicationInfo> appToUpdate,
                                       Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        ApplicationInfo info = getInfo(REPORT_UPDATE_NAME, appToUpdate);
        Exception error = getError(REPORT_UPDATE_NAME, appUpdateError);
        if (error != null) {

            // something bad while updating jre
            String errorMessage;
            if (info != null && info.needAuthentication) {
                errorMessage = t("tutti.updateReport.error.with.auth");
            } else {
                errorMessage = t("tutti.updateReport.error.with.noauth");
            }
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(errorMessage, error));
        } else if (info != null) {

            if (log.isInfoEnabled()) {
                log.info(String.format(
                        "A report update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                        info.oldVersion, info.newVersion));
            }
            doRestart = true;
        }
        return doRestart;
    }

    protected Exception getError(String type,
                                 Map<String, Exception> appUpdateError) {
        return appUpdateError.get(type.toLowerCase());
    }

    protected ApplicationInfo getInfo(String type,
                                      Map<String, ApplicationInfo> appToUpdate) {
        return appToUpdate.get(type.toLowerCase());
    }
}
