package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiApplicationUpdaterCallBack.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiApplicationUpdaterCallBack.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.nuiton.jaxx.application.ApplicationIOUtil;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.jaxx.application.swing.action.ApplicationActionException;
import fr.ifremer.tutti.LabelAware;
import fr.ifremer.tutti.TuttiConfiguration;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.ui.swing.action.AbstractTuttiAction;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationInfo;
import org.nuiton.updater.ApplicationUpdaterCallback;

import java.io.File;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.t;
import static org.nuiton.i18n.I18n.n;

/**
 * CallBack to update jre, application, i18n.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class TuttiApplicationUpdaterCallBack implements ApplicationUpdaterCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiApplicationUpdaterCallBack.class);

    public enum UpdateType implements LabelAware {
        JRE(n("tutti.update.jre")),
        TUTTI(n("tutti.update.tutti")),
        I18N(n("tutti.update.i18n")),
        HELP(n("tutti.update.help"));

        private final String i18nKey;

        private UpdateType(String i18nKey) {
            this.i18nKey = i18nKey;
        }

        @Override
        public String getLabel() {
            return t(i18nKey);
        }
    }

    protected final TuttiUIContext context;

    protected List<UpdateType> types;

    protected ProgressionModel progressionModel;

    protected boolean applicationUpdated;

    protected final AbstractTuttiAction action;

    public TuttiApplicationUpdaterCallBack(AbstractTuttiAction action,
                                           ProgressionModel progressionModel) {
        this.action = action;
        this.context = action.getContext();
        this.progressionModel = progressionModel;
    }

    public void setTypes(UpdateType... types) {
        this.types = Lists.newArrayList(types);
    }

    public boolean isApplicationUpdated() {
        return applicationUpdated;
    }

    @Override
    public Map<String, ApplicationInfo> updateToDo(Map<String, ApplicationInfo> appToUpdate) {
        Map<String, ApplicationInfo> result = Maps.newHashMap();

        for (UpdateType type : types) {
            ApplicationInfo info = getInfo(type, appToUpdate);
            if (info != null) {
                result.put(info.name, info);
            }
        }
        return result;
    }

    @Override
    public void startUpdate(ApplicationInfo info) {
        if (UpdateType.JRE.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(t("tutti.applicationUpdater.startUpdate.jre", info.newVersion));
        }

        if (UpdateType.TUTTI.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(t("tutti.applicationUpdater.startUpdate.tutti", info.newVersion));
        }

        if (UpdateType.I18N.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(t("tutti.applicationUpdater.startUpdate.i18n", info.newVersion));
        }
        if (UpdateType.HELP.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(t("tutti.applicationUpdater.startUpdate.help", info.newVersion));
        }
    }

    @Override
    public void updateDone(Map<String, ApplicationInfo> appToUpdate,
                           Map<String, Exception> appUpdateError) {

        boolean updateJRE = updateDoneJre(appToUpdate, appUpdateError);
        boolean updateTutti = updateDoneTutti(appToUpdate, appUpdateError);
        boolean updateI18n = updateDoneI18n(appToUpdate, appUpdateError);
        boolean updateHelp = updateDoneHelp(appToUpdate, appUpdateError);

        boolean doRestart = updateJRE || updateTutti || updateI18n || updateHelp;

        if (doRestart) {

            applicationUpdated = true;
        }
    }

    @Override
    public void aborted(String propertiesURL, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error("Could not update from " + propertiesURL, eee);
        }
        throw ApplicationActionException.propagateError(action, eee);
    }

    protected boolean updateDoneJre(Map<String, ApplicationInfo> appToUpdate,
                                    Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.JRE, appUpdateError);
        if (error != null) {

            // something bad while updating jre
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(t("tutti.applicationUpdater.jre.error"), error));
        } else {
            ApplicationInfo info = getInfo(UpdateType.JRE, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A jre update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneTutti(Map<String, ApplicationInfo> appToUpdate,
                                      Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.TUTTI, appUpdateError);
        if (error != null) {

            // something bad while updating application
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(t("tutti.applicationUpdater.tutti.error"), error));
        } else {
            ApplicationInfo info = getInfo(UpdateType.TUTTI, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A tutti update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;

                TuttiConfiguration config = context.getConfig();

                // must remove the enumeration file at exit
                File enumerationPath = config.getDbEnumerationPath();
                ApplicationIOUtil.forceDeleteOnExit(
                        enumerationPath,
                        t("tutti.applicationUpdater.updateDone.deleteDirectory.enum.error", enumerationPath)
                );

                // must remove the db conf file at exit
                File dbConfPath = config.getDbConfigurationPath();
                ApplicationIOUtil.forceDeleteOnExit(
                        dbConfPath,
                        t("tutti.applicationUpdater.updateDone.deleteDirectory.dbConf.error", dbConfPath)
                );

                //FIXME-check this is necessary: i18n is no more generated ?
                // must also remove i18n directory
                File i18nDirectory = context.getConfig().getI18nDirectory();
                ApplicationIOUtil.forceDeleteOnExit(
                        i18nDirectory,
                        t("tutti.applicationUpdater.updateDone.deleteDirectory.i18n.error", i18nDirectory)
                );

                // must remove db cache directory
                File cacheDirectory = config.getCacheDirectory();
                ApplicationIOUtil.forceDeleteOnExit(
                        cacheDirectory,
                        t("tutti.applicationUpdater.updateDone.deleteDirectory.caches.error", cacheDirectory)
                );
            }
        }
        return doRestart;
    }

    protected boolean updateDoneI18n(Map<String, ApplicationInfo> appToUpdate,
                                     Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.I18N, appUpdateError);
        if (error != null) {

            // something bad while updating i18n
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(t("tutti.applicationUpdater.i18n.error"), error));
        } else {
            ApplicationInfo info = getInfo(UpdateType.I18N, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A i18n update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneHelp(Map<String, ApplicationInfo> appToUpdate,
                                     Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.HELP, appUpdateError);
        if (error != null) {

            // something bad while updating help
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(t("tutti.applicationUpdater.help.error"), error));
        } else {
            ApplicationInfo info = getInfo(UpdateType.HELP, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A help update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected ApplicationInfo getInfo(UpdateType type,
                                      Map<String, ApplicationInfo> appToUpdate) {
        return appToUpdate.get(type.name().toLowerCase());
    }

    protected Exception getError(UpdateType type,
                                 Map<String, Exception> appUpdateError) {
        return appUpdateError.get(type.name().toLowerCase());
    }

    protected File getDbDirectory(ApplicationInfo info) {
        File[] sources = info.destDir.listFiles();
        Preconditions.checkState(
                sources != null && sources.length == 1,
                "Downloaded db should contains one directory at " + info.destDir);
        File result = sources[0];
        return result;
    }
}
