package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: RemoveBenthosProtocolAction.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/action/RemoveBenthosProtocolAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolSpeciesRowModel;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolSpeciesTableModel;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUI;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUIHandler;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUIModel;
import jaxx.runtime.SwingUtil;

import javax.swing.JTable;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * To remove all the selected benthos rows from protocol.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class RemoveBenthosProtocolAction extends AbstractTuttiAction<EditProtocolUIModel, EditProtocolUI, EditProtocolUIHandler> {


    /**
     * Set of removed species.
     *
     * @since 2.8
     */
    protected Set<Species> removedSpecies;

    /**
     * Set of removed rows.
     *
     * @since 2.8
     */
    protected Set<EditProtocolSpeciesRowModel> removedRows;


    public RemoveBenthosProtocolAction(EditProtocolUIHandler handler) {
        super(handler, false);
    }

    @Override
    public void doAction() throws Exception {

        JTable table = handler.getBenthosTable();

        // need to have a selection
        Preconditions.checkState(!table.getSelectionModel().isSelectionEmpty());

        EditProtocolSpeciesTableModel tableModel =
                (EditProtocolSpeciesTableModel) table.getModel();

        removedSpecies = Sets.newHashSet();
        removedRows = Sets.newHashSet();

        EditProtocolUIModel model = getModel();

        for (Integer rowIndex : SwingUtil.getSelectedModelRows(table)) {

            // get row to remove
            EditProtocolSpeciesRowModel selectedRow =
                    tableModel.getEntry(rowIndex);

            // re-add all synonym of this taxon to the species / benthos combobox
            Species species = selectedRow.getSpecies();
            removedSpecies.add(species);

            Integer taxonId = species.getReferenceTaxonId();
            List<Species> allSynonyms = Lists.newArrayList(
                    model.getAllSynonyms(String.valueOf(taxonId)));
            allSynonyms.remove(species);
            model.getAllSynonyms().addAll(allSynonyms);

            // mark row to be removed at the very last moment
            removedRows.add(selectedRow);
        }

        // reorder the list by name, otherwise,
        // all the species without a reftax code will be at the end
        Collections.sort(model.getAllSynonyms(),
                         TuttiEntities.SPECIES_BY_NAME_COMPARATOR);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        // update comboboxes
        getUI().getBenthosComboBox().addItems(removedSpecies);
        getUI().getSpeciesComboBox().addItems(removedSpecies);

        // remove all rows from model
        getModel().getBenthosRow().removeAll(removedRows);

        // fire table data changed
        handler.getBenthosTableModel().fireTableDataChanged();

        // clear table selection
        handler.getBenthosTable().clearSelection();

        // notify user
        sendMessage(t("tutti.flash.info.benthos.remove.from.protocol"));
    }
}
