package fr.ifremer.tutti.ui.swing.content.report;

/*
 * #%L
 * Tutti :: UI
 * $Id: ReportUIHandler.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/report/ReportUIHandler.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.nuiton.jaxx.application.swing.util.CloseableUI;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.service.report.ReportService;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.swing.editor.FileEditor;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JComponent;
import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler of {@link ReportUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.9
 */
public class ReportUIHandler extends AbstractTuttiUIHandler<ReportUIModel, ReportUI> implements CloseableUI {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ReportUIHandler.class);

    @Override
    public void beforeInit(ReportUI ui) {

        super.beforeInit(ui);

        ReportService reportService = getContext().getReportService();

        ReportUIModel model = new ReportUIModel();

        // set cruiseId
        String cruiseId = getDataContext().getCruiseId();
        model.setCruiseId(cruiseId);

        // get all report availables
        List<File> reports = reportService.getAvailableReports();
        model.setReports(reports);

        // get all fishing operation
        List<FishingOperation> fishingOperations =
                getContext().getPersistenceService().getAllFishingOperation(cruiseId);
        model.setFishingOperations(fishingOperations);

        // get default output file name
        File outputFile = reportService.getDefaultOutputFile();
        model.setOutputFile(outputFile);

        listModelIsModify(model);

        ui.setContextValue(model);
    }

    @Override
    public void afterInit(ReportUI ui) {

        initUI(ui);

        ReportUIModel model = getModel();

        initBeanFilterableComboBox(ui.getReportComboBox(),
                                   Lists.newArrayList(model.getReports()),
                                   model.getReport());

        initBeanFilterableComboBox(ui.getFishingOperationComboBox(),
                                   Lists.newArrayList(model.getFishingOperations()),
                                   model.getFishingOperation());

        FileEditor outputFileField = ui.getOutputFileField();
        outputFileField.setDirectoryEnabled(false);
        outputFileField.setFileEnabled(true);
        outputFileField.setExts(".pdf");
        outputFileField.setExtsDescription(t("tutti.common.file.pdf"));

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        registerValidators(validator);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getFishingOperationComboBox();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
    }

    @Override
    public boolean quitUI() {
        return true;
    }

    @Override
    public SwingValidator<ReportUIModel> getValidator() {
        return ui.getValidator();
    }

    public void showReport() {

        File file = getModel().getOutputFile();

        Preconditions.checkState(file.exists());

        Desktop desktop = TuttiUIUtil.getDesktopForBrowse();
        try {
            desktop.browse(file.toURI());

        } catch (IOException e) {
            getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
        }
    }

}