package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.LabelAware;
import fr.ifremer.tutti.TuttiConfiguration;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.TuttiDataContext;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.service.catches.ValidateCruiseOperationsService;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.attachment.ButtonAttachment;
import fr.ifremer.tutti.ui.swing.util.computable.ComputableDataEditor;
import fr.ifremer.tutti.util.Weights;
import jaxx.runtime.swing.JAXXWidgetUtil;
import jaxx.runtime.swing.editor.cell.NumberCellEditor;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.JXTitledPanel;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.FontHighlighter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.AbstractApplicationUIHandler;
import org.nuiton.jaxx.application.swing.action.ApplicationActionUI;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;
import org.nuiton.jaxx.widgets.number.NumberEditor;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.border.LineBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DecimalFormat;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * Contract of any UI handler.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public abstract class AbstractTuttiUIHandler<M, UI extends TuttiUI<M, ?>> extends AbstractApplicationUIHandler<M, UI> implements UIMessageNotifier {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiUIHandler.class);

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void showInformationMessage(String message) {
        getContext().showInformationMessage(message);
    }

    @Override
    public TuttiUIContext getContext() {
        return (TuttiUIContext) super.getContext();
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    public TuttiConfiguration getConfig() {
        return getContext().getConfig();
    }

    public PersistenceService getPersistenceService() {
        return getContext().getPersistenceService();
    }

    public ValidationService getValidationService() {
        return getContext().getValidationService();
    }

    public ValidateCruiseOperationsService getValidateCruiseOperationsService() {
        return getContext().getValidateCruiseOperationsService();
    }

    @Override
    public Component getTopestUI() {
        Component result;
        ApplicationActionUI actionUI = getContext().getActionUI();
//        if (actionUI.isVisible()) {
        result = actionUI;
//        } else {
//            result = getContext().getMainUI();
//        }
        return result;
    }

    public void clearValidators() {
        MainUI main = getContext().getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
    }

    public String getWeightStringValue(WeightUnit weightUnit, Float weight) {
        String textValue;
        if (weight != null) {

            int numberDigits = weightUnit.getNumberDigits();

            DecimalFormat weightDecimalFormat = Weights.getDecimalFormat(1, numberDigits);
            textValue = weightDecimalFormat.format(weight);

        } else {
            textValue = "";
        }
        return textValue;
    }

    public String getWeightStringValue(JComponent component, Float weight) {


        WeightUnit weightUnit = (WeightUnit) component.getClientProperty("addWeightUnit");
        return getWeightStringValue(weightUnit, weight);
    }


    @Override
    public <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                getContext().getDecoratorService();

        Preconditions.checkNotNull(type);

        Decorator decorator = decoratorService.getDecoratorByType(type, name);
        if (decorator == null) {

            if (LabelAware.class.isAssignableFrom(type)) {
                decorator = getDecorator(LabelAware.class, null);
            }
        }
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    @Override
    protected void addHighlighters(final JXTable table) {

        HighlightPredicate notSelectedPredicate = new HighlightPredicate.NotHighlightPredicate(HighlightPredicate.IS_SELECTED);
        HighlightPredicate rowIsInvalidPredicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {

                boolean result = false;
                if (adapter.isEditable()) {
                    AbstractApplicationTableModel model = (AbstractApplicationTableModel) table.getModel();
                    int viewRow = adapter.row;
                    int modelRow = adapter.convertRowIndexToModel(viewRow);
                    AbstractTuttiBeanUIModel row = (AbstractTuttiBeanUIModel) model.getEntry(modelRow);
                    result = !row.isValid();
                }
                return result;
            }
        };
        HighlightPredicate rowIsValidPredicate =
                new HighlightPredicate.NotHighlightPredicate(rowIsInvalidPredicate);
        Highlighter selectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.IS_SELECTED,
                getConfig().getColorSelectedRow());
        table.addHighlighter(selectedHighlighter);

        // paint in a special color for read only cells (not selected)
        Highlighter readOnlyHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.READ_ONLY,
                        notSelectedPredicate),
                getConfig().getColorRowReadOnly());
        table.addHighlighter(readOnlyHighlighter);

        // paint in a special color for read only cells (selected)
        Highlighter readOnlySelectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.READ_ONLY,
                        HighlightPredicate.IS_SELECTED),
                getConfig().getColorRowReadOnly().darker());
        table.addHighlighter(readOnlySelectedHighlighter);

        // paint in a special color inValid rows (not selected)
        Highlighter validHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.EDITABLE,
                        notSelectedPredicate,
                        rowIsInvalidPredicate),
                getConfig().getColorRowInvalid());
        table.addHighlighter(validHighlighter);

        // paint in a special color inValid rows (selected)
        Highlighter validSelectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.EDITABLE,
                        HighlightPredicate.IS_SELECTED,
                        rowIsInvalidPredicate),
                getConfig().getColorRowInvalid().darker());
        table.addHighlighter(validSelectedHighlighter);

        // use configured color odd row (not for selected)
        Highlighter evenHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.ODD,
                        notSelectedPredicate,
                        rowIsValidPredicate,
                        HighlightPredicate.READ_ONLY),
                getConfig().getColorAlternateRow().darker());
        table.addHighlighter(evenHighlighter);

        Highlighter evenNotReadOnlyHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.ODD,
                        notSelectedPredicate,
                        rowIsValidPredicate,
                        HighlightPredicate.EDITABLE),
                getConfig().getColorAlternateRow());
        table.addHighlighter(evenNotReadOnlyHighlighter);

        // use configured color odd row (for selected)
        Highlighter evenSelectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.ODD,
                        HighlightPredicate.IS_SELECTED,
                        rowIsValidPredicate,
                        HighlightPredicate.EDITABLE),
                getConfig().getColorSelectedRow());
        table.addHighlighter(evenSelectedHighlighter);


        // paint in a special color inValid rows
        Font font = table.getFont().deriveFont(Font.BOLD);
        Highlighter selectHighlighter = new FontHighlighter(HighlightPredicate.IS_SELECTED, font);
        table.addHighlighter(selectHighlighter);
    }

    protected void listenModelModifiy(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(AbstractTuttiBeanUIModel.PROPERTY_MODIFY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean modify = (Boolean) evt.getNewValue();
                if (modify != null && modify) {
                    ((AbstractTuttiBeanUIModel) getModel()).setModify(true);
                }
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- Init methods                                                       --//
    //------------------------------------------------------------------------//

    @Override
    protected void initUIComponent(Object component) {
        if (component instanceof NumberEditor) {
            initNumberEditor((NumberEditor) component);
        } else if (component instanceof JXTitledPanel) {
            initJXTitledPanel((JXTitledPanel) component);
        } else if (component instanceof ButtonAttachment) {

            initButtonAttachment((ButtonAttachment) component);
        } else {
            super.initUIComponent(component);
        }
    }

    protected void initJXTitledPanel(JXTitledPanel jTextField) {
//        Boolean boldFont = (Boolean) jTextField.getClientProperty("boldFont");
//        if (boldFont!= null && boldFont) {
//            Font font = jTextField.getFont().deriveFont(Font.BOLD, 13.f);
//            jTextField.setTitleFont(font);
//        }
    }

    @Override
    protected void initTextField(JTextField jTextField) {
        super.initTextField(jTextField);
        Boolean computed = (Boolean) jTextField.getClientProperty("computed");
        if (computed != null && computed) {
            Font font = jTextField.getFont().deriveFont(Font.ITALIC);
            jTextField.setFont(font);
            jTextField.setEditable(false);
            jTextField.setEnabled(false);
            jTextField.setDisabledTextColor(getConfig().getColorComputedWeights());
        }
    }

    protected void initButtonAttachment(ButtonAttachment component) {

        component.init();
    }

    @Override
    protected void initLabel(JLabel jLabel) {

        WeightUnit weightUnit = (WeightUnit) jLabel.getClientProperty("addWeightUnit");
        if (weightUnit != null) {
            String text = weightUnit.decorateLabel(jLabel.getText());
            jLabel.setText(text);

            String tip = weightUnit.decorateTip(jLabel.getToolTipText());
            jLabel.setToolTipText(tip);

            Component labelFor = jLabel.getLabelFor();
            if (labelFor instanceof ComputableDataEditor) {

                // set also the number of digits (4 for kg, 1 for g)
                ComputableDataEditor editor = (ComputableDataEditor) labelFor;
                editor.setNumberPattern(weightUnit.getNumberEditorPattern());
                editor.setDecimalNumber(weightUnit.getNumberDigits());
            } else if (labelFor instanceof NumberEditor) {

                // set also the number of digits (4 for kg, 1 for g)
                NumberEditor editor = (NumberEditor) labelFor;
                editor.setNumberPattern(weightUnit.getNumberEditorPattern());
            }
        }
    }

    protected void initNumberEditor(NumberEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init number editor " + editor.getName());
        }
        editor.init();

        // Force binding if value is already in model
        Number model = editor.getModel().getNumberValue();
        if (model != null) {
            editor.setNumberValue(null);
            editor.setNumberValue(model);
        }

        if (isAutoSelectOnFocus(editor)) {

            addAutoSelectOnFocus(editor.getTextField());
        }
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void registerValidators(SwingValidator... validators) {
        MainUI main = getContext().getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
        for (SwingValidator validator : validators) {
            handler.registerValidator(validator);
        }
    }

    protected void listenValidatorValid(SimpleBeanValidator validator,
                                        final AbstractTuttiBeanUIModel model) {
        validator.addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" +
                              evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    protected void listenValidationTableHasNoFatalError(final SimpleBeanValidator validator,
                                                        final AbstractTuttiBeanUIModel model) {
        getContext().getMainUI().getValidatorMessageWidget().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                boolean valid = !validator.hasFatalErrors();
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" + valid + "]");
                }
                model.setValid(valid);
            }
        });
    }

    protected void listModelIsModify(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(new PropertyChangeListener() {

            final Set<String> excludeProperties = getPropertiesToIgnore();

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (!excludeProperties.contains(evt.getPropertyName())) {
                    ((AbstractTuttiBeanUIModel) evt.getSource()).setModify(true);
                }
            }
        });
    }

    protected Set<String> getPropertiesToIgnore() {
        return Sets.newHashSet(
                AbstractTuttiBeanUIModel.PROPERTY_MODIFY,
                AbstractTuttiBeanUIModel.PROPERTY_VALID);
    }

    protected void closeUI(TuttiUI ui) {
        ui.getHandler().onCloseUI();
    }

    protected String buildReminderLabelTitle(Species species,
                                             Iterable<SampleCategory<?>> categories,
                                             String prefix,
                                             String suffix) {
        return buildReminderLabelTitle(
                decorate(species, DecoratorService.WITH_SURVEY_CODE),
                categories,
                prefix,
                suffix);

    }

    protected String buildReminderLabelTitle(String species,
                                             Iterable<SampleCategory<?>> categories,
                                             String prefix,
                                             String suffix) {
        StringBuilder title = new StringBuilder("<html><body style='color:black;'>" + prefix);

        title.append(" - [<strong>").append(species).append("</strong>]");

        if (categories != null) {
            for (SampleCategory<?> sampleCategory : categories) {
                if (sampleCategory.getCategoryValue() != null) {
                    title.append(" - ");
                    title.append(decorate(sampleCategory.getCategoryValue()));
                }
            }
        }

        title.append(" - ").append(suffix).append("</body></html>");
        return title.toString();
    }

    protected <R> TableColumnExt addFloatColumnToModel(TableColumnModel model,
                                                       ColumnIdentifier<R> identifier,
                                                       WeightUnit weightUnit,
                                                       JTable table) {

        Preconditions.checkNotNull(weightUnit);
        NumberCellEditor<Float> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(weightUnit.getNumberEditorPattern());

        TableCellRenderer renderer =
                newNumberCellRenderer(table.getDefaultRenderer(Number.class));

        return addColumnToModel(model, editor, renderer, identifier, weightUnit);
    }

    protected <R> TableColumnExt addColumnToModel(TableColumnModel model,
                                                  TableCellEditor editor,
                                                  TableCellRenderer renderer,
                                                  ColumnIdentifier<R> identifier,
                                                  WeightUnit weightUnit) {

        TableColumnExt col = new TableColumnExt(model.getColumnCount());
        col.setCellEditor(editor);
        col.setCellRenderer(renderer);
        String label = t(identifier.getHeaderI18nKey());
        if (weightUnit != null) {
            label = weightUnit.decorateLabel(label);
        }
        col.setHeaderValue(label);
        String tip = t(identifier.getHeaderTipI18nKey());
        if (weightUnit != null) {
            tip = weightUnit.decorateTip(tip);
        }
        col.setToolTipText(tip);

        col.setIdentifier(identifier);
        model.addColumn(col);
        // by default no column is sortable, must specify it
        col.setSortable(false);
        return col;
    }

}
