package fr.ifremer.tutti.ui.swing.spatial;

/*
 * #%L
 * Tutti :: UI
 * $Id: DmsCoordinate.java 1316 2013-10-25 16:43:35Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.8/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/spatial/DmsCoordinate.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.jdesktop.beans.AbstractSerializableBean;

/**
 * Geo coordinate in degree, minute, second format.
 * <p/>
 * Created on 10/23/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 2.8
 */
public class DmsCoordinate extends AbstractSerializableBean {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SIGN = "sign";

    public static final String PROPERTY_DEGREE = "degree";

    public static final String PROPERTY_MINUTE = "minute";

    public static final String PROPERTY_SECOND = "second";

    protected boolean sign;

    protected Integer degree;

    protected Integer minute;

    protected Integer second;

    public static DmsCoordinate empty() {
        DmsCoordinate r = new DmsCoordinate();
        return r;
    }

    /**
     * Methode statique de fabrique de position a partir d'une valeur du format
     * decimal.
     * <p/>
     * Note : Si la valeur (au format decimal) vaut <code>null</code>, alors on
     * reinitialise les composants de la position a <code>null</code> et la
     * methode {@link #isNull()} vaudra alors {@code true}.
     *
     * @param decimal la valeur au format decimal
     * @return une nouvelle instance de position convertie
     */
    public static DmsCoordinate valueOf(Float decimal) {
        DmsCoordinate r = new DmsCoordinate();
        r.fromDecimal(decimal);
        return r;
    }

    /**
     * Methode statique de fabrique de position a partir d'une valeur du format
     * degre-minute-seconde.
     *
     * @param d la valeur des degres
     * @param m la valeur des minutes
     * @param s la valeur des secondes
     * @return une nouvelle instance de position convertie
     */
    public static DmsCoordinate valueOf(boolean sign, int d, int m, int s) {
        DmsCoordinate r = new DmsCoordinate();
        r.setSign(sign);
        r.setDegree(d);
        r.setMinute(m);
        r.setSecond(s);
        return r;
    }

    public boolean isSign() {
        return sign;
    }

    public Integer getDegree() {
        return degree;
    }

    public Integer getMinute() {
        return minute;
    }

    public Integer getSecond() {
        return second;
    }

    public void setSign(boolean sign) {
        Object oldValue = isSign();
        this.sign = sign;
        firePropertyChange(PROPERTY_SIGN, oldValue, sign);
    }

    public void setDegree(Integer degree) {
        Object oldValue = getDegree();
        this.degree = degree;
        firePropertyChange(PROPERTY_DEGREE, oldValue, degree);
    }

    public void setMinute(Integer minute) {
        Object oldValue = getMinute();
        this.minute = minute;
        firePropertyChange(PROPERTY_MINUTE, oldValue, minute);
    }

    public void setSecond(Integer second) {
        Object oldValue = getSecond();
        this.second = second;
        firePropertyChange(PROPERTY_SECOND, oldValue, second);
    }

    public boolean isDegreeNull() {
        return degree == null || degree == 0;
    }

    public boolean isMinuteNull() {
        return minute == null || minute == 0;
    }

    public boolean isSecondNull() {
        return second == null || second == 0;
    }

    /**
     * @return {@code true} si aucune composante n'est renseignée,
     * {@code false} autrement.
     */
    public boolean isNull() {
        return degree == null && minute == null && second == null;
    }

    /**
     * Mets a jour les composants de la position a partir d'une valeur decimal.
     * <p/>
     * Note : Si la valeur (au format decimal) vaut <code>null</code>, alors on
     * reinitialise les composants de la position a <code>null</code> et la
     * methode {@link #isNull()} vaudra alors {@code true}.
     *
     * @param decimal la valeur decimale a convertir (qui peut etre nulle).
     */
    public void fromDecimal(Float decimal) {
        Integer d = null;
        Integer m = null;
        Integer s = null;
        boolean si = false;
        if (decimal != null) {
            si = decimal < 0;

            decimal = Math.abs(decimal);
            int remain = 0;

            d = (int) (Math.round(decimal + 0.5) - 1);
            m = 0;
            s = 0;
            decimal = 60.0f * (decimal - d);
            if (decimal > 0) {
                m = (int) (Math.round(decimal + 0.5) - 1);
                decimal = 60 * (decimal - m);
                if (decimal > 0) {
                    s = (int) (Math.round(decimal + 0.5) - 1);
                    remain = (int) (10 * (decimal - s));
                }
            }
            if (remain > 9) {
                s++;
            }
            if (s == 60) {
                m++;
                s = 0;
            }
            if (m == 60) {
                d++;
                m = 0;
            }

            // clean not used values
            if (m == 0) {
                m = null;
            }
            if (s == 0) {
                s = null;
            }
        }

        if (d == 0) {
            d = null;
        }
        degree = d;
        minute = m;
        second = s;
        sign = si;

    }

    public Float toDecimal() {
        if (isNull()) {
            return null;
        }
        Integer d = getNotNullDegree();
        Integer m = getNotNullMinute();

        Integer s = getNotNullSecond();

        Float result = Float.valueOf(d);

        if (m > 0) {
            result += (float) m / 60;
            if (s == 0) {
                result += 0.5f / 3600;
            }
        }
        if (s > 0) {
            result += ((float) s + 0.5f) / 3600;
        }

        if (sign) {
            result *= -1;
        }
        return result;
    }

    public Integer getSignedDegree() {
        Integer result = null;
        if (!isDegreeNull()) {
            result = degree;
            if (isSign()) {
                result *= -1;
            }
        }
        return result;
    }

    public int getNotNullDegree() {
        return isDegreeNull() ? 0 : degree;
    }

    public int getNotNullMinute() {
        return isMinuteNull() ? 0 : minute;
    }


    public int getNotNullSecond() {
        return isSecondNull() ? 0 : second;
    }

    public boolean isLatitudeDegreeValid() {
        boolean result = isDegreeValid(false);
        return result;
    }

    public boolean isLongitudeDegreeValid() {
        boolean result = isDegreeValid(true);
        return result;
    }

    public boolean isMinuteValid() {
        boolean result = true;
        if (!isMinuteNull()) {
            if (60 == minute) {

                // check minute and second are null
                result = isSecondNull();
            } else {
                result = 0 <= minute && minute < 60;
            }
        }
        return result;
    }

    public boolean isSecondValid() {
        boolean result = isSecondNull() || (0 <= second && second < 60);
        return result;
    }

    @Override
    public String toString() {
        return "DmsCoordinateComponent{" +
               "sign=" + sign +
               ", degree=" + degree +
               ", minute=" + minute +
               ", second=" + second +
               '}';
    }

    protected boolean isDegreeValid(boolean longitude) {
        boolean result = true;
        if (!isDegreeNull()) {
            int bound = longitude ? 180 : 90;
            if (bound == degree) {

                // check minute and second are null
                result = isMinuteNull() && isSecondNull();
            } else {
                result = degree < bound;
            }
        }
        return result;
    }
}
