package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiAction.java 1274 2013-10-04 07:14:17Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.7/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/AbstractTuttiAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.service.TuttiDataContext;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import jaxx.runtime.FileChooserUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.decorator.Decorator;
import org.nuiton.util.StringUtil;

import javax.swing.JOptionPane;
import java.awt.Component;
import java.io.File;

import static org.nuiton.i18n.I18n._;

/**
 * Tutti base action.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public abstract class AbstractTuttiAction<M extends AbstractBean, UI extends TuttiUI<M, ?>, H extends AbstractTuttiUIHandler<M, UI>>
        extends AbstractBean {

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractTuttiAction.class);

    public static final String PROPERTY_DONE = "done";

    protected final H handler;

    protected String actionDescription;

    protected final boolean hideBody;

    protected final Object lock = new Object();

    public abstract void doAction() throws Exception;

    protected AbstractTuttiAction(H handler, boolean hideBody) {
        this.handler = handler;
        this.hideBody = hideBody;
    }

    public boolean prepareAction() throws Exception {
        // by default nothing to prepare
        return true;
    }

    protected void releaseAction() {
        // by default nothing to clean
        firePropertyChange(PROPERTY_DONE, null, true);
    }

    public void postSuccessAction() {
        // by default nothing to do after action
    }

    public void postFailedAction(Throwable error) {
        // by default nothing to do after action
    }

    public H getHandler() {
        return handler;
    }

    public M getModel() {
        return handler.getModel();
    }

    public final UI getUI() {
        return handler.getUI();
    }

    public TuttiUIContext getContext() {
        return handler.getContext();
    }

    public String getActionDescription() {
        return actionDescription;
    }

    public void setActionDescription(String actionDescription) {
        this.actionDescription = actionDescription;
    }

    public void setProgressionModel(ProgressionModel progressionModel) {
        getContext().getActionUI().getModel().setProgressionModel(progressionModel);
    }

    protected ProgressionModel getProgressionModel() {
        return getContext().getActionUI().getModel().getProgressionModel();
    }

    public boolean isHideBody() {
        return hideBody;
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    protected TuttiApplicationConfig getConfig() {
        return getContext().getConfig();
    }

    protected void sendMessage(String message) {
        getContext().showInformationMessage(message);
    }

    protected <O> Decorator<O> getDecorator(Class<O> type, String name) {
        Decorator<O> decorator = handler.getDecorator(type, name);
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        return getDecorator(object.getClass(), null).toString(object);
    }

    protected String decorate(Object object, String context) {
        return getDecorator(object.getClass(), context).toString(object);
    }

    protected boolean askAdminPassword(String askMessage,
                                       String askMessageTitle,
                                       String errorMessage,
                                       String errorMessageTitle) {

        Component container = getContext().getActionUI();

        String answer;
        boolean result;
        do {
            answer = JOptionPane.showInputDialog(container,
                                                 askMessage,
                                                 askMessageTitle,
                                                 JOptionPane.WARNING_MESSAGE);
            if (answer != null) {
                String cryptedAnswer = StringUtil.encodeMD5(answer);
                String correctAnswer = getConfig().getAdminPassword();
                result = StringUtils.equals(cryptedAnswer, correctAnswer);

                if (!result) {
                    JOptionPane.showMessageDialog(container,
                                                  errorMessage,
                                                  errorMessageTitle,
                                                  JOptionPane.ERROR_MESSAGE);
                }

            } else {
                result = false;
            }

        } while (!result && answer != null);
        return result;
    }

    /**
     * Choisir un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou le fichier incoming si l'opération a été
     * annulée
     */
    protected File chooseFile(String title,
                              String buttonLabel,
                              String... filters) {

        File file = FileChooserUtil.getFile(title,
                                            buttonLabel,
                                            getContext().getMainUI(),
                                            filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        if (file != null) {
            File newDir = file.isDirectory() ? file : file.getParentFile();
            FileChooserUtil.setCurrentDirectory(newDir);
        }
        return file;
    }

    /**
     * Sauver un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou {@code null} si pas de fichier choisi ou
     * pas voulu écrasé un fichier existant.
     */
    protected File saveFile(File defaultFile,
                            String filename,
                            String extension,
                            String title,
                            String buttonLabel,
                            String... filters) {

        if (defaultFile != null && FileChooserUtil.isCurrentDirectoryDefault()) {

            // set default directory to this one
            FileChooserUtil.setCurrentDirectory(defaultFile);
        }
        File file = saveFile(filename, extension, title, buttonLabel, filters);
        return file;
    }

    /**
     * Sauver un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou {@code null} si pas de fichier choisi ou
     * pas voulu écrasé un fichier existant.
     */
    protected File saveFile(String filename,
                            String extension,
                            String title,
                            String buttonLabel,
                            String... filters) {

        boolean withExtension = StringUtils.isNotBlank(extension);
        String filenameSuffix = withExtension ? "." + extension : "";
        File file = FileChooserUtil.saveFile(filename + filenameSuffix, title, buttonLabel, getContext().getMainUI(), filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        if (file != null) {
            Preconditions.checkState(!file.isDirectory());

            // add extension if missing
            if (withExtension && !file.getName().endsWith(filenameSuffix)) {
                file = new File(file.getParentFile(), file.getName() + filenameSuffix);
            }

            // ask user to confirm overwrite.
            boolean confirm = askOverwriteFile(file);

            if (confirm) {

                // on conserve le répertoire (pour une prochaine utilisation)
                FileChooserUtil.setCurrentDirectory(file.getParentFile());
            } else {

                // l'utilisateur n'a pas confirmé l'écrasement
                // donc pas de fichier en retour
                file = null;
            }
        }

        return file;
    }

    private boolean askOverwriteFile(File file) {
        boolean result;
        if (file.exists()) {

            // file exists ask user to overwrite
            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    _("tutti.common.askOverwriteFile.message", file),
                    _("tutti.common.askOverwriteFile.help"));

            result = JOptionPane.showConfirmDialog(
                    getHandler().getTopestUI(),
                    htmlMessage,
                    _("tutti.common.askOverwriteFile.title"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE) == JOptionPane.OK_OPTION;
        } else {

            // file does not exist
            result = true;
        }
        return result;
    }

    protected boolean askBeforeDelete(String title, String message) {
        String htmlMessage = String.format(
                AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                message,
                _("tutti.common.askBeforeDelete.help"));
        int i = JOptionPane.showConfirmDialog(
                getHandler().getTopestUI(),
                htmlMessage,
                title,
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);

        boolean result = i == JOptionPane.OK_OPTION;
        return result;
    }

    protected void displayInfoMessage(String title, String message) {
        JOptionPane.showMessageDialog(
                getHandler().getTopestUI(),
                message,
                title,
                JOptionPane.INFORMATION_MESSAGE);
    }

    protected void displayWarningMessage(String title, String message) {
        JOptionPane.showMessageDialog(
                getHandler().getTopestUI(),
                message,
                title,
                JOptionPane.WARNING_MESSAGE);
    }

    protected void createProgressionModelIfRequired(int total) {
        ProgressionModel progressionModel = getProgressionModel();
        if (progressionModel==null) {
            progressionModel = new ProgressionModel();
            progressionModel.setTotal(total);
            progressionModel.setMessage("");
            progressionModel.setCurrent(0);
            setProgressionModel(progressionModel);
        } else {
            progressionModel.adaptTotal(total);
        }
    }
}
