package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiActionSwingWorker.java 983 2013-05-22 12:28:46Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/TuttiActionSwingWorker.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Worker to execute logic action.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.3
 */
public class TuttiActionSwingWorker<A extends AbstractTuttiAction> extends SwingWorker<Void, String> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiActionSwingWorker.class);

    /**
     * Timer used to launch timerTask (to open load dialog).
     *
     * @since 1.0.2
     */
    private static final Timer t = new Timer();

    protected final TuttiActionUI actionUI;

    protected final A action;

    protected Throwable error;

    protected TimerTask timer;

    protected TuttiActionSwingWorker(A action) {

        this.action = action;
        this.actionUI = action.getContext().getActionUI();
    }

    public Throwable getError() {
        return error;
    }

    public boolean isFailed() {
        return error != null;
    }

    @Override
    protected Void doInBackground() throws Exception {

        if (!isCancelled()) {

            timer = new TuttiActionTimerTask();

            t.schedule(timer, 1000);

            try {

                action.doAction();

            } catch (Throwable e) {
                if (log.isErrorEnabled()) {
                    log.error("Task [" + this + "] Error while doAction: ", e);
                }
                error = e;
            } finally {
                if (log.isDebugEnabled()) {
                    log.debug("Task [" + this + "] done");
                }
            }
        }
        return null;
    }

    @Override
    protected void done() {
        super.done();

        if (log.isDebugEnabled()) {
            log.debug("Task [" + this + "] execute done method after all");
        }

        if (error == null) {

            // success hook
            action.postSuccessAction();
        } else {

            // fail hook
            action.postFailedAction(error);
        }

        action.releaseAction();

        if (timer != null) {

            timer.cancel();
        }

        actionUI.close();

        updateBusyState(false);
    }

    protected void updateBusyState(boolean busy) {

        boolean hideBody = action.isHideBody();

        action.getContext().setBusy(busy);

        if (hideBody) {
            action.getContext().setHideBody(!busy);
        }

    }

    protected class TuttiActionTimerTask extends TimerTask {

        public TuttiActionTimerTask() {
            action.addPropertyChangeListener(AbstractTuttiAction.PROPERTY_DONE, new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    if (actionUI.isVisible()) {
                        actionUI.close();
                    }

                    // we do NOT want the timer to wake up then the actionUI dialog
                    cancel();
                }
            });
        }

        @Override
        public void run() {
            if (isCancelled() || isDone()) {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] was already canceled or done, do nothing");
                }
            } else {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] is started, show waiting dialog");
                }

                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        actionUI.open(action);
                    }
                });
            }
        }
    }
}
