package fr.ifremer.tutti.ui.swing.util.editor;

/*
 * #%L
 * Tutti :: UI
 * $Id: SimpleTimeEditorHandler.java 557 2013-03-07 09:06:54Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/editor/SimpleTimeEditorHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class SimpleTimeEditorHandler {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SimpleTimeEditorHandler.class);

    private final SimpleTimeEditor editor;

    private final SimpleTimeEditorModel model;

    /** the mutator method on the property of boxed bean in the editor */
    protected Method mutator;

    protected final Calendar calendarDate;

    protected final Calendar calendarMinute;

    protected final Calendar calendarHour;

    public SimpleTimeEditorHandler(SimpleTimeEditor editor) {
        this.editor = editor;
        this.model = editor.getModel();
        this.calendarMinute = Calendar.getInstance();
        this.calendarHour = Calendar.getInstance();
        this.calendarDate = Calendar.getInstance();
    }

    public void init() {

        if (model.getBean() == null) {
            throw new NullPointerException("can not have a null bean in ui " + editor);
        }

        editor.getMinute().setEditor(new JSpinner.DateEditor(editor.getMinute(), "mm"));
//        editor.getHour().setEditor(new JSpinner.DateEditor(editor.getHour(), "HH"));


//        TuttiUIUtil.autoSelectOnFocus(minuteEditor.getTextField());
//        JSpinner.NumberEditor hourEditor = (JSpinner.NumberEditor) editor.getHour().getEditor();
//        TuttiUIUtil.autoSelectOnFocus(hourEditor.getTextField());

        // listen when date changes (should come from outside)
        model.addPropertyChangeListener(SimpleTimeEditorModel.PROPERTY_DATE, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Date date = (Date) evt.getNewValue();

                if (date != null) {
                    calendarDate.setTime(date);
                    int hours = calendarDate.get(Calendar.HOUR_OF_DAY);
                    int minutes = calendarDate.get(Calendar.MINUTE);
                    if (log.isDebugEnabled()) {
                        log.debug("date changed : new value " + hours + ":" + minutes);
                    }
                    model.setTimeModel(hours * 60 + minutes);

                } else {
                    model.setTimeModel(null);
                }

            }
        });

        // When time model change, let's push it back in bean
        model.addPropertyChangeListener(SimpleTimeEditorModel.PROPERTY_TIME_MODEL, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Integer time = (Integer) evt.getNewValue();

                int hours = time / 60;
                int minutes = time % 60;

                calendarDate.set(Calendar.HOUR_OF_DAY, hours);
                calendarDate.set(Calendar.MINUTE, minutes);

                // push it back into the bean

                Date newValue = calendarDate.getTime();

                if (log.isDebugEnabled()) {
                    log.debug(model.getProperty() + " on " + model.getBean().getClass() + " :: " + newValue);
                }

                try {
                    getMutator().invoke(model.getBean(), newValue);

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        });
    }

    public SimpleTimeEditor getEditor() {
        return editor;
    }

    protected Date setMinuteModel(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
            calendarMinute.set(Calendar.MINUTE, 0);
        } else {
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
        }
        incomingDate = calendarMinute.getTime();
        return incomingDate;
    }

    public void updateTimeModelFromHour(Integer hour) {
        model.setTimeModel(hour * 60 + model.getMinute());
    }

    public void updateTimeModelFromMinute(Date minuteDate) {

        calendarMinute.setTime(minuteDate);
        int newHour = calendarMinute.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarMinute.get(Calendar.MINUTE);

        int oldHour = model.getHour();
        int oldMinute = model.getMinute();

        if (oldHour == newHour && oldMinute == newMinute) {

            // do nothing, same data
            if (log.isDebugEnabled()) {
                log.debug("Do not update time model , stay on same time = " + oldHour + ":" + oldMinute);
            }
            return;
        }

        // by default stay on same hour
        int hour = oldHour;

        // by default, use the new minute data
        int minute = newMinute;

        if (log.isDebugEnabled()) {
            log.debug("hh:mm (old from dateModel)   = " + oldHour + ":" + oldMinute);
            log.debug("hh:mm (new from minuteModel) = " + newHour + ":" + newMinute);
        }

        SpinnerDateModel minuteModel = editor.getMinuteModel();

        if (newMinute == 0) {

            // minute pass to zero (check if a new hour is required)
            if (newHour == 1) {

                if (oldHour == 23) {

                    // can't pass from 23:59 to 0:00, stay on 23:59
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    minuteModel.setValue(minuteModel.getPreviousValue());
                    return;
                }
                hour = (oldHour + 1) % 24;
            }
        } else if (newMinute == 59) {

            // minute pass to 59 (check if a new hour is required)

            if (newHour == 23) {

                if (oldHour == 0) {

                    // can't pass from 0:00 to 23:59, stay on 0:00
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    minuteModel.setValue(minuteModel.getNextValue());
                    return;
                }

                // decrease hour
                hour = (oldHour - 1) % 24;
            }
        }

        // date has changed
        if (log.isDebugEnabled()) {
            log.debug("Update time model to hh:mm = " + hour + ":" + minute);
        }
        model.setTimeModel(hour * 60 + minute);
    }

    protected void setDate(Date oldValue, Date newValue) {
        if (model.getBean() != null) {

            if (log.isDebugEnabled()) {
                log.debug(model.getProperty() + " on " + model.getBean().getClass() + " :: " + oldValue + " to " + newValue);
            }

            try {
                getMutator().invoke(model.getBean(), newValue);

            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    protected Method getMutator() {
        if (mutator == null) {
            Object bean = model.getBean();
            if (bean == null) {
                throw new NullPointerException("could not find bean in " + editor);
            }
            String property = model.getProperty();
            if (property == null) {
                throw new NullPointerException("could not find property in " + editor);
            }

            try {
                PropertyDescriptor descriptor = PropertyUtils.getPropertyDescriptor(bean, property);
                mutator = descriptor.getWriteMethod();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return mutator;
    }
}
