package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: ValidateCruiseUIHandler.java 827 2013-04-18 19:11:54Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/ValidateCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.CloseableUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.validator.NuitonValidatorResult;
import org.nuiton.validator.NuitonValidatorScope;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Map;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class ValidateCruiseUIHandler extends AbstractTuttiUIHandler<ValidateCruiseUIModel, ValidateCruiseUI> implements CloseableUI {

    private final static Log log = LogFactory.getLog(ValidateCruiseUIHandler.class);

    /** Persistence service. */
    private final PersistenceService persistenceService;

    /** Validation service. */
    private final ValidationService validationService;

    protected EditCruiseFishingOperationAction editFishingOperationAction;

    protected Map<FishingOperation, NuitonValidatorResult> validator;

    public ValidateCruiseUIHandler(TuttiUI parentUi, ValidateCruiseUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        persistenceService = context.getPersistenceService();
        validationService = context.getValidationService();
    }

    @Override
    public void beforeInitUI() {
        validator = context.getValidateCruiseOperationsService().validateCruiseOperations();

        ValidateCruiseUIModel model = new ValidateCruiseUIModel();
        ui.setContextValue(model);

    }

    @Override
    public void afterInitUI() {
        initUI(ui);

        editFishingOperationAction = TuttiActionHelper.createLogicAction(this, EditCruiseFishingOperationAction.class);

        ui.getOperationPanel().getModel().setSelectedFishingOperation(null);

        DefaultMutableTreeNode root = new DefaultMutableTreeNode();
        for (FishingOperation operation : validator.keySet()) {
            operation = persistenceService.getFishingOperation(operation.getId());
            NuitonValidatorResult result = validator.get(operation);

            Multimap<NuitonValidatorScope, String> messages = LinkedListMultimap.create();
            messages.putAll(NuitonValidatorScope.ERROR, result.getMessagesForScope(NuitonValidatorScope.ERROR));
            messages.putAll(NuitonValidatorScope.WARNING, result.getMessagesForScope(NuitonValidatorScope.WARNING));

            OperationTreeNode node = new OperationTreeNode(operation, messages);
            root.add(node);
        }
        TreeModel model = new DefaultTreeModel(root);

        final JTree navigation = ui.getNavigation();
        navigation.setModel(model);
        navigation.setCellRenderer(new ValidationTreeCellRenderer());
        navigation.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    Object[] paths = navigation.getSelectionPath().getPath();
                    FishingOperation operation = null;
                    for (Object o : paths) {
                        if (o != null && OperationTreeNode.class.isAssignableFrom(o.getClass())) {
                            operation = (FishingOperation) ((OperationTreeNode) o).getUserObject();
                            break;
                        }
                    }
                    editFishingOperationAction.setFishingOperation(operation);
                    AbstractTuttiAction.runAction(editFishingOperationAction);
                }
            }
        });

        FishingOperationsUI operationPanel = ui.getOperationPanel();
        operationPanel.getTopPanel().setVisible(false);
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
    }

    @Override
    public SwingValidator<ValidateCruiseUIModel> getValidator() {
        return null;
    }

    @Override
    protected JComponent getComponentToFocus() {
        return null;
    }

    @Override
    public boolean quitUI() {
        return true;
    }

    protected class ValidationTreeCellRenderer extends DefaultTreeCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            JLabel label = (JLabel) super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);

            if (value != null) {
                if (value instanceof OperationTreeNode) {
                    OperationTreeNode node = (OperationTreeNode) value;
                    label.setText(getDecorator(FishingOperation.class, null).toString(node.getUserObject()));

                } else if (value instanceof MessageTreeNode) {
                    MessageTreeNode node = (MessageTreeNode) value;
                    label.setText(_(String.valueOf(node.getUserObject())));
                    String iconName = node.getScope().toString().toLowerCase();
                    label.setIcon(SwingUtil.createImageIcon(iconName + ".png"));
                }
            }
            return label;
        }
    }

    protected class OperationTreeNode extends DefaultMutableTreeNode {

        private static final long serialVersionUID = 1L;

        public OperationTreeNode(FishingOperation operation,
                                 Multimap<NuitonValidatorScope, String> messages) {
            super(operation);
            setAllowsChildren(messages != null);
            createChildren(messages);
        }

        public void createChildren(Multimap<NuitonValidatorScope, String> messages) {
            for (NuitonValidatorScope scope : messages.keySet()) {
                for (String message : messages.get(scope)) {
                    MessageTreeNode child = new MessageTreeNode(scope, message);
                    this.add(child);
                }
            }
        }
    }

    protected class MessageTreeNode extends DefaultMutableTreeNode {

        private static final long serialVersionUID = 1L;

        protected NuitonValidatorScope scope;

        public MessageTreeNode(NuitonValidatorScope scope, String message) {
            super(message);
            this.scope = scope;
        }

        public NuitonValidatorScope getScope() {
            return scope;
        }

    }
}
