package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIUtil.java 432 2013-02-15 20:28:55Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/TuttiUIUtil.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.TuttiTechnicalException;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.nuiton.util.FileUtil;

import javax.swing.JOptionPane;
import javax.swing.UIManager;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.io.File;
import java.util.Date;

import static org.nuiton.i18n.I18n._;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    private TuttiUIUtil() {
        // never instanciate util class
    }

    private static final JAXXContextEntryDef<Program> PROGRAM_ENTRY =
            JAXXUtil.newContextEntryDef("loadedProgram", Program.class);

    private static final JAXXContextEntryDef<Cruise> CRUISE_ENTRY =
            JAXXUtil.newContextEntryDef("loadedCruise", Cruise.class);

    private static final JAXXContextEntryDef<TuttiProtocol> PROTOCOL_ENTRY =
            JAXXUtil.newContextEntryDef("loadedProtocol", TuttiProtocol.class);

//    public static Program getProgram(JAXXContext context) {
//        return PROGRAM_ENTRY.getContextValue(context);
//    }

//    public static Cruise getCruise(JAXXContext context) {
//        return CRUISE_ENTRY.getContextValue(context);
//    }

//    public static TuttiProtocol getProtocol(JAXXContext context) {
//        return PROTOCOL_ENTRY.getContextValue(context);
//    }

    public static Program getProgram(TuttiUIContext context) {
        return PROGRAM_ENTRY.getContextValue(context.getMainUI());
    }

    public static Cruise getCruise(TuttiUIContext context) {
        return CRUISE_ENTRY.getContextValue(context.getMainUI());
    }

    public static TuttiProtocol getProtocol(TuttiUIContext context) {
        return PROTOCOL_ENTRY.getContextValue(context.getMainUI());
    }

    public static void setProgram(TuttiUIContext context, Program value) {
        if (value == null) {

            PROGRAM_ENTRY.removeContextValue(context.getMainUI());
        } else {

            PROGRAM_ENTRY.setContextValue(context.getMainUI(), value);
        }
    }

    public static void setCruise(TuttiUIContext context, Cruise value) {
        if (value == null) {
            CRUISE_ENTRY.removeContextValue(context.getMainUI());
        } else {
            CRUISE_ENTRY.setContextValue(context.getMainUI(), value);
        }
    }

    public static void setProtocol(TuttiUIContext context, TuttiProtocol value) {
        if (value == null) {

            PROTOCOL_ENTRY.removeContextValue(context.getMainUI());
        } else {
            PROTOCOL_ENTRY.setContextValue(context.getMainUI(), value);
        }
    }

    //    public static JDialog openInDialog(JComponent ui,
//                                       Frame frame,
//                                       String title,
//                                       boolean modal,
//                                       Action closeAction) {
//        return openInDialog(ui, frame, title, modal, closeAction, null);
//    }
//
//    public static JDialog openInDialog(JComponent ui,
//                                       Frame frame,
//                                       String title,
//                                       boolean modal,
//                                       Action closeAction,
//                                       Dimension dim) {
//
//        JDialog result = new JDialog(frame, modal);
//        result.setTitle(title);
//        result.add(ui);
//        result.setResizable(true);
//        if (dim == null) {
//            result.setSize(550, 450);
//        } else {
//            result.setSize(dim);
//        }
//        if (closeAction != null) {
//            JRootPane rootPane = result.getRootPane();
//            rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke("ESCAPE"), "close");
//            rootPane.getActionMap().put("close", closeAction);
//        }
//        result.addWindowListener(new WindowAdapter() {
//
//            @Override
//            public void windowClosed(WindowEvent e) {
//                Component ui = (Component) e.getSource();
//                if (log.isInfoEnabled()) {
//                    log.info("Destroy ui " + ui);
//                }
//                JAXXUtil.destroy(ui);
//            }
//        });
//        SwingUtil.center(frame, result);
//        result.setVisible(true);
//        return result;
//    }

    /**
     * Choisir un fichier via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param incoming    le fichier de base à utilier
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou le fichier incoming si l'opération a été
     *         annulée
     */
    public static File chooseFile(Component parent,
                                  String title,
                                  String buttonLabel,
                                  File incoming,
                                  String... filters) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        File file = FileUtil.getFile(title, buttonLabel, parent, filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        File result = file == null ? incoming : file;
        return result;
    }

    /**
     * Choisir un répertoire via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre de la boite de dialogue de sléection
     * @param buttonLabel le label de l'action d'acceptation
     * @param incoming    le fichier de base à utiliser
     * @return le répertoire choisi ou le répertoire incoming si l'opération a
     *         été annulée
     */
    public static File chooseDirectory(Component parent,
                                       String title,
                                       String buttonLabel,
                                       File incoming) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        String file = FileUtil.getDirectory(parent, title, buttonLabel);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        return file == null ? incoming : new File(file);
    }

    public static void updateBusyState(Component ui, boolean busy) {
        if (busy) {
            // ui bloquee
            if (log.isDebugEnabled()) {
                log.debug("block ui in busy mode");
            }
            ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        } else {
            // ui debloquee
            if (log.isDebugEnabled()) {
                log.debug("unblock ui in none busy mode");
            }
            ui.setCursor(Cursor.getDefaultCursor());
        }
    }

    public static void showSuccessMessage(Component ui, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                "Success",
                JOptionPane.INFORMATION_MESSAGE,
                UIManager.getIcon("info")
        );
    }

    public static void showErrorMessage(Component ui, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                "Error",
                JOptionPane.ERROR_MESSAGE,
                UIManager.getIcon("error")
        );
    }

    public static boolean askQuestion(Component ui, String message) {
        int i = JOptionPane.showConfirmDialog(
                ui,
                message,
                "Question?",
                JOptionPane.YES_NO_OPTION);
        return i == JOptionPane.YES_OPTION;
    }

    public static void setProperty(Object bean, String property, Object value) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            PropertyUtils.setSimpleProperty(bean, property, value);
        } catch (Exception e) {
            throw new TuttiTechnicalException("Could not set property '" + property + "' on object of type '" + bean.getClass().getName() + "'", e);
        }
    }

    public static Object getProperty(Object bean, String property) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            return PropertyUtils.getSimpleProperty(bean, property);
        } catch (Exception e) {
            throw new TuttiTechnicalException("Could not get property '" + property + "' on object of type '" + bean.getClass().getName() + "'", e);
        }
    }

    public static Highlighter newBackgroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, false);
    }

    public static Highlighter newForegroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, true);
    }

    public static String getDuration(Date startDate, Date endDate) {
        String duration = "";
        if (startDate != null && endDate != null) {
            duration = DurationFormatUtils.formatPeriod(
                    startDate.getTime(),
                    endDate.getTime(),
                    _("tutti.duration.format"));
        }
        return duration;
    }

    public static int computeDistanceInMeters(Float startLatitude,
                                              Float startLongitude,
                                              Float endLatitude,
                                              Float endLongitude) {

        double sLat = startLatitude * Math.PI / 180;
        double sLong = startLongitude * Math.PI / 180;
        double eLat = endLatitude * Math.PI / 180;
        double eLong = endLongitude * Math.PI / 180;

        Double d = 6378288 *
                   (Math.PI / 2 - Math.asin(Math.sin(eLat) * Math.sin(sLat)
                                            + Math.cos(eLong - sLong) * Math.cos(eLat) * Math.cos(sLat)));
        return d.intValue();

    }
}
