
package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: SaveFishingOperationAction.java 512 2013-03-01 20:37:07Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/SaveFishingOperationAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.CaracteristicTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.environment.EnvironmentTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.gearshooting.GearShootingTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.hydrology.HydrologyTabUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.editor.CoordinateEditorType;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import static org.nuiton.i18n.I18n._;

/**
 * Saves a fishing operation and potentially sets another screen or switch to another tab.
 *
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class SaveFishingOperationAction extends AbstractTuttiAction<EditFishingOperationUIModel, EditFishingOperationUI, EditFishingOperationUIHandler> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SaveFishingOperationAction.class);


    /**
     * A flag to update ui after create or save the edit fishing operation.
     *
     * @since 1.0
     */
    protected boolean updateUI;

    public SaveFishingOperationAction(EditFishingOperationUIHandler handler) {
        super(handler,
              "save",
              _("tutti.action.saveFishingOperation"),
              _("tutti.action.saveFishingOperation.tip"),
              true
        );
    }

    public void setUpdateUI(boolean updateUI) {
        this.updateUI = updateUI;
    }

    @Override
    protected void releaseAction() {
        updateUI = true;
        super.releaseAction();
    }

    /**
     * If the event source is an EditFishingOperationUIHandler,
     * then the tab panel of the parent switch to the index set
     * in the ID param of the event.
     * If the event source is a TuttiScreen, then the screen changes to the source.
     */
    @Override
    protected void doAction() {

        EditFishingOperationUIHandler handler = getHandler();

        TuttiBeanMonitor<EditFishingOperationUIModel> monitor =
                handler.getFishingOperationMonitor();

        // previous fishingOperation was modified, let's save it
        EditFishingOperationUIModel beanToSave = monitor.getBean();

        // must save when bean is new or was modifiy and is valid
        boolean mustSave = (beanToSave.isCreate() || !beanToSave.isEmpty()) &&
                           beanToSave.isValid();

        if (mustSave) {

            // prepare model

            if (beanToSave.getCoordinateEditorType() != CoordinateEditorType.DD) {
                beanToSave.convertGearShootingCoordinatesDMSToDD();
            }

            // save modified fishing operation
            FishingOperation toSave = beanToSave.toBean();

            CaracteristicTabUIModel[] subModels = handler.getSubModels();
            for (CaracteristicTabUIModel subModel : subModels) {
                Class<?> modelClass = subModel.getClass();
                CaracteristicMap caracteristics = subModel.getCaracteristicMap();

                if (modelClass.isAssignableFrom(EnvironmentTabUIModel.class)) {
                    toSave.setEnvironmentCaracteristics(caracteristics);

                } else if (modelClass.isAssignableFrom(GearShootingTabUIModel.class)) {
                    toSave.setGearShootingCaracteristics(caracteristics);

                } else if (modelClass.isAssignableFrom(HydrologyTabUIModel.class)) {
                    toSave.setHydrologyCaracteristics(caracteristics);
                }
                subModel.setModify(false);
            }

            sendMessage(
                    "[ Trait - Caractéristiques générales ] " +
                    "Sauvegarde des modifications de " + decorate(toSave) +
                    ".");

            if (log.isInfoEnabled()) {
                log.info("FishingOperation " + toSave.getId() +
                         " was modified, will save it.");
            }

            monitor.clearModified();
            getModel().setModify(false);

            saveFishingOperation(toSave);

            handler.getParentUi().getTabPane().setSelectedIndex(1);
        }

    }

    protected void saveFishingOperation(FishingOperation toSave) {

        PersistenceService service = getContext().getPersistenceService();

        boolean create = TuttiEntities.isNew(toSave);

        FishingOperationsUIModel model =
                getHandler().getParentUi().getModel();
        FishingOperation savedFishingOperation;

        if (create) {

            // create fishing operation
            savedFishingOperation = service.createFishingOperation(toSave);

            // create then the CatchBatch
            CatchBatch catchBatch = new CatchBatch();
            catchBatch.setFishingOperation(savedFishingOperation);
            service.createCatchBatch(catchBatch);

            // add new created fishing operation to list
            model.addFishingOperation(savedFishingOperation);

            // select it (will reload editing fishing operation)
            model.setSelectedFishingOperation(savedFishingOperation);

        } else {

            model.setEditionAdjusting(true);

            try {
                // save fishing operation
                savedFishingOperation = service.saveFishingOperation(toSave);

                // reinject it in model
                model.updateFishingOperation(savedFishingOperation);
            } finally {
                model.setEditionAdjusting(false);
            }
        }
    }

}
