package fr.ifremer.tutti.ui.swing.content.home;

/*
 * #%L
 * Tutti :: UI
 * $Id: SelectCruiseUIHandler.java 512 2013-03-01 20:37:07Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/home/SelectCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.Action;
import javax.swing.JComboBox;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Main ui content to select cruise.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SelectCruiseUIHandler extends AbstractTuttiUIHandler<SelectCruiseUIModel, SelectCruiseUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SelectCruiseUIHandler.class);

    protected final PersistenceService persistenceService;

    public SelectCruiseUIHandler(TuttiUI<?, ?> parentUI, SelectCruiseUI ui) {
        super(parentUI.getHandler().getContext(), ui);
        persistenceService = context.getPersistenceService();
    }

    @Override
    public SwingValidator<SelectCruiseUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {
        SelectCruiseUIModel model = new SelectCruiseUIModel();

        List<Program> programs = persistenceService.getAllProgram();
        model.setPrograms(programs);

        if (programs.isEmpty()) {
            // do nothing

            if (log.isDebugEnabled()) {
                log.debug("No program found.");
            }
        } else {

            // get selected program (if any)

            Program selectedProgram = null;
            List<Cruise> cruises = null;
            if (context.isProgramFilled()) {
                selectedProgram = getDataContext().getProgram();
                cruises = persistenceService.getAllCruise(selectedProgram.getId());
            }

            model.setProgram(selectedProgram);
            model.setCruises(cruises);

            if (CollectionUtils.isEmpty(cruises)) {

                // nothing to select
            } else {

                Cruise selectedCruise = null;
                if (context.isCruiseFilled()) {
                    selectedCruise = getDataContext().getCruise();
                }
                model.setCruise(selectedCruise);
            }
        }

        List<TuttiProtocol> protocols = persistenceService.getAllProtocol();
        model.setProtocols(protocols);

        if (protocols.isEmpty()) {
            // do nothing

            if (log.isDebugEnabled()) {
                log.debug("No protocol found.");
            }
        } else {

            TuttiProtocol selectedProtocol = null;
            if (context.isProgramFilled()) {
                selectedProtocol = getDataContext().getProtocol();
            }
            model.setProtocol(selectedProtocol);
        }
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        SelectCruiseUIModel model = getModel();

        initBeanComboBox(ui.getProgramCombobox(),
                         model.getPrograms(),
                         model.getProgram());

        initBeanComboBox(ui.getCruiseCombobox(),
                         model.getCruises(),
                         model.getCruise());

        initBeanComboBox(ui.getProtocolCombobox(),
                         model.getProtocols(),
                         model.getProtocol());

        ui.getEditProtocolComboBox().putClientProperty("JComboBox.isTableCellEditor", Boolean.TRUE);
        ui.getNewProtocolComboBox().putClientProperty("JComboBox.isTableCellEditor", Boolean.TRUE);

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_PROTOCOLS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // reload combo box
                BeanComboBox<TuttiProtocol> combobox = ui.getProtocolCombobox();
                List<TuttiProtocol> protocols = (List<TuttiProtocol>) evt.getNewValue();

                combobox.setData(null);
                combobox.setData(protocols);

                ui.applyDataBinding(SelectCruiseUI.BINDING_PROTOCOL_COMBOBOX_ENABLED);
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_PROGRAM, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Program newValue = (Program) evt.getNewValue();
                boolean noProgram = newValue == null;
                context.setProgramId(noProgram ? null : newValue.getId());
                List<Cruise> campaigns;
                if (noProgram) {
                    campaigns = Lists.newArrayList();
                } else {
                    campaigns = persistenceService.getAllCruise(newValue.getId());
                }
                SelectCruiseUIModel source = (SelectCruiseUIModel) evt.getSource();
                source.setCruises(campaigns);
                source.setCruise(null);
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_CRUISES, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // reload combo box
                BeanComboBox<Cruise> combobox = ui.getCruiseCombobox();
                List<Cruise> campaigns = (List<Cruise>) evt.getNewValue();
                combobox.setData(null);
                combobox.setData(campaigns);
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_CRUISE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Cruise newValue = (Cruise) evt.getNewValue();
                context.setCruiseId(newValue == null ? null : newValue.getId());
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_PROTOCOL, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                TuttiProtocol newValue = (TuttiProtocol) evt.getNewValue();
                context.setProtocolId(newValue == null ? null : newValue.getId());
            }
        });

        registerValidators(getValidator());

        listenValidatorValid(getValidator(), model);

        getValidator().setBean(model);

        ui.applyDataBinding(SelectCruiseUI.BINDING_EDIT_PROGRAM_BUTTON_ENABLED);
        ui.applyDataBinding(SelectCruiseUI.BINDING_EDIT_CRUISE_BUTTON_ENABLED);
        ui.applyDataBinding(SelectCruiseUI.BINDING_EDIT_CATCHES_BUTTON_ENABLED);
        ui.applyDataBinding(SelectCruiseUI.BINDING_VALIDATE_CATCHES_BUTTON_ENABLED);
    }

    protected boolean resetEditProtolAction;

    public void resetEditProtolAction() {
        resetEditProtolAction = true;
        try {
            ui.getEditProtocolComboBox().setSelectedIndex(0);
        } finally {
            resetEditProtolAction = false;
        }
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
        PropertyChangeListener[] listeners = getModel().getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            getModel().removePropertyChangeListener(listener);
        }
        clearValidators();
    }

    public void startExistingProtocolAction(MouseEvent event) {
        JComboBox existingProtocolCombo = (JComboBox) event.getSource();
        Action selectedAction = (Action) existingProtocolCombo.getItemAt(0);
        selectedAction.actionPerformed(null);
    }

    public void startExistingProtocolAction(ActionEvent event) {
        if (!resetEditProtolAction) {
            JComboBox existingProtocolCombo = (JComboBox) event.getSource();
            Action selectedAction = (Action) existingProtocolCombo.getSelectedItem();
            // hide popup before performing the action, otherwise, if the action
            // opens a popup, the user must click a first time to hide the combobox
            // popup to then interact with the popup opened by the action
            // (see http://forge.codelutin.com/issues/2032)
            existingProtocolCombo.setSelectedIndex(0);
            existingProtocolCombo.hidePopup();
            selectedAction.actionPerformed(event);
        }
    }

    public void startNewProtocolAction(MouseEvent event) {
        JComboBox newProtocolCombo = (JComboBox) event.getSource();
        Action selectedAction = (Action) newProtocolCombo.getItemAt(0);
        selectedAction.actionPerformed(null);
    }

    public void startNewProtocolAction(ActionEvent event) {
        JComboBox newProtocolCombo = (JComboBox) event.getSource();
        Action selectedAction = (Action) newProtocolCombo.getSelectedItem();
        // hide popup before performing the action, otherwise, if the action
        // opens a popup, the user must click a first time to hide the combobox
        // popup to then interact with the popup opened by the action
        // (see http://forge.codelutin.com/issues/2032)
        newProtocolCombo.setSelectedIndex(0);
        newProtocolCombo.hidePopup();
        selectedAction.actionPerformed(event);
    }
}
