package fr.ifremer.tutti.ui.swing.content.operation.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchRowModel.java 77 2012-12-15 16:05:45Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/species/SpeciesBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.Sex;
import fr.ifremer.tutti.persistence.entities.referential.SortedUnsortedCategory;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.persistence.entities.referential.WeightCategory;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.Collections;
import java.util.List;

/**
 * Represents a species batch (i.e a row in the batch table).
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesBatchRowModel extends AbstractTuttiBeanUIModel<SpeciesBatch, SpeciesBatchRowModel> {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SPECIES = "species";

    public static final String PROPERTY_SPECIES_TO_CONFIRM = "speciesToConfirm";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY = "sortedUnsortedCategory";

    public static final String PROPERTY_WEIGHT_CATEGORY = "weightCategory";

    public static final String PROPERTY_SEX = "sex";

    public static final String PROPERTY_MATURITY = "maturity";

    public static final String PROPERTY_AGE = "age";

    public static final String PROPERTY_WEIGHT = "weight";

    public static final String PROPERTY_SAMPLE_WEIGHT = "sampleWeight";

    public static final String PROPERTY_SAMPLING_RATIO = "samplingRatio";

    public static final String PROPERTY_COMMENT = "comment";

    public static final String PROPERTY_FREQUENCY = "frequency";

    public static final String PROPERTY_COMPUTED_NUMBER = "computedNumber";

    public static final String PROPERTY_COMPUTED_WEIGHT = "computedWeight";
    
    public static final String PROPERTY_ATTACHMENTS = "attachments";

    /**
     * Species observed.
     *
     * @since 0.2
     */
    protected Species species;

    /**
     * Is the species need to be confirmed?.
     *
     * @since 0.2
     */
    protected boolean speciesToConfirm;

    /**
     * Is catch is vrac or  horsVrac?.
     *
     * @since 0.2
     */
    protected SortedUnsortedCategory sortedUnsortedCategory;

    /**
     * Weight category (can be null).
     *
     * @since 0.2
     */
    protected WeightCategory weightCategory;

    /**
     * Sex (can be null).
     *
     * @since 0.2
     */
    protected Sex sex;

    /**
     * Maturity (can be null).
     *
     * @since 0.2
     */
    protected Float maturity;

    /**
     * Age (can be null).
     *
     * @since 0.2
     */
    protected Float age;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected Float weight;

    /**
     * Total computed weight (from frequencies).
     *
     * @since 0.2
     */
    protected Float computedWeight;

    /**
     * Total computed number (from frequencies).
     *
     * @since 0.2
     */
    protected Float computedNumber;

    /**
     * Sample weight.
     *
     * @since 0.2
     */
    protected Float sampleWeight;

    /**
     * Sampling ratio.
     *
     * @since 0.2
     */
    protected Float samplingRatio;

    /**
     * Comment on this catches.
     *
     * @since 0.2
     */
    protected String comment;
    
    /**
     * Attachments
     * 
     * @since 0.2
     */
    protected List<Attachment> attachments;

    /**
     * List of frequencies observed for this batch.
     *
     * @since 0.2
     */
    protected List<SpeciesFrequencyRowModel> frequency;

    protected static final Binder<SpeciesBatch, SpeciesBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(SpeciesBatch.class,
                                    SpeciesBatchRowModel.class);

    protected static final Binder<SpeciesBatchRowModel, SpeciesBatch> toBeanBinder =
            BinderFactory.newBinder(SpeciesBatchRowModel.class,
                                    SpeciesBatch.class);

    public SpeciesBatchRowModel() {
        super(SpeciesBatch.class, fromBeanBinder, toBeanBinder);
        frequency = Lists.newArrayList();
    }

    public SpeciesBatchRowModel(SpeciesBatch aBatch,
                                List<SpeciesBatchFrequency> frequencies) {
        this();
        fromBean(aBatch);
        List<SpeciesFrequencyRowModel> frequencyRows =
                SpeciesFrequencyRowModel.fromBeans(frequencies);
        frequency.addAll(frequencyRows);
        Collections.sort(frequency);
        updateTotalFromFrequencies();
    }

    public void updateTotalFromFrequencies() {
        Float totalNumber = null;
        Float totalWeight = null;
        if (CollectionUtils.isNotEmpty(frequency)) {
            totalNumber = 0f;
            totalWeight = 0f;
            for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                totalNumber += frequencyModel.getNumber();
                Float w = frequencyModel.getWeight();
                if (w == null) {

                    // can't sum when a null value appears
                    totalWeight = null;
                } else if (totalWeight != null) {

                    // still can sum weights
                    totalWeight += w;
                }
            }
        }

        setComputedNumber(totalNumber);
        setComputedWeight(totalWeight);
    }

    public Species getSpecies() {
        return species;
    }

    public void setSpecies(Species species) {
        Object oldValue = getSpecies();
        this.species = species;
        firePropertyChange(PROPERTY_SPECIES, oldValue, species);
    }

    public Boolean getSpeciesToConfirm() {
        return speciesToConfirm;
    }

    public void setSpeciesToConfirm(Boolean speciesToConfirm) {
        Object oldValue = getSpeciesToConfirm();
        this.speciesToConfirm = speciesToConfirm;
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    public SortedUnsortedCategory getSortedUnsortedCategory() {
        return sortedUnsortedCategory;
    }

    public void setSortedUnsortedCategory(SortedUnsortedCategory sortedUnsortedCategory) {
        Object oldValue = getSortedUnsortedCategory();
        this.sortedUnsortedCategory = sortedUnsortedCategory;
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY, oldValue, sortedUnsortedCategory);
    }

    public WeightCategory getWeightCategory() {
        return weightCategory;
    }

    public void setWeightCategory(WeightCategory weightCategory) {
        Object oldValue = getWeightCategory();
        this.weightCategory = weightCategory;
        firePropertyChange(PROPERTY_WEIGHT_CATEGORY, oldValue, weightCategory);
    }

    public Sex getSex() {
        return sex;
    }

    public void setSex(Sex sex) {
        Object oldValue = getSex();
        this.sex = sex;
        firePropertyChange(PROPERTY_SEX, oldValue, sex);
    }

    public Float getMaturity() {
        return maturity;
    }

    public void setMaturity(Float maturity) {
        Object oldValue = getMaturity();
        this.maturity = maturity;
        firePropertyChange(PROPERTY_MATURITY, oldValue, maturity);
    }

    public Float getAge() {
        return age;
    }

    public void setAge(Float age) {
        Object oldValue = getAge();
        this.age = age;
        firePropertyChange(PROPERTY_AGE, oldValue, age);
    }

    public Float getWeight() {
        return weight;
    }

    public void setWeight(Float weight) {
        Object oldValue = getWeight();
        this.weight = weight;
        firePropertyChange(PROPERTY_WEIGHT, oldValue, weight);
    }

    public Float getSampleWeight() {
        return sampleWeight;
    }

    public void setSampleWeight(Float sampleWeight) {
        Object oldValue = getSampleWeight();
        this.sampleWeight = sampleWeight;
        firePropertyChange(PROPERTY_SAMPLE_WEIGHT, oldValue, sampleWeight);
    }

    public Float getSamplingRatio() {
        return samplingRatio;
    }

    public void setSamplingRatio(Float samplingRatio) {
        Object oldValue = getSamplingRatio();
        this.samplingRatio = samplingRatio;
        firePropertyChange(PROPERTY_SAMPLING_RATIO, oldValue, samplingRatio);
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        Object oldValue = getComment();
        this.comment = comment;
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    public List<Attachment> getAttachments() {
        return attachments;
    }

    public void setAttachments(List<Attachment> attachments) {
        Object oldValue = getAttachments();
        this.attachments = attachments;
        firePropertyChange(PROPERTY_ATTACHMENTS, oldValue, attachments);
    }
    
    public List<SpeciesFrequencyRowModel> getFrequency() {
        return frequency;
    }

    public void setFrequency(List<SpeciesFrequencyRowModel> frequency) {
        Object oldValue = getFrequency();
        this.frequency = frequency;
        firePropertyChange(PROPERTY_FREQUENCY, oldValue, frequency);
    }

    public Float getComputedWeight() {
        return computedWeight;
    }

    public void setComputedWeight(Float computedWeight) {
        Object oldValue = getComputedWeight();
        this.computedWeight = computedWeight;
        firePropertyChange(PROPERTY_COMPUTED_WEIGHT, oldValue,
                           computedWeight);
    }

    public Float getComputedNumber() {
        return computedNumber;
    }

    public void setComputedNumber(Float computedNumber) {
        Object oldValue = getComputedNumber();
        this.computedNumber = computedNumber;
        firePropertyChange(PROPERTY_COMPUTED_NUMBER, oldValue,
                           computedNumber);
    }

    public int getRowCount() {
        return frequency == null ? 0 : frequency.size();
    }
}
