/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: EmailHandler.java 1789 2010-11-16 15:18:03Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-swing/src/main/java/com/jurismarches/vradi/ui/email/EmailHandler.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.email;

import static com.jurismarches.vradi.VradiConstants.SessionStatus;
import static com.jurismarches.vradi.VradiContext.EMAIL_UI_ENTRY_DEF;
import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import java.awt.Component;
import java.awt.Cursor;
import java.lang.reflect.Constructor;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;

import com.jurismarches.vradi.beans.SendingHelper;
import com.jurismarches.vradi.services.VradiDataService;
import com.jurismarches.vradi.ui.email.helpers.SendingViewManager;
import javax.swing.*;
import javax.swing.tree.TreePath;

import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.ui.email.helpers.EmailDataHelper;
import com.jurismarches.vradi.ui.offer.OfferListHandler;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.ui.email.helpers.EmailBeanConstant;
import com.jurismarches.vradi.ui.email.helpers.EmailNavigationTreeHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.task.AbstractProgressBarUI;
import com.jurismarches.vradi.ui.task.VradiTask;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;

/**
 * Handler de l'UI de gestion des email.
 * 
 * @author morin
 */
public class EmailHandler {

    static private final Log log = LogFactory.getLog(EmailHandler.class);

    public EmailPopupUI init(JAXXContext rootContext) {
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(rootContext);

        if (ui == null) {
            JAXXInitialContext context = new JAXXInitialContext()
                    .add(rootContext).add(this);

            ui = new EmailPopupUI(context);

            // Register for save disposition
            UIHelper.registerComponentToSaveDispositionConfig(ui);

            EMAIL_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }
        return ui;
    }

    public EmailPopupUI getUI(JAXXContext context) {
        if (context instanceof EmailPopupUI) {
            return (EmailPopupUI) context;
        }
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(context);
        return ui;
    }

    public SessionViewUI getSessionsListUI(JAXXContext context) {
        if (context instanceof SessionViewUI) {
            return (SessionViewUI) context;
        }
        return context.getContextValue(SessionViewUI.class);
    }

    public SendingViewUI getSessionViewUI(JAXXContext context) {
        if (context instanceof SendingViewUI) {
            return (SendingViewUI) context;
        }
        return context.getContextValue(SendingViewUI.class);
    }

    public VradiStorageService getStorageService() {
        return VradiService.getVradiStorageService();
    }

    public VradiDataService getDataService() {
        return VradiService.getVradiDataService();
    }

    public WikittyProxy getProxy() {
        return VradiService.getWikittyProxy();
    }

    protected void close(JAXXContext context) {
        EmailPopupUI ui = getUI(context);
        UIHelper.saveComponentDispositionConfig();
        ui.setVisible(false);
        ui.dispose();
    }

    public List<Session> getSessions(Date date) {
        if (log.isDebugEnabled()){
            log.debug("getSessions for date : " + date);
        }
        List<Session> sessions = null;
        try {
            sessions = getDataService().getSessions(date);
        } catch (VradiException eee) {
            log.error(eee);
            ErrorDialogUI.showError(eee);
        }

        return sessions;
    }

    /**
     * Creer une nouvelle session.
     * @param ui email ui
     */
    public void createNewSession(EmailPopupUI ui) {

        try {
            Session session = getStorageService().createNewSession();
            EmailNavigationTreeHelper helper = ui.getHelper();
            VradiTreeNode node = helper.createSessionNode(session);
            helper.selectNode(node);
        }
        catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't create new session", ex);
            }
            JOptionPane.showMessageDialog(ui, _("Can't create new session for reason : %s", ex.getMessage()), _("Can't create new session"), JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Lien les formulaires non lié sur la session selectionnée.
     * Seulement pour une session active
     * 
     * @param ui ui
     */
    public void bindEmailsWithForms(final EmailPopupUI ui) {

        VradiTask<Void> task = new VradiTask<Void>(ui) {
            @Override
            public Void doAction() throws Exception {

                try {
                    Session selectedSession = ui.getHelper().getSelectedActiveSession();
                    if (!SessionStatus.isActive(selectedSession)) {

                        JOptionPane.showMessageDialog(ui,
                                _("vradi.error.session.bindNonActiveSession"),
                                _("vradi.error.session.bindNonActiveSession.title"),
                                JOptionPane.ERROR_MESSAGE);

                        return null;
                    }

                    getStorageService().bindForms(selectedSession);

                } catch (WikittyException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("An exception occured while binding form", eee);
                    }
                    JOptionPane.showMessageDialog(ui,
                            _("vradi.binding.forms.concurrence.message"));
                }
                return null;
            }
        };

        task.execute();
    }
    
    /**
     * Open UI depending on node type.
     *
     * @param context JaxxContext
     * @param contentLayout Layout of content
     * @param content in layout
     * @param node to display
     */
    protected void openUI(JAXXContext context, CardLayout2 contentLayout, JPanel content, VradiTreeNode node) {

        // Get node type
        Class<?> editType = node.getInternalClass();
        Class<? extends EmailContentUI> uiClass = EmailBeanConstant.getUIClass(editType);

        if (uiClass == null) {
            // show none ui
            contentLayout.show(content, EmailPopupUI.NONE);
            return;
        }

        // Verify if instance is existing
        EmailContentUI ui = UIHelper.getContentIfExist(contentLayout, content, uiClass);

        // Get layout identifier
        String contentName = editType.getName();

        if (ui == null) {
            try {
                // Get constructor
                Constructor<? extends EmailContentUI> constructor =
                        uiClass.getConstructor(JAXXContext.class);

                // Invoke instance creation
                ui = constructor.newInstance(context);

                // Open UI
                ui.openUI(node);
            } catch (Exception eee) {
                log.error(eee);
                ErrorDialogUI.showError(eee);
            }

            // Add to content panel
            content.add(ui, contentName);
        }

        // show ui
        contentLayout.show(content, contentName);
    }

    public void displaySession(JAXXContext context, EmailNavigationTreeHelper helper, VradiTreeNode node) {
        SendingViewUI ui = getSessionViewUI(context);
        
        // Get datas
        Sending sending = helper.getSendingToDisplay(node);

        // Get session
        Session session = helper.getSelectedActiveSession();

        // Open ui
        SendingViewManager manager = SendingViewManager.getInstance();
        context.setContextValue(manager);

        manager.openUI(ui, session, sending);
        if (log.isDebugEnabled()){
            log.debug("Sending displayed : " + sending);
        }
    }

    public boolean askToRemoveUserForm(JPanel ui, String userName, String formAsString) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmRemoveUserForms", formAsString, userName),
                _("vradi.email.confirmRemoveUserFormsTitle"),
                JOptionPane.YES_NO_OPTION);

        return yes == JOptionPane.YES_OPTION;
    }

    public boolean askToRemoveAllForm(JPanel ui, String formAsString) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmRemoveForms", formAsString),
                _("vradi.email.confirmRemoveFormsTitle"),
                JOptionPane.YES_NO_OPTION);

        return yes == JOptionPane.YES_OPTION;
    }

    public void revertDeletion(SendingViewUI ui, final Sending sending, final List<String> formsIdsToUpdate) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmRevertFormDeletion", EmailDataHelper.convertFormsIdsToString(formsIdsToUpdate)),
                _("vradi.email.confirmRevertFormDeletionTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION) {
            VradiTask<Void> revertSendingTask = new VradiTask<Void>(ui) {

                @Override
                public Void doAction() throws Exception {
                    VradiService.getVradiStorageService().revertDeletion(sending.getWikittyId(), formsIdsToUpdate);
                    return null;
                }
            };
            revertSendingTask.execute();
        }
    }

    public void revertUserDeletion(QueryMakerViewUI<?> ui, Session session, String formId, Object[] selectedValues) {
        if (askUserRevert(ui, EmailDataHelper.convertFormToString(formId), EmailDataHelper.convertObjectBeanToString(Arrays.asList(selectedValues), User.class))){
            for (Object o : selectedValues) {
                revertUserDeletion(ui, session, (User)o, formId, false);
            }
        }
    }
    
    public void revertUserDeletion(QueryMakerViewUI<?> ui, Session session, User user, String formToRevert) {
        revertUserDeletion(ui, session, user, formToRevert, true);
    }

    protected void revertUserDeletion(QueryMakerViewUI<?> ui, final Session session, final User user, final String formToRevert, boolean ask) {
        if (!ask || askUserRevert(ui, EmailDataHelper.convertFormToString(formToRevert), VradiHelper.getEntityName(user))){
            VradiTask<Void> revertSendingTask = new VradiTask<Void>(ui) {

                @Override
                public Void doAction() throws Exception {
                    VradiService.getVradiStorageService().revertUserDeletion(session.getWikittyId(), user.getWikittyId(), formToRevert);
                    return null;
                }
            };
            revertSendingTask.execute();
        }
    }

    public void revertGroupDeletion(QueryMakerViewUI<?> ui, Session session, String formId, Object[] selectedValues) {
        if (askGroupRevert(ui, EmailDataHelper.convertFormToString(formId), EmailDataHelper.convertObjectBeanToString(Arrays.asList(selectedValues), Group.class))){
            for (Object o : selectedValues) {
                revertGroupDeletion(ui, session, (Group)o, formId, false);
            }
        }
    }

    public void revertGroupDeletion(QueryMakerViewUI<?> ui, Session session, Group group, String formToRevert) {
        revertGroupDeletion(ui, session, group, formToRevert, true);
    }

    protected void revertGroupDeletion(QueryMakerViewUI<?> ui, final Session session, final Group group, final String formToRevert, boolean ask) {
        if (!ask || askGroupRevert(ui, EmailDataHelper.convertFormToString(formToRevert), VradiHelper.getEntityName(group))){
            VradiTask<Void> revertSendingTask = new VradiTask<Void>(ui) {

                @Override
                public Void doAction() throws Exception {
                    VradiService.getVradiStorageService().revertGroupDeletion(session.getWikittyId(), group.getWikittyId(), formToRevert);
                    return null;
                }
            };
            revertSendingTask.execute();
        }
    }

    protected boolean askUserRevert(JPanel ui, String formAsString, String userAsString) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmRevertUserFormDeletion", formAsString,
                        userAsString),
                _("vradi.email.confirmRevertUserFormDeletionTitle"),
                JOptionPane.YES_NO_OPTION);

        return yes == JOptionPane.YES_OPTION;
    }

    protected boolean askGroupRevert(JPanel ui, String formAsString, String userAsString) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmRevertGroupFormDeletion", formAsString,
                        userAsString),
                _("vradi.email.confirmRevertGroupFormDeletionTitle"),
                JOptionPane.YES_NO_OPTION);

        return yes == JOptionPane.YES_OPTION;
    }

    public void selectOffer(JAXXContext context, JTable listTable) {
        OfferListTableModel tableModel = (OfferListTableModel)listTable.getModel();
        int selectedRow = listTable.getSelectedRow();

        if (selectedRow >= 0 && selectedRow < tableModel.getRowCount()) {
            selectedRow = listTable.convertRowIndexToModel(selectedRow);

            FormPagedResult formPageDTO = tableModel.getFormPagedResult();
            List<String> formsToShow = formPageDTO.getFormsIdsToShow();

            String id = formsToShow.get(selectedRow);
            if (id != null) {
                UIHelper.getHandler(context, OfferListHandler.class).addEditPane(context, id);
            }
        }
    }

    public void displayForm(JAXXContext context, VradiTreeNode node) {

        Session session = EmailNavigationTreeHelper.getParentSession(node);

        if (session != null) {
            context.setContextValue(session);
        }

        String id = node.getId();

        Form form = getProxy().restore(Form.class, id);

        context.setContextValue(form);
    }

    public void addAllUserSending(QueryMakerViewUI ui, Session session, String formId, List values) {
        if (askToAddUsers(ui, EmailDataHelper.convertFormToString(formId), EmailDataHelper.convertObjectBeanToString(values, User.class))){
            for (Object selected : values) {
                addUserSending(ui, session, formId, selected, false);
            }
        }
    }

    public void addAllUserSending(QueryMakerViewUI ui, Session session, List<String> formsIds, List values) {

        if (askToAddUsers(ui, EmailDataHelper.convertFormsIdsToString(formsIds), EmailDataHelper.convertObjectBeanToString(values, User.class))){
            for (Object selected : values) {
                addUserSending(ui, session, formsIds, selected, false);
            }
        }
    }

    public void addUserSending(
            JAXXContext context,
            Session session,
            List<String> formsId,
            Object selected) {

       addUserSending(
            context,
            session,
            formsId,
            selected,
            true);
    }

    protected void addUserSending(
            JAXXContext context,
            final Session session,
            final List<String> formsIds,
            final Object selected,
            boolean ask) {

        User user = (User) selected;
        if (!ask || askToAddUsers(context, EmailDataHelper.convertFormsIdsToString(formsIds), VradiHelper.getEntityName(user))){

            VradiTask<Void> createSendingTask = new VradiTask<Void>(context) {

                @Override
                public Void doAction() throws Exception {
                    // Create sendings
                    getStorageService().createSending(session.getWikittyId(), (User)selected, formsIds);
                    return null;
                }
            };

            createSendingTask.execute();
        }
    }

    public void addUserSending(
            JAXXContext context,
            Session session,
            String formId,
            Object selected) {

        addUserSending(
            context,
            session,
            formId,
            selected, true);
    }

    protected void addUserSending(
            JAXXContext context,
            Session session,
            String formId,
            Object selected, boolean ask) {

        User user = (User) selected;
        if (!ask || askToAddUsers(context, EmailDataHelper.convertFormToString(formId), VradiHelper.getEntityName(user))){
            List<String> formsIds = new ArrayList<String>();
            formsIds.add(formId);
            addUserSending(context, session, formsIds, user, false);
        }
    }

    protected boolean askToAddUsers(JAXXContext context, String formAsString, String userAsString) {

        if (StringUtils.isEmpty(userAsString)) {
            return true;
        }
        int yes = JOptionPane.showConfirmDialog(
                (Component)context,
                _("vradi.email.confirmAddUser", formAsString,
                        userAsString),
                _("vradi.email.confirmAddTitle"),
                JOptionPane.YES_NO_OPTION);

        return yes == JOptionPane.YES_OPTION;
    }

    public void removeUserSending(
            JAXXContext context,
            Session session,
            List<String> formsIds,
            Object[] selecteds) {
        removeUserSending(context, session, formsIds, selecteds, true);
    }

    public void removeUserSending(
            JAXXContext context,
            final Session session,
            final List<String> formsIds,
            final Object[] selecteds, boolean ask) {

        int yes = JOptionPane.YES_OPTION;

        if (ask) {
            yes = JOptionPane.showConfirmDialog(
                (Component)context,
                _("vradi.email.confirmDelete", EmailDataHelper.convertFormsIdsToString(formsIds)),
                _("vradi.email.confirmDeleteTitle"),
                JOptionPane.YES_NO_OPTION);
        }

        if (yes == JOptionPane.YES_OPTION){

            VradiTask<Void> removeSendingTask = new VradiTask<Void>(context) {

                @Override
                public Void doAction() throws Exception {
                    for (Object selected : selecteds) {
                        getStorageService().removeSending(session.getWikittyId(), formsIds, (User)selected);
                    }
                    return null;
                }
            };

            removeSendingTask.execute();
        }
    }

    @SuppressWarnings({"unchecked"})
    public void removeUserSending(
            JAXXContext context,
            Session session,
            String formId,
            Object[] selecteds) {

        removeUserSending(context, session, formId, selecteds, true);
    }

    @SuppressWarnings({"unchecked"})
    protected void removeUserSending(
            JAXXContext context,
            Session session,
            String formId,
            Object[] selecteds, boolean ask) {

        List<String> formsIds = new ArrayList<String>();
        formsIds.add(formId);

        removeUserSending(context, session, formsIds, selecteds, ask);
    }

    public void addAllGroupSending(JAXXContext context, Session session, String formId, List values) {
        if (askToAddGroups(context, EmailDataHelper.convertFormToString(formId),EmailDataHelper.convertObjectBeanToString(values, Group.class))){
            for (Object selected : values) {
                addGroupSending(context, session, formId, selected, false);
            }
        }
    }

    public void addAllGroupSending(JAXXContext context, Session session, List<String> beanList, List values) {

        if (askToAddGroups(context, EmailDataHelper.convertFormsIdsToString(beanList), EmailDataHelper.convertObjectBeanToString(values, Group.class))){
            for (Object selected : values) {
                addGroupSending(context, session, beanList, selected, false);
            }
        }
    }

    protected void addGroupSending(
            JAXXContext context,
            Session session,
            List<String> formsIds,
            Object selected) {

        addGroupSending(
            context,
            session,
            formsIds,
            selected,
            true);
    }

    protected void addGroupSending(
            JAXXContext context,
            final Session session,
            final List<String> formsIds,
            Object selected,
            boolean ask) {

        final Group group = (Group) selected;
        if (!ask || askToAddGroups(context, EmailDataHelper.convertFormsIdsToString(formsIds), VradiHelper.getEntityName(group))){

            VradiTask<Void> createSendingTask = new VradiTask<Void>(context) {
                @Override
                public Void doAction() throws Exception {
                    // Create sendings
                    getStorageService().createAllSending(session.getWikittyId(), group, formsIds);

                    return null;
                }
            };

            createSendingTask.execute();
        }
    }

    public void addGroupSending(
            JAXXContext context,
            Session session,
            String formId,
            Object selected) {

           addGroupSending(context, session, formId, selected, true);
    }

    public void addGroupSending(
            JAXXContext context,
            Session session,
            String formId,
            Object selected,
            boolean ask) {

        List<String> formsIds = new ArrayList<String>();
        formsIds.add(formId);
        
        addGroupSending(context, session, formsIds, selected, ask);
    }

    protected boolean askToAddGroups(JAXXContext context, String formAsString, String groupAsString) {

        if (StringUtils.isEmpty(groupAsString)) {
            return true;
        }
        int yes = JOptionPane.showConfirmDialog(
                (Component)context,
                _("vradi.email.confirmAddGroup", formAsString,
                        groupAsString),
                _("vradi.email.confirmAddTitle"),
                JOptionPane.YES_NO_OPTION);
        return yes == JOptionPane.YES_OPTION;
    }

    public void removeGroupSending(
            JAXXContext context,
            final Session session,
            final List<String> formsIds,
            final Object[] selecteds) {

        int yes = JOptionPane.showConfirmDialog(
                (Component)context,
                _("vradi.email.confirmDelete", EmailDataHelper.convertFormsIdsToString(formsIds)),
                _("vradi.email.confirmDeleteTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION){

            VradiTask<Void> removeAllSendingTask = new VradiTask<Void>(context) {
                @Override
                public Void doAction() throws Exception {
                    for (Object selected : selecteds) {
                        getStorageService().removeAllSending(session.getWikittyId(), formsIds, (Group)selected);
                    }
                    return null;
                }
            };

            removeAllSendingTask.execute();
        }
    }

    @SuppressWarnings({"unchecked"})
    public void removeGroupSending(
            JAXXContext context,
            Session session,
            String formId,
            Object[] selecteds) {

        List<String> formsIds = new ArrayList<String>();
        formsIds.add(formId);
        removeGroupSending(context, session, formsIds, selecteds);
    }

    protected void reloadTree(EmailPopupUI ui) {

        // Refresh
        EmailNavigationTreeHelper helper = ui.getHelper();

        // Get selected path
        TreePath[] selectionPaths = helper.getSelectionModel().getSelectionPaths();

        // Update datas
        ui.updateTree();

        // Select
        if (selectionPaths != null){
            helper.getSelectionModel().setSelectionPaths(selectionPaths);
        }
    }

//    public Map<VradiClientDTO, Set<String>> getAllClientsAndUsers() {
//        SortedMap<VradiClientDTO, Set<String>> clientsAndUsers = new TreeMap<VradiClientDTO, Set<String>>(VradiComparators.CLIENT_COMPARATOR);
//
//        // map client id = user ids
//        Map<String, Set<String>> clientsIdAndUsers = new TreeMap<String, Set<String>>();
//
//        List<VradiClientDTO> clients = getAllClients();
//        List<User> users = VradiContext.getUsersInEntryDef();
//
//        for (VradiClientDTO client : clients) {
//            Set<String> ids = new HashSet<String>();
//            clientsAndUsers.put(client, ids);
//            clientsIdAndUsers.put(client.getWikittyId(), ids);
//        }
//
//        for (User user : users) {
//            clientsIdAndUsers.get(user.getClient()).add(user.getWikittyId());
//        }
//
//        return clientsAndUsers;
//    }

    /**
     * Called to send only one simple sending.
     * 
     * @param ui parent ui
     * @param sending sending to send
     */
    public void sendSending(SendingViewUI ui, final Sending sending){
        Object[] options = {_("vradi.email.confirmSendSending.optionYes"),
                            _("vradi.email.confirmSendSending.optionNo"),
                            _("vradi.email.confirmSendSending.optionOtherMail")};
        
        int i = JOptionPane.showOptionDialog(ui,
                _("vradi.email.confirmSendSending", getUserName(sending)),
                _("vradi.email.confirmSendSendingTitle"),
                JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
                null,
                options,
                options[2]);

        String email = null;

        if (i == JOptionPane.CANCEL_OPTION) {

            // Ask email to send
            email = JOptionPane.showInputDialog(
                ui,
                _("vradi.email.confirmSendSending.askMail"),
                _("vradi.email.confirmSendSending.askMailTitle"),
                JOptionPane.PLAIN_MESSAGE);

            i = JOptionPane.OK_OPTION;
        }

        if (i == JOptionPane.OK_OPTION) {

            // Get session paragraph
            Session session = getSessionBySending(sending.getWikittyId());
            final String sessionParagraph = session.getParagraph();

            final String finalEmail = email;
            final EmailPopupUI finalUI = ui.getParentContainer(EmailPopupUI.class);
            VradiTask<Void> sendSendingTask = new VradiTask<Void>(ui, "vradi.email.sendSending"){
                
                @Override
                public Void doAction() throws VradiException {
                    try {
                        String sendingLogs = getStorageService().sendMessage(sending.getWikittyId(), sessionParagraph, finalEmail);

                        if (sendingLogs != null && !sendingLogs.isEmpty()) {
                            JOptionPane.showMessageDialog(finalUI, _("vradi.error.sending.sendDialogError", sendingLogs),
                                    _("vradi.error.sending.sendDialogError.title"),
                                    JOptionPane.ERROR_MESSAGE);
                        }
                    } catch (VradiException eee) {
                        JOptionPane.showMessageDialog(finalUI, _("vradi.email.sendSending.sendError",
                                eee.getMessage()), _("vradi.email.sendSession.error"),
                                JOptionPane.ERROR_MESSAGE);

                        // Set session status to error
//                        updateSendingStatus(sendingResult, VradiConstants.SendingStatus.ERROR);
                    }
                    return null;
                }

                @Override
                public void doWhenError(Exception eee){
//                    updateSendingStatus(sending, VradiConstants.SendingStatus.ERROR);
                }
            };
            sendSendingTask.execute();
        }
    }

    /**
     * Called to send a full session.
     * 
     * Don't call without VradiSessionDTO selection !
     * 
     * @param ui parent ui
     * @return {@code true} if task has been 
     */
    public boolean send(final EmailPopupUI ui) {
        final Session session = ui.getHelper().getSelectedActiveSession();

        // ask user for confirmation
        int confirm = JOptionPane.showConfirmDialog(ui,
                _("vradi.email.confirmSendSession"),
                _("vradi.email.confirmSendSessionTitle"),
                JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
        if (confirm != JOptionPane.YES_OPTION) {
            return false;
        }

        // Check if emails and templates exists
        if (!checkAllSending(ui, session)){
            return false;
        }

        // Display progess bar with cancel button
        final AbstractProgressBarUI pb = new AbstractProgressBarUI(ui) {
            /** serialVersionUID. */
            private static final long serialVersionUID = 5926558779822425487L;

            @Override
            public void cancel() {
                try {
                    VradiService.getVradiStorageService().stopSentMail(session);
                    this.dispose();
                } catch (VradiException eee) {
                    log.info("Cant stop sending emails : ", eee);
                    ErrorDialogUI.showError(eee);
                }
                ui.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            }
        };

        if (log.isInfoEnabled()) {
            SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
            log.info("Sending session : " + session.getNum() + " - " +
                format.format(session.getSessionDate()) + " status : " +
                SessionStatus.getStatus(session.getStatus()).getDescription());
        }


        // premiere tache de génération des PDF
        VradiTask<Boolean> pdfGenerationAndSendingTask = new VradiTask<Boolean>(
                ui, pb, n_("vradi.email.generation")) {

            @Override
            public Boolean doAction() throws Exception {
                try {

                    VradiService.getVradiStorageService().generatePDFForSession(
                            session.getWikittyId());

                } catch (Exception eee){
                    if (log.isErrorEnabled()) {
                        log.error("Error generating pdf : ", eee);
                    }
                    doWhenError(eee);
                    ErrorDialogUI.showError(eee);
                    return false;
                }
                return true;
            }

            @Override
            public void doWhenDone() throws Exception {

                // If pdf generation is successfully done
                Boolean result = get();
                if (result != null && result) {

                    if (log.isInfoEnabled()) {
                        log.info("Start sending mails");
                    }

                    VradiTask<Void> sentTask = new VradiTask<Void>(ui, pb, n_("vradi.email.sendSession")){

                        @Override
                        public Void doAction() throws Exception {
                            try {
                                String sessionLogs = VradiService.getVradiStorageService().sendMessages(session.getWikittyId());

                                if (!sessionLogs.isEmpty()) {
                                    JOptionPane.showMessageDialog(ui,
                                            _("vradi.error.session.sendDialogError", sessionLogs),
                                            _("vradi.error.session.sendDialogError.title"),
                                            JOptionPane.ERROR_MESSAGE);
                                }
                            } catch (VradiException eee) {

                                log.error("Failed to send message : ", eee);
                                
                                JOptionPane.showMessageDialog(ui, _("vradi.email.sendSending.sendError",
                                        eee.getMessage()), _("vradi.email.sendSession.error"),
                                        JOptionPane.ERROR_MESSAGE);

                                // Set session status to error
//                                updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
                            }
                            return null;
                        }



                        @Override
                        public void doWhenError(Exception eee){
//                            updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
                        }
                    };

                    sentTask.execute();
                }
            }

            @Override
            public void doWhenError(Exception eee){
//                updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
            }
        };
        
        // Executes task
        pdfGenerationAndSendingTask.execute();
        
        return true;
    }

    protected boolean checkAllSending(EmailPopupUI ui, Session session){

        for (Sending sending : EmailDataHelper.getAllSending(session)){

            boolean emailEmpty = VradiService.getVradiDataService().checkUserEmail(sending);

            if (emailEmpty){
                User user = VradiService.getWikittyProxy().restore(User.class, sending.getUser());
                Client client = VradiService.getWikittyProxy().restore(Client.class, user.getClient());
                boolean abord = askAbordSending(ui, _("vradi.email.askUserEmailEmpty", user.getName(), client.getName()));
                if (abord){

                    // Set to error status
                    updateSendingStatus(sending, VradiConstants.SendingStatus.ERROR);

                    return false;
                }
            }

            // Check PDF generation
            Set<String> formIds = SendingHelper.extractForms(VradiService.getWikittyProxy(), sending.getGroupForms());

            // If forms are deleted, nothing to send
            if (formIds == null) {
                return true;
            }
            List<Form> forms = getProxy().restore(Form.class, new ArrayList<String>(formIds));

            List<String> alreadyAskFormType = new ArrayList<String>();
            for (Form form : forms) {

                String extensionName = null;
                for(String ext : form.getExtensionNames()) {
                    if(!ext.equals(Infogene.EXT_INFOGENE)
                            && !ext.equals(Form.EXT_FORM)) {
                        extensionName = ext;
                        break;
                    }
                }
                boolean isTemplateExist = false;
                try {
                    isTemplateExist = VradiService.getVradiStorageService().checkTemplateExist(extensionName);
                } catch (VradiException eee) {
                    log.error("Cant check if template exist : ", eee);
                    ErrorDialogUI.showError(eee);  
                }

                if (!isTemplateExist && !alreadyAskFormType.contains(extensionName)){
                    boolean abord = askAbordSending(ui, _("vradi.email.askModelEmpty", extensionName));
                    if (abord){

                        // Set to error status
                        updateSendingStatus(sending, VradiConstants.SendingStatus.ERROR);
                        
                        return false;
                    }
                    alreadyAskFormType.add(extensionName);
                }
            }
        }
        return true;
    }

    protected boolean askAbordSending(EmailPopupUI ui, String msg) {
        int i = JOptionPane.showConfirmDialog(ui,
                msg,
                _("vradi.email.askAbordSendingTitle"),
                JOptionPane.YES_NO_OPTION);

        return i != JOptionPane.YES_OPTION;
    }

    public Sending updateSendingStatus(Sending sending, VradiConstants.SendingStatus status) {
        if (sending == null) {
            return null;
        }

        Sending sendingRestored = getProxy().restore(Sending.class, sending.getWikittyId());

        sendingRestored.setStatus(status.getValue());

        log.info("Saving sending " + sendingRestored.getWikittyId() + " with status " + status.getDescription());
        return getProxy().store(sendingRestored);

    }

//    public Session updateSessionStatus(Session session, VradiConstants.SessionStatus status) {
//        if (session == null) {
//            return null;
//        }
//
//        Session sessionRestored = getProxy().restore(Session.class, session.getWikittyId());
//
//        sessionRestored.setStatus(status.getValue());
//        sessionRestored.setSessionDate(new Date());
//
//        log.info("Saving session " + sessionRestored.getWikittyId() + " with status " + status.getDescription());
//        return getProxy().store(sessionRestored);
//
//    }

    protected String getUserName(Sending sending){
        try {
            String userId = sending.getUser();
            return getProxy().restore(User.class, userId).getName();
        } catch (Exception eee) {
            log.error("Cant get sending user name : ", eee);
            ErrorDialogUI.showError(eee);
        }
        return null;
    }

    /**
     * Close selected session in tree.
     *
     * Don't call it if no session is selected
     *
     * @param ui parent ui
     */
    public void deleteSession(EmailPopupUI ui) {
        Session session = ui.getHelper().getSessionSelected();
        if (session != null) {

            int i = JOptionPane.showConfirmDialog(
                    ui, _("vradi.email.confimCloseSession"),
                    _("vradi.email.confimCloseSession.titles"),
                    JOptionPane.YES_NO_OPTION);

            if (i == JOptionPane.YES_OPTION){
                try {
                    VradiService.getVradiStorageService().deleteSession(session.getWikittyId());
                } catch (VradiException eee) {
                    log.error("Cant delete session : ", eee);
                    ErrorDialogUI.showError(eee);
                }
            }
        }
    }

    public void updateReceptionProof(Sending sending, boolean isSelected){
        sending.setReceptionProof(isSelected);
        getProxy().store(sending);
    }

    public void updateParagraph(Sending sending, String text){
        sending.setParagraph(text);
        getProxy().store(sending);
    }

    protected Sending getSendingForUser(Session session, String userId) {
        List<Sending> sendings = EmailDataHelper.getAllSending(session);
        for (Sending sending : sendings) {
            if (sending != null) {
                if (sending.getUser().equals(userId)) {
                    return sending;
                }
            }
        }
        return null;
    }

//    protected Sending getSendingForClient(Session session, String clientId) {
//        List<Sending> sendings = getSendingList(session);
//        for (Sending sending : sendings) {
//            if (sending.getClient().equals(clientId)) {
//                return sending;
//            }
//        }
//        return null;
//    }

    protected Session getSessionBySending(String sendingId) {
        // Get all sending with form
        return VradiService.getVradiDataService().getSessionBySending(sendingId);
    }

    protected List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }
}

