/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: EmailNavigationTreeHelper.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.2/vradi-swing/src/main/java/com/jurismarches/vradi/ui/email/helpers/EmailNavigationTreeHelper.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.email.helpers;

import static com.jurismarches.vradi.VradiConstants.SessionStatus;
import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.jurismarches.vradi.beans.SendingHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.ui.email.loadors.EmailGroupUsersNodeLoadors;
import javax.swing.SwingUtilities;
import javax.swing.tree.TreeModel;

import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.email.loadors.AbstractEmailNodeLoador;
import com.jurismarches.vradi.ui.email.loadors.EmailFormNodeLoadors;
import com.jurismarches.vradi.ui.email.loadors.EmailGroupNodeLoadors;
import com.jurismarches.vradi.ui.email.loadors.EmailUsersNodeLoadors;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.helpers.VradiTreeHelper;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.nav.NavHelper;
import jaxx.runtime.swing.nav.tree.NavTreeNodeChildLoador;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jurismarches.vradi.ui.email.EmailHandler;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.*;

/**
 * @author letellier
 */
public class EmailNavigationTreeHelper extends VradiTreeHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(EmailNavigationTreeHelper.class);

    protected List<Session> sessions;
    protected Date searchDate;

    // All filter
    public enum EmailFilterEnum {
        // FILTER_BY_CLIENT(_("vradi.email.clientFilter")),
        FILTER_BY_USER(_("vradi.email.userFilter")),
        FILTER_BY_GROUP(_("vradi.email.groupFilter")),
        FILTER_BY_FORM(_("vradi.email.formFilter"));

        protected int value;
        protected String text;

        EmailFilterEnum(String text) {
            this.text = text;
        }

        @Override
        public String toString(){
            return text;
        }
    }

    public static String PATH_SEPARATOR = "/";

    // Filter by client by defaut
    protected EmailFilterEnum filter = EmailFilterEnum.FILTER_BY_USER;

    public EmailNavigationTreeHelper(VradiDataProvider dataProvider) {
        super();
        setDataProvider(dataProvider);
        
        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL);
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    public WikittyProxy getProxy() {
        return getDataProvider().getWikittyProxy();
    }

    // Get email handler
    protected EmailHandler getHandler(JAXXContext context) {
        return UIHelper.getHandler(context, EmailHandler.class);
    }

    public String getPathSeparator() {
        return PATH_SEPARATOR;
    }

    public boolean isByForm() {
        return filter.equals(EmailFilterEnum.FILTER_BY_FORM);
    }

    public boolean isByUser() {
        return filter.equals(EmailFilterEnum.FILTER_BY_USER);
    }

    /* public boolean isByClient() {
        return filter.equals(EmailFilterEnum.FILTER_BY_CLIENT);
    }*/

    public boolean isByGroup() {
        return filter.equals(EmailFilterEnum.FILTER_BY_GROUP);
    }

    /**
     * Return selected session active session.
     * 
     * @return session entity
     */
    public Session getSelectedActiveSession() {
        
        Session activeSession = null;
        VradiTreeNode selectedNode = getSelectedNode();

        if (selectedNode != null) {
            Session session = getParentSession(selectedNode);
    
            if (SessionStatus.isToSend(session)) {
                activeSession = session;
            }
        }

        return activeSession;
    }

    public void updateTree(List<Session> sessions, EmailFilterEnum filter, Date searchDate) {
        if (log.isDebugEnabled()) {
            log.debug("update data, nb sessions : " + sessions.size());
        }
        this.filter = filter;
        this.sessions = sessions;
        this.searchDate = searchDate;

        VradiTreeNode rootNode = new VradiTreeNode(
                String.class,
                "Root node",
                null,
                null
        );

        for (Session session : sessions) {
            rootNode.add(createSessionNode(session));
        }
        TreeModel model = createModel(rootNode);
        getBridge().setRoot(rootNode);

        // Populate childs nodes
        rootNode.populateChilds(getBridge(), getDataProvider());
                                  
        getUI().setModel(model);
    }

    public VradiTreeNode createSessionNode(Session session) {
        if (log.isDebugEnabled()){
            log.debug("createSession node " + session.getSessionDate() + " num " + session.getNum());
        }
        String sessionId = session.getWikittyId();

        NavTreeNodeChildLoador<?, ?, VradiTreeNode> nodeLoador = null;

        if (isByForm()){
            nodeLoador = new EmailFormNodeLoadors();
        } else {
            if (isByUser()){
                nodeLoador = new EmailUsersNodeLoadors();
            } else if (isByGroup()){
                nodeLoador = new EmailGroupNodeLoadors();
            }
        }

        return new VradiTreeNode(
                Session.class,
                sessionId,
                null,
                nodeLoador);

    }

    public Session getSessionSelected(){
        return getSessionToDisplay(getSelectedNode());
    }
    
    public Session getSessionToDisplay(VradiTreeNode node){
        Session session = getParentSession(node);

        return session;
    }

    public Sending getSendingToDisplay(VradiTreeNode node){
        String id = node.getId();

        Session session = getParentSession(node);

        return EmailDataHelper.getUserSending(session, id);
    }

    public List<Sending> getSendingListToDisplay(VradiTreeNode node){
        Map<String, Sending> result = new HashMap<String, Sending>();
        Class type = node.getInternalClass();
        String id = node.getId();

        Session session = getParentSession(node);

        if (type.equals(Form.class)){

            // Get sending by form
            List<Sending> sendings = getSendingByForm(session, id);
            addSendingToMap(result, sendings);

        }

        return new ArrayList<Sending>(result.values());
    }

    protected void addSendingToMap(Map<String, Sending> result, List<Sending> sendings) {
        for (Sending sending : sendings){
            result.put(sending.getWikittyId(), sending);
        }
    }

    protected List<Sending> getSendingByForm(Session session, String formId) {
        Set<String> sendingIdsSet = session.getSending();
        List<Sending> result = new ArrayList<Sending>();
        if (sendingIdsSet == null) {
            return result;
        }
        List<Sending> sendings = getProxy().restore(Sending.class, new ArrayList<String>(sendingIdsSet));
        for (Sending sending : sendings) {
            Set<String> formIdsSet = SendingHelper.extractForms(VradiService.getWikittyProxy(), sending.getGroupForms());
            if (formIdsSet == null) {
                return result;
            }
            if (formIdsSet.contains(formId)) {
                result.add(sending);
            }
        }
        return result;
    }

    protected List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }

    public static Session getParentSession(VradiTreeNode node) {
        VradiTreeNode sessionNode = getParentSessionNode(node);
        if (sessionNode == null) {
            return null;
        }
        return VradiService.getWikittyProxy().restore(Session.class, sessionNode.getId());
    }

    public static VradiTreeNode getParentSessionNode(VradiTreeNode node) {
        if (node == null) {
            return null;
        }
        if (node.getInternalClass().equals(Session.class)) {
            return node;
        }
        VradiTreeNode parent = node.getParent();
        return getParentSessionNode(parent);
    }

    protected void refresh(VradiTreeNode node) {

        if (node == null) {
            return;
        }

        getBridge().nodeChanged(node);
    }

    @SuppressWarnings({"ConstantConditions"})
    @Override
    public void putWikitty(final WikittyServiceEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("[Email] put wikitty");
        }

        // map between id and extensions "name" (not extension ids)
        final Map<String, Set<String>> idAndExtensions = event.getIdExtensions();

        SwingUtilities.invokeLater(new Thread() {
            @Override
            public void run() {
                
                for (final String wikittyId : event.getIds()) {
                    Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);


                    // Session
                    if (wikittyExtensions.contains(Session.EXT_SESSION)) {
                        if (log.isDebugEnabled()) {
                            log.debug("[Session] put wikitty");
                        }

                        // Check if session exist
                        VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);

                        // cas modification
                        if (existingNode != null) {

                            if (log.isDebugEnabled()) {
                                log.debug("[Session] Modification case :  " + existingNode.getId());
                            }

                            refresh(existingNode);
                        }

                        // Creation case
                        else {

                            Session session = getProxy().restore(Session.class, wikittyId);

                            if (session == null) {
                                return;
                            }

                            // Si la date correspond
                            if (0 == DateUtil.getDifferenceInDays(searchDate, session.getSessionDate())){

                                if (log.isDebugEnabled()) {
                                    log.debug("[Session] Creation case :  " + session.getSessionDate());
                                }

                                // Tri
                                VradiTreeNode newSessionNode = createSessionNode(session);

                                // On l'insert
                                insertNode(getRootNode(), newSessionNode);

                                refresh(newSessionNode);
                            }

                        }
                    }

                    // Sending
                    if (wikittyExtensions.contains(Sending.EXT_SENDING)) {
                        if (log.isDebugEnabled()) {
                            log.debug("[Sending] put wikitty");
                        }

                        // Restore sending
                        Sending sending = VradiService.getWikittyProxy().restore(Sending.class, wikittyId);

                        updateSendingInTree(sending);
                    }
                }
            }
        });
    }

    protected void updateSendingInTree(Sending sending) {

        if (sending == null) {
            return;
        }

        // Find session node
        Session session = null;
        String wikittyId = sending.getWikittyId();
        try {
            session = VradiService.getVradiDataService().getSessionOfSending(wikittyId);
        } catch (VradiException eee) {
            log.error("Can't find session for sending '" + wikittyId + "'", eee);
            ErrorDialogUI.showError(eee);
        }

        if (session == null) {
            return;
        }

        // Verify session is already in tree
        VradiTreeNode sessionNode = findNode(getRootNode(), session.getWikittyId());

        // If not found do nothing
        if (sessionNode == null) {
            return;
        }

        // If its sort by form
        if (isByForm()) {
            Set<String> formsIds = SendingHelper.extractForms(VradiService.getWikittyProxy(), sending.getGroupForms());
            if (formsIds != null) {
                for (String formId : formsIds) {
                    VradiTreeNode formNode = findNode(sessionNode, formId);

                    // If form is not in tree, add it
                    if (formNode == null) {
                        if (log.isDebugEnabled()) {
                            log.debug("[Sending] add form " + formId);
                        }
                        createNode(sessionNode, Form.class, formId, null);
                    }
                }
            }
        // Sort by user
        } else if (isByUser()) {
            String userId = sending.getUser();
            VradiTreeNode userNode = findNode(sessionNode, userId);

            // Refresh if already in tree
            if (userNode != null) {
                if (log.isDebugEnabled()) {
                    log.debug("[Sending] refresh user " + userId);
                }
                refresh(userNode);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("[Sending] create user " + userId);
                }

                // Create if user is not in tree
                createNode(sessionNode, User.class, userId, null);
            }

        // If its sort by group
        } else if (isByGroup()){
            String userId = sending.getUser();
            VradiTreeNode userNode = findNode(sessionNode, userId);

            // If user node is in tree, refresh it
            if (userNode != null) {
                if (log.isDebugEnabled()) {
                    log.debug("[Sending] refresh user " + userId);
                }
                refresh(userNode);
            } else {

                // Else, try to add it, we search all user group and for all
                // we will create missing node
                List<Group> groupList = VradiService.getVradiDataService().findUserGroups(userId);
                for (Group group : groupList) {
                    String groupId = group.getWikittyId();

                    // if group is already in tree
                    VradiTreeNode groupNode = findNode(sessionNode, groupId);
                    if (groupNode != null) {

                        if (log.isDebugEnabled()) {
                            log.debug("[Sending] create user " + userId);
                        }

                        // Create userNode
                        createNode(sessionNode, User.class, userId, null);
                    } else {

                        if (log.isDebugEnabled()) {
                            log.debug("[Sending] create group " + userId);
                        }
                        // Else create groupNode
                        createNode(sessionNode, Group.class, groupId, EmailGroupUsersNodeLoadors.class);
                    }
                }
            }
        }
    }

    protected void createNode(VradiTreeNode sessionNode, Class typeNode, String id, Class<? extends AbstractEmailNodeLoador<?>> loadorClass) {
        AbstractEmailNodeLoador<?> nodeLoador = null;
        if (loadorClass != null) {
            nodeLoador = NavHelper.getChildLoador(loadorClass);
        }
        VradiTreeNode newNode = new VradiTreeNode(typeNode, id, null, nodeLoador);
        sessionNode.add(newNode);
        getBridge().notifyNodeInserted(newNode);
    }
}
