/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.offer.thesaurus.helpers;

import javax.swing.tree.TreeModel;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.VradiComparators;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.RootThesaurusNodeLoadors;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.ThesaurusNodeLoadors;

import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;
import jaxx.runtime.swing.nav.tree.NavTreeHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;

import java.util.*;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public class ThesaurusTreeHelper extends NavTreeHelper<VradiTreeNode> implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeHelper.class);

    protected RootThesaurus rootThesaurus;

    protected Thesaurus thesaurus;

    protected boolean listening = true;

    /**
     * Create helper with first nodes are the root thesaurus
     *
     * @param dataProvider data provider
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider) {
        super();
        init(dataProvider);
    }

    /**
     * Create helper with root thesaurus for root node
     *
     * @param dataProvider  data provider
     * @param rootThesaurus root thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, RootThesaurus rootThesaurus) {
        super();
        this.rootThesaurus = rootThesaurus;
        init(dataProvider);
    }

    /**
     * Create helper with thesaurus for root node
     *
     * @param dataProvider data provider
     * @param thesaurus    thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, Thesaurus thesaurus) {
        super();
        this.thesaurus = thesaurus;
        init(dataProvider);
    }

    protected void init(VradiDataProvider dataProvider) {
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    @SuppressWarnings({"unchecked"})
    public TreeModel createTreeModel() {
        
        VradiTreeNode root;
        if (thesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] thesaurus " + thesaurus.getName());
            }

            root = new VradiTreeNode(
                    Thesaurus.class,
                    thesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else if (rootThesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] root thesaurus " + rootThesaurus.getName());
            }
            root = new VradiTreeNode(
                    RootThesaurus.class,
                    rootThesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] all root thesaurus");
            }
            root = new VradiTreeNode(
                    String.class,
                    "Root node",
                    null,
                    getChildLoador(RootThesaurusNodeLoadors.class)
            );
        }

        TreeModel model = createModel(root);

        // Populate childs nodes
        root.populateChilds(getBridge(), getDataProvider());

        return model;
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to select
     */
    public void tryToSelect(List<Thesaurus> thesaurus) {
        tryToSelect(extractIds(thesaurus));
    }

    /**
     * Try to select thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToSelect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            if (log.isDebugEnabled()) {
                log.debug("Try to select, ids are null");
            }
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (String id : thesaurusIds) {

            // Skip if root
            if (!id.equals(getRootNode().getId())) {

                VradiTreeNode rootNode = getRootNode();
                VradiTreeNode nodeFound = findNode(rootNode, id);
                if (nodeFound != null) {
                    nodes.add(nodeFound);
                }

            }
        }

        selectNodes(nodes);
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to unselect
     */
    public void tryToUnselect(List<Thesaurus> thesaurus) {
        tryToUnselect(extractIds(thesaurus));
    }

    /**
     * Try to unselect thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToUnselect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (String id : thesaurusIds) {
            VradiTreeNode nodeFound = findNode(getRootNode(), id);
            if (nodeFound != null) {
                nodes.add(nodeFound);
            }
        }

        unSelectNodes(nodes);
    }

    protected List<String> extractIds(List<Thesaurus> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Thesaurus bean : beans){
                if (bean != null) {
                    ids.add(bean.getWikittyId());
                }
            }
        }
        return ids;
    }

    public static RootThesaurus getRootThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    public static RootThesaurus getRootThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    protected int getIndexOfThesaurusSorted(String wikittyId) {
        // Tri
        Thesaurus concerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);
        List<Thesaurus> thesauruses = ThesaurusDataHelper.getChildrenOfThesaurus(wikittyId);
        Collections.sort(thesauruses, VradiComparators.THESAURUS_COMPARATOR);
        return thesauruses.indexOf(concerned);
    }

    protected int getIndexOfRootThesaurusSorted(String wikittyId) {
        // Tri
        RootThesaurus concerned = ThesaurusDataHelper.restoreRootThesaurus(wikittyId);
        List<RootThesaurus> rootThesauruses = ThesaurusDataHelper.getAllRootThesaurus();
        Collections.sort(rootThesauruses, VradiComparators.ROOT_THESAURUS_COMPARATOR);
        return rootThesauruses.indexOf(concerned);
    }

    protected void refresh(VradiTreeNode parentNode) {

        if (parentNode == null) {
            return;
        }
        
        if (log.isDebugEnabled()) {
            if (parentNode.getInternalClass().equals(Thesaurus.class)) {
                log.debug("Refresh thesaurus " + getThesaurus(parentNode).getName());
            } else if (parentNode.getInternalClass().equals(RootThesaurus.class)){
                log.debug("Refresh rootThesaurus " + getRootThesaurus(parentNode).getName());
            } else {
                log.debug("Refresh root node");
            }
        }
        
        List<VradiTreeNode> selectedNodes = getSelectedNodes();
        parentNode.removeAllChildren();
        parentNode.populateChilds(getBridge(), getDataProvider());

        getBridge().nodeStructureChanged(parentNode);
        selectNodes(selectedNodes);
    }

    @SuppressWarnings({"ConstantConditions"})
    @Override
    public void putWikitty(WikittyServiceEvent event) {

        if (!VradiContext.LISTEN_THESAURUS_MODIF) {
            return;
        }

        // map between id and extensions "name" (not extension ids)
        Map<String, Set<String>> idAndExtensions = event.getIdExtensions();
        for (String wikittyId : event.getIds()) {
            Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);

            // Thesaurus
            if (wikittyExtensions.contains(Thesaurus.EXT_THESAURUS)) {

                VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);
                
                // cas modification
                if (existingNode != null) {

                    // Parent id
                    VradiTreeNode parent = existingNode.getParent();

                    Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                    // Move case
                    String newParentId = thesaurusConcerned.getParent();
                    if (parent != null && !parent.getId().equals(newParentId)) {
                        if (log.isDebugEnabled()) {

                            Thesaurus parentFrom = ThesaurusDataHelper.restoreThesaurus(parent.getId());
                            Thesaurus newParent = ThesaurusDataHelper.restoreThesaurus(newParentId);
                            log.debug("Move case : " + thesaurusConcerned.getName()
                                    + " from parent" + parentFrom.getName()
                                    + " to parent" + newParent.getName());
                        }


                        // Delete old
                        VradiTreeNode oldParent = existingNode.getParent();
                        refresh(oldParent);

                        // Destination sort
                        VradiTreeNode newParentNode = findNode(getRootNode(), newParentId);
                        refresh(newParentNode);

                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("Modification case :  " + thesaurusConcerned.getName());
                        }

                        // Modification case
                        VradiTreeNode parentNode = existingNode.getParent();
                        refresh(parentNode);
                    }
                    return;
                }

                // Creation case
                else {
                    Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                    if (log.isDebugEnabled()) {
                        log.debug("Creation case :  " + thesaurusConcerned.getName());
                    }

                    // Find parent node
                    VradiTreeNode parentNode = findNode(getRootNode(), thesaurusConcerned.getParent());

                    // Tri
                    refresh(parentNode);
                }
            }

            // Root thesaurus
            if (wikittyExtensions.contains(RootThesaurus.EXT_ROOTTHESAURUS)) {
                refresh(getRootNode());
            }

        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {

        if (!VradiContext.LISTEN_THESAURUS_MODIF) {
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }

        for (String wikittyId : event.getIds()) {
            VradiTreeNode node = findNode(getRootNode(), wikittyId);

            // if null : not in tree
            if (node != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing node " + wikittyId + " from parent");
                }

                VradiTreeNode parent = node.getParent();
                refresh(parent);
            }
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        // should not happen in vradi
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service put extension event : " + event);
        }
        // TODO ???
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove extension event : " + event);
        }
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        // should not happen in vradi
    }
}
