package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: ConfigurationService.java 177 2014-06-10 09:09:58Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/ConfigurationService.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.franciaflex.faxtomail.persistence.entities.Configuration;
import com.franciaflex.faxtomail.persistence.entities.ConfigurationImpl;
import com.franciaflex.faxtomail.persistence.entities.ConfigurationTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.EmailAccount;
import com.franciaflex.faxtomail.persistence.entities.EmailAccountImpl;
import com.franciaflex.faxtomail.persistence.entities.EmailAccountTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserGroup;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserGroupTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.GroupChef;
import com.franciaflex.faxtomail.persistence.entities.GroupChefTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.MailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailFilterTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since x.x
 */
public class ConfigurationService extends FaxToMailServiceSupport {

    private static final Log log = LogFactory.getLog(ConfigurationService.class);

    /**
     * Return unique database configuration.
     * 
     * @return configuration (not {@code null}, created on not found in database)
     */
    public Configuration getConfiguration() {
        ConfigurationTopiaDao dao = getPersistenceContext().getConfigurationDao();
        Configuration config = dao.forAll().findUniqueOrNull();
        if (config == null) {
            config = new ConfigurationImpl();
        }
        return config;
    }

    /**
     * Save configuration.
     * 
     * @param config configuration to save
     */
    public void saveConfiguration(Configuration config) {
        ConfigurationTopiaDao dao = getPersistenceContext().getConfigurationDao();
        if (config.isPersisted()) {
            config = dao.update(config);
        } else {
            config = dao.create(config);
        }
        getPersistenceContext().commit();
    }

    /**
     * Get all mail filter ordered by position.
     * 
     * @return all mail filter
     */
    public List<MailFilter> getMailFilters() {
        MailFilterTopiaDao mailFilterDao = getPersistenceContext().getMailFilterDao();
        List<MailFilter> result = mailFilterDao.forAll().setOrderByArguments(MailFilter.PROPERTY_POSITION + " asc").findAll();
        return result;
    }

    /**
     * Save all mail filters.
     * 
     * @param mailFilters mail filters to save
     */
    public void saveMailFilters(List<MailFilter> mailFilters) {
        MailFilterTopiaDao dao = getPersistenceContext().getMailFilterDao();

        int position = 0;
        List<MailFilter> filters = dao.findAll();
        Map<String, MailFilter> filterById = new HashMap<>(Maps.uniqueIndex(filters, TopiaEntities.getTopiaIdFunction()));
        for (MailFilter mailFilter : mailFilters) {

            MailFilter filter;
            boolean create = mailFilter.getTopiaId() == null;
            if (create) {
                filter = dao.newInstance();
            } else {
                filter = filterById.remove(mailFilter.getTopiaId());
            }
            filter.setExpression(mailFilter.getExpression());
            filter.setMailFolder(mailFilter.getMailFolder());
            filter.setPosition(position);

            if (create) {
                filter = dao.create(filter);
            } else {
                filter = dao.update(filter);
            }
            
            position++;
        }
        
        dao.deleteAll(filterById.values());
        getPersistenceContext().commit();
    }

    /**
     * Return all email accounts (without password info).
     * 
     * @return all email account
     */
    public List<EmailAccount> getEmailAccountsWithoutPasswords() {
        List<EmailAccount> accounts = getEmailAccounts();
        // remove password information from bean
        final Binder<EmailAccount, EmailAccount> binderEmailAccount = BinderFactory.newBinder(EmailAccount.class);
        List<EmailAccount> result = Lists.transform(accounts, new Function<EmailAccount, EmailAccount>() {
            @Override
            public EmailAccount apply(EmailAccount input) {
                EmailAccount account = new EmailAccountImpl();
                binderEmailAccount.copyExcluding(input, account, EmailAccount.PROPERTY_PASSWORD);
                return account;
            }
        });

        return result;
    }

    public List<EmailAccount> getEmailAccounts() {
        EmailAccountTopiaDao emailAccountTopiaDao = getPersistenceContext().getEmailAccountDao();
        List<EmailAccount> result = emailAccountTopiaDao.findAll();
        return result;
    }

    /**
     * 
     * @param newEmailAccounts new email account list to save
     */
    public void saveEmailAccounts(List<EmailAccount> newEmailAccounts) {
        EmailAccountTopiaDao emailAccountTopiaDao = getPersistenceContext().getEmailAccountDao();
        List<EmailAccount> emailAccounts = emailAccountTopiaDao.findAll();
        Map<String, EmailAccount> emailAccountMap = new HashMap<>(Maps.uniqueIndex(emailAccounts, TopiaEntities.getTopiaIdFunction()));
        Binder<EmailAccount, EmailAccount> binder = BinderFactory.newBinder(EmailAccount.class);

        for (EmailAccount newEmailAccount : newEmailAccounts) {
            EmailAccount emailAccount;
            if (StringUtils.isNotBlank(newEmailAccount.getTopiaId())) {
                emailAccount = emailAccountMap.remove(newEmailAccount.getTopiaId());
            } else {
                emailAccount = new EmailAccountImpl();
            }

            binder.copyExcluding(newEmailAccount, emailAccount,
                    EmailAccount.PROPERTY_TOPIA_ID,
                    EmailAccount.PROPERTY_TOPIA_CREATE_DATE,
                    EmailAccount.PROPERTY_TOPIA_VERSION,
                    EmailAccount.PROPERTY_PASSWORD);

            // password is only present for new account (not for edited ones)
            if (StringUtils.isNotEmpty(newEmailAccount.getPassword())) {
                emailAccount.setPassword(newEmailAccount.getPassword());
            }

            if (emailAccount.isPersisted()) {
                emailAccountTopiaDao.update(emailAccount);
            } else {
                emailAccountTopiaDao.create(emailAccount);
            }
        }
        
        emailAccountTopiaDao.deleteAll(emailAccountMap.values());
        getPersistenceContext().commit();
    }

    public List<FaxToMailUser> getAllUsers() {
        FaxToMailUserTopiaDao faxToMailUserDao = getPersistenceContext().getFaxToMailUserDao();
        return faxToMailUserDao.forAll().setOrderByArguments(FaxToMailUser.PROPERTY_FIRST_NAME, FaxToMailUser.PROPERTY_LAST_NAME).find(0, 50);
    }

    public void saveUserFolders(Map<String, Collection<MailFolder>> userFolders) {
        log.warn("user folders = " + userFolders);

        FaxToMailUserTopiaDao faxToMailUserTopiaDao = getPersistenceContext().getFaxToMailUserDao();
        
        for (Map.Entry<String, Collection<MailFolder>> entry : userFolders.entrySet()) {
            String userId = entry.getKey();
            Collection<MailFolder> mailFolders = entry.getValue();
            
            FaxToMailUser user = faxToMailUserTopiaDao.findByTopiaId(userId);
            user.clearAffectedFolders();
            for (MailFolder mailFolder : mailFolders) {
                user.addAffectedFolders(mailFolder);
            }
            faxToMailUserTopiaDao.update(user);
        }
        
        getPersistenceContext().commit();
    }

    public List<FaxToMailUserGroup> getAllGroups() {
        FaxToMailUserGroupTopiaDao faxToMailUserGroupDao = getPersistenceContext().getFaxToMailUserGroupDao();
        return faxToMailUserGroupDao.forAll().setOrderByArguments(FaxToMailUserGroup.PROPERTY_FULL_PATH).findAll();
    }

    /**
     * Retourne l'ensemble des utilisateurs appartenant aux groupes gérés par les groupes chef.
     * 
     * @param user current user
     * @return all managed users
     */
    public List<FaxToMailUser> getUserManagedUsers(FaxToMailUser user) {

        List<FaxToMailUser> result = new ArrayList<>();

        // s'il n'y a pas de group, il ne peut pas y avoir de droits
        if (user.getUserGroups() != null) {

            // recuperation de tout les groupes chef dont l'utilsateur est membre
            GroupChefTopiaDao groupChefTopiaDao = getPersistenceContext().getGroupChefDao();
            List<GroupChef> groupChefs = groupChefTopiaDao.forUserGroupIn(user.getUserGroups()).findAll();
            
            // recuperation de tous les users des groups gérés
            FaxToMailUserTopiaDao faxToMailUserTopiaDao = getPersistenceContext().getFaxToMailUserDao();
            for (GroupChef groupChef : groupChefs) {
                // groups
                if (groupChef.getManagedGroups() != null) {
                    for (FaxToMailUserGroup group : groupChef.getManagedGroups()) {
                        List<FaxToMailUser> users = faxToMailUserTopiaDao.forUserGroupsContains(group).findAll();
                        result.addAll(users);
                    }
                }
                // users
                if (groupChef.getManagedUsers() != null) {
                    result.addAll(groupChef.getManagedUsers());
                }
            }
        }

        return result;
    }
}
