/*
 * #%L
 * ToPIA :: Service Replication
 * 
 * $Id: ReplicationNode.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.3/topia-service-replication/src/main/java/org/nuiton/topia/replication/model/ReplicationNode.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.replication.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.util.EntityOperator;
import org.nuiton.topia.persistence.util.EntityOperatorStore;

/**
 * Le modele d'un noeud de replication.
 *
 * Un noeud represente exactement un type d'entite a repliquer
 * 
 * @author chemit
 */
public class ReplicationNode {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(ReplicationNode.class);
    final TopiaEntityEnum contract;
    final EntityOperator<? super TopiaEntity> operator;
    /**
     * les couples d'association a partir de ce noeud
     */
    Map<String, ReplicationNode> associations;
    /**
     * les couples de composition a partir de ce noeud
     */
    Map<String, ReplicationNode> dependencies;
    /**
     * l'ensemble des associations a dettacher pendant la replication
     */
    Set<String> associationsToDettach;
    /**
     * l'ensemble des dependences a dettacher pendant la replication
     */
    Set<String> dependenciesToDettach;
    /**
     * la couverture de ce noeud
     */
    Set<ReplicationNode> shell;
    /**
     * la sequences d'operations a realiser lors de la replication de ce node
     */
    List<ReplicationOperationDef> operations;

    public ReplicationNode(TopiaEntityEnum contract) {        
        this.contract = contract;
        operator = EntityOperatorStore.getOperator(contract);
        associations = new HashMap<String, ReplicationNode>();
        dependencies = new HashMap<String, ReplicationNode>();
        shell = new HashSet<ReplicationNode>();
        associationsToDettach = new HashSet<String>();
        dependenciesToDettach = new HashSet<String>();
        operations = new ArrayList<ReplicationOperationDef>();
        if (log.isTraceEnabled()) {
            log.trace("new node : " + this);
        }
    }

    public void addAssociation(String name, ReplicationNode node) {
        associations.put(name, node);
    }

    public void addOperation(int index, ReplicationOperationDef op) {
        operations.add(index, op);
        if (log.isDebugEnabled()) {
            log.debug(op + " to node " + this);
        }
    }

    public void addOperation(ReplicationOperationDef op) {
        operations.add(op);
        if (log.isDebugEnabled()) {
            log.debug(op + " to node " + this);
        }
    }

    public List<ReplicationOperationDef> getOperations() {
        return operations;
    }

    public boolean hasAssociation() {
        return !associations.isEmpty();
    }

    public boolean hasAssociationsToDettach() {
        return !associationsToDettach.isEmpty();
    }

    public String[] getAssociationsDettached(ReplicationNode node) {
        Set<String> result = new HashSet<String>();
        for (String name : associationsToDettach) {
            ReplicationNode get = associations.get(name);
            if (get == node) {
                result.add(name);
            }
        }
        return result.toArray(new String[result.size()]);
    }

    public String[] getDependenciesDettached(ReplicationNode node) {
        Set<String> result = new HashSet<String>();
        for (String name : dependenciesToDettach) {
            ReplicationNode get = dependencies.get(name);
            if (get == node) {
                result.add(name);
            }
        }
        return result.toArray(new String[result.size()]);
    }

    public boolean hasDependenciesToDettach() {
        return !dependenciesToDettach.isEmpty();
    }

    public boolean hasDependency() {
        return !dependencies.isEmpty();
    }

    public void addDependency(String name, ReplicationNode node) {
        dependencies.put(name, node);
    }

    public void addAssociationToDettach(String key) {
        associationsToDettach.add(key);
    }

    public void addDependencyToDettach(String key) {
        dependenciesToDettach.add(key);
    }

    public Map<String, ReplicationNode> getAssociations() {
        return associations;
    }

    public Set<String> getAssociationsToDettach() {
        return associationsToDettach;
    }

    public Set<String> getDependenciesToDettach() {
        return dependenciesToDettach;
    }

    public TopiaEntityEnum getContract() {
        return contract;
    }

    public EntityOperator<? super TopiaEntity> getOperator() {
        return operator;
    }

    public Map<String, ReplicationNode> getDependencies() {
        return dependencies;
    }

    public Set<ReplicationNode> getShell() {
        return shell;
    }

    public void setShell(Set<ReplicationNode> shell) {
        this.shell = shell;
    }

    /**
     * Les operations sont triees sur leur phase.
     *
     * @see ReplicationOperationPhase
     * @see ReplicationNode#OPERATION_COMPARATOR
     */
    public void sortOperations() {
        Collections.sort(operations, OPERATION_COMPARATOR);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        ReplicationNode other = (ReplicationNode) obj;
        return contract == other.contract;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 37 * hash + contract.hashCode();
        return hash;
    }

    @Override
    public String toString() {
        return contract.toString();
    }
    public static final Comparator<ReplicationOperationDef>
            OPERATION_COMPARATOR = new Comparator<ReplicationOperationDef>() {

        @Override
        public int compare(ReplicationOperationDef o1,
                           ReplicationOperationDef o2) {
            int result = o1.getPhase().compareTo(o2.getPhase());
            return result;
        }
    };
}
