package org.nuiton.license.plugin.repository;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author chemit
 * @since 1.0.3
 */
public class LicenseRepositoryFactory {

    /**
     * class-path directory where is the licenses repository
     */
    public static final String JAR_LICENSE_REPOSITORY = "/META-INF/licenses";
    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(LicenseRepositoryFactory.class);

    /**
     * Obtain a new repository.
     *
     * @param useJarRepository flag to use at first a jar repository
     * @param load flag to load the repository
     * @param extraResolvers extra baseURLs to defined extra license resolver
     * @return the instanciate and ready to use license repository
     * @throws IllegalArgumentException
     * @throws IOException 
     */
    public static LicenseRepository newLicenseRepository(boolean useJarRepository, boolean load, String... extraResolvers) throws IllegalArgumentException, IOException {

        List<URL> baseURLs = new ArrayList<URL>();
        if (extraResolvers != null) {
            for (String exUrl : extraResolvers) {
                URL url;
                try {
                    url = new URL(exUrl);
                } catch (Exception e) {
                    if (log.isDebugEnabled()) {
                        log.warn("could not convert url [" + exUrl + "], for reason " + e.getMessage());
                    } else {
                        log.warn("could not convert url [" + exUrl + "], for reason " + e.getMessage());
                    }
                    log.warn("will skip the url [" + exUrl + "]");
                    continue;
                }
                baseURLs.add(url);
            }
        }
        if (!useJarRepository && baseURLs.isEmpty()) {
            log.warn("no repository to create!");
            // no repository to create
            return null;
        }

        LicenseRepository result = null;
        LicenseRepository current = null;
        if (useJarRepository) {
            current = result = newJarLicenseRepository(false);
        }

        for (URL baseURL : baseURLs) {
            if (baseURL != null && !baseURL.toString().trim().isEmpty()) {
                LicenseRepository r = new LicenseRepository();
                r.setBaseURL(baseURL);
                if (current == null) {
                    // no previous repository to chain
                    result = r;

                } else {
                    // chain to previous repository
                    current.appendRepository(r);
                }
                // new repository to chain
                current = r;
            }
        }
        if (load) {
            result.load();
        }
        return result;
    }

    public static LicenseRepository newJarLicenseRepository(boolean load) throws IOException {
        LicenseRepository result = new LicenseRepository();
        // the first repository is always a jar repository
        URL baseURL = LicenseRepositoryFactory.class.getResource(JAR_LICENSE_REPOSITORY);
        result.setBaseURL(baseURL);
        if (load) {
            result.load();
        }
        return result;
    }
}
