/*
 * #%L
 * Maven helper plugin
 * 
 * $Id: RestSession.java 767 2010-10-08 10:23:25Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-helper-plugin/tags/maven-helper-plugin-1.2.8/src/main/java/org/nuiton/io/rest/RestSession.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.io.rest;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnection;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.StatusLine;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * A REST session.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.3
 */
public class RestSession {
//    uncomment this to debug the request send by httpclient
//    static {
//        System.setProperty("org.apache.commons.logging.Log", "org.apache.commons.logging.impl.SimpleLog");
//        System.setProperty("org.apache.commons.logging.simplelog.showdatetime", "true");
//        System.setProperty("org.apache.commons.logging.simplelog.log.httpclient.wire.header", "debug");
//        System.setProperty("org.apache.commons.logging.simplelog.log.org.apache.commons.httpclient", "debug");
//    }

    protected HttpClient client;

    protected static final Log log = LogFactory.getLog(RestSession.class);

    protected boolean open;

    protected boolean showRequest;

    protected RestClientConfiguration configuration;

    RestSession(RestClientConfiguration configuration) {
        showRequest = configuration.isVerbose();
        this.configuration = configuration;
        client = new HttpClient();
        HostConfiguration hostConfiguration = new HostConfiguration();

        // Anomalie #917: Can not specify port in rest api (tchemit 2010-10-07)
        int port = configuration.getRestUrl().getPort();
        if (port == -1) {

            // no port set
            hostConfiguration.setHost(configuration.getRestUrl().getHost());
        } else {

            // a specific port is asked
            hostConfiguration.setHost(configuration.getRestUrl().getHost(), port);
        }

        // set encoding (will then encode parameters fine)
        client.getParams().setParameter(HttpMethodParams.HTTP_CONTENT_CHARSET, configuration.getEncoding());
        client.setHostConfiguration(hostConfiguration);
        client.setState(new HttpState());
    }

    public InputStream askData(RestRequest request) throws RestException, IOException {

        if (!open) {
            throw new IllegalStateException("the session is not opened");
        }
        HttpMethod gm;

        gm = doRequest(request);

        StatusLine sl = gm.getStatusLine();
        int statusCode = sl.getStatusCode();

        if (log.isDebugEnabled()) {
            log.debug("status code " + statusCode + " for " + gm.getPath());
        }

        if (statusCode == HttpStatus.SC_NOT_FOUND) {
            throw new RestDataNotFoundException(
                    "could not retreave some datas : " +
                    gm.getResponseBodyAsString());
        }

        if (statusCode != HttpStatus.SC_OK) {
            log.error("Error = " + gm.getResponseBodyAsString());
            releaseConnection(getConnection());
            throw new RestException(
                    "Got error code <" + statusCode + ":" +
                    sl.getReasonPhrase() + "> on " + gm.getPath());
        }
        InputStream stream = gm.getResponseBodyAsStream();
        return stream;
    }

    public InputStream sendData(RestRequest request) throws RestException,
            IOException {

        if (!open) {
            throw new IllegalStateException("the session is not opened");
        }

        HttpMethod gm = doRequest(request);

        StatusLine sl = gm.getStatusLine();
        int statusCode = sl.getStatusCode();

        if (log.isDebugEnabled()) {
            log.debug("status code " + statusCode + " for " + gm.getPath());
        }

        if (statusCode != HttpStatus.SC_OK) {
            log.error(gm.getResponseBodyAsString());
            releaseConnection(getConnection());
            throw new RestException(
                    "Got error code <" + statusCode + ":" +
                    sl.getReasonPhrase() + "> on " + gm.getPath());
        }

        InputStream stream = gm.getResponseBodyAsStream();
        return stream;
    }

    public void setOpen(boolean open) {
        this.open = open;
    }

    public void close() throws IOException {
        client.getState().clearCookies();
        releaseConnection(getConnection());
        open = false;
    }

    protected void releaseConnection(HttpConnection connection) {
        if (connection != null && connection.isOpen()) {
            client.getHttpConnectionManager().releaseConnection(connection);
        }
    }

    protected HttpConnection getConnection() {
        return client.getHttpConnectionManager().getConnection(
                client.getHostConfiguration()
        );
    }

    protected String getUri(String... paths) {
        String uri = client.getHostConfiguration().getHostURL();
        for (String path : paths) {
            uri += '/' + path;
        }
        return uri;
    }

    public HttpMethod doRequest(RestRequest request) throws IOException {

        String uri = getUri(request.getPath());
        String[] parameters = request.getParameters();
        Map<String, File> attachments = request.getAttachments();

        if (showRequest) {
            log.info("doRequest " + uri);
        }
        if (log.isDebugEnabled()) {
            log.debug("doRequest with parameters : " +
                      Arrays.toString(parameters));
        }
        PostMethod gm = new PostMethod(uri);
        if (open) {
            Cookie[] cookies = client.getState().getCookies();
            for (Cookie c : cookies) {
                if (log.isDebugEnabled()) {
                    log.debug("push cookie : " +
                              c.getName() + "=" + c.getValue());
                }
                gm.addRequestHeader("Cookie", c.toExternalForm());
            }
        }

        if (attachments == null) {

            // not a multi-part request

            if (parameters.length > 0) {

                // add parameters

                int nbParams = parameters.length / 2;
                for (int i = 0; i < nbParams; i++) {
                    String key = parameters[2 * i];
                    String value = parameters[2 * i + 1];
                    if (value == null) {
                        if (log.isDebugEnabled()) {
                            log.debug("skip null parameter " + key);
                        }
                        continue;
                    }
                    gm.addParameter(key, value);
                }
            }
        } else {

            // multi-part request

            List<Part> paramParts = new ArrayList<Part>();

            if (parameters.length > 0) {

                // add parameters as part of multi-part

                int nbParams = parameters.length / 2;
                for (int i = 0; i < nbParams; i++) {
                    String key = parameters[2 * i];
                    String value = parameters[2 * i + 1];
                    if (value == null) {
                        if (log.isDebugEnabled()) {
                            log.debug("skip null parameter " + key);
                        }
                        continue;
                    }
                    paramParts.add(new StringPart(key, value));
                    if (log.isDebugEnabled()) {
                        log.debug("add parameter [" + key + "]=" + value);
                    }
                }
            }

            // add file parts
            for (Entry<String, File> entry : attachments.entrySet()) {
                String key = entry.getKey();
                File file = entry.getValue();
                if (log.isDebugEnabled()) {
                    log.debug("add attachment " + key + "=" + file);
                }
                paramParts.add(new FilePart(
                        key,
                        file.getName(),
                        file,
                        FilePart.DEFAULT_CONTENT_TYPE,
                        configuration.getEncoding())
                );
            }
            if (attachments.isEmpty()) {
                log.warn("no attachment in a multi-part request!");
            }

            Part[] parts = paramParts.toArray(new Part[paramParts.size()]);

            MultipartRequestEntity entity =
                    new MultipartRequestEntity(parts, gm.getParams());
            if (log.isDebugEnabled()) {
                entity.writeRequest(System.out);
            }
            gm.setRequestEntity(entity);
//        gm.addRequestHeader("content-type", "multipart/form-data");
            gm.addRequestHeader("content-type", entity.getContentType());
            gm.addRequestHeader("content-length", entity.getContentLength() +
                                                  "");
            if (showRequest) {
                log.info("content-type : " + entity.getContentType() +
                         ", content-length : " + entity.getContentLength());
            }

        }

        client.executeMethod(gm);

        return gm;
    }
}
