/**
 * *##% Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.license;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.TreeMap;

/**
 * Un resolver de license a partir d'un {@link #baseURL} .
 * <p/>
 * On doit avoir un fichier
 * <pre>baseURL/licenses.txt
 * </pre>  qui contient la liste des licenses gerees par ce resolver.
 * <p/>
 * Chaque license est definit par deux fichiers  :
 * <pre>
 * baseURL/licenseName/license.txt
 * baseURL/licenseName/header.txt
 * </pre>
 *
 * @author chemit
 */
public class LicenseResolver {

    protected static final Log log = LogFactory.getLog(LicenseResolver.class);

    /**
     * BaseURL
     *
     * @required
     */
    protected String baseURL;

    /** list of known licenses with this resolver */
    protected Map<String, String> licenseNames;

    public LicenseResolver() {
    }

    public LicenseResolver(String baseURL) {
        this.baseURL = baseURL;
    }

    public String getBaseURL() {
        return baseURL;
    }

    public void setBaseURL(String baseURL) {
        this.baseURL = baseURL;
        // reset licenseNames
        licenseNames = null;
    }

    public Map<String, String> getLicenseNames() {
        if (licenseNames == null) {
            if (baseURL == null || baseURL.trim().isEmpty()) {
                throw new IllegalArgumentException("no baseURL defined in " + this);
            }

            Map<String, String> list = new TreeMap<String, String>();
            try {
                URL licenseURL = new URL(baseURL + "/licenses.properties");

                if (!checkExists(licenseURL)) {
                    throw new IllegalArgumentException("no licenses.properties found withurl [" + licenseURL + "] for resolver " + this);
                }
                Properties p = new Properties();
                p.load(licenseURL.openStream());
                for (Entry<Object, Object> entry : p.entrySet()) {
                    list.put((String) entry.getKey(), entry.getValue() + " [" + baseURL + "]");
                }
            } catch (MalformedURLException e) {
                throw new IllegalArgumentException(e);
            } catch (IOException e) {
                return null;
            }

            licenseNames = Collections.unmodifiableMap(list);
        }
        return licenseNames;
    }

    public License resolv(String licenseName) throws IllegalArgumentException {
        if (baseURL == null || baseURL.trim().isEmpty()) {
            throw new IllegalArgumentException("no baseURL defined in " + this);
        }
        if (licenseName == null || licenseName.trim().isEmpty()) {
            throw new IllegalArgumentException("licenceName can not be null");
        }
        licenseName = licenseName.trim().toLowerCase();

        if (!getLicenseNames().containsKey(licenseName)) {
            // license not found
            return null;
        }

        try {
            URL licenseURL = new URL(baseURL + "/" + licenseName + "/license.txt");
            URL headerURL = new URL(baseURL + "/" + licenseName + "/header.txt");

            if (!checkExists(licenseURL) || !checkExists(headerURL)) {
                // no content find
                return null;
            }

            License license = new License();
            license.setName(licenseName);
            license.setLicenseURL(licenseURL);
            license.setHeaderURL(headerURL);
            return license;


        } catch (MalformedURLException e) {
            throw new IllegalArgumentException(e);
        } catch (IOException e) {
            return null;
        }
    }

    @Override
    public String toString() {
        return super.toString() + "<baseURL:" + baseURL + ", licenseNames:" + licenseNames + ">";
    }

    protected boolean checkExists(URL url) throws IOException {
        return url.openConnection().getContentLength() > 0;
    }

}
