/*
 * *##% 
 * JAXX Action
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.action;

import jaxx.runtime.JAXXObject;

import javax.swing.JComponent;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * Action factory using the <code>ActionConfig-like</code> annotations to configure actions.
 * <p/>
 * <p/>
 * An {@link ActionFactory} builds actions always on a same type <code>A</code> and obtain them from some
 * {@link ActionProvider} via methods {@link #newAction(String, JComponent)} and {@link #newAction(String)} .
 * <p/>
 * If the action coming from the provider is not on the same type <code>A</code>, then the action is boxed in a *
 * action <code>A</code> and use the generic mecanism of delegation provided by {@link MyAbstractAction}.
 * <p/>
 * Use after the {@link #loadActions(jaxx.runtime.JAXXObject)} to instanciate actions in ui with id equals a known
 * action...
 * <p/>
 * All actions instanciated are stored in a cache that you can request via method {@link #getActionFromCache(String)},
 * {@link #cacheEntrySet()} and {@link #resetCache()}.
 * <p/>
 * You can also from this factory fires some action via the methods {@link #fireAction(String, Object, JComponent)} ,
 * {@link #fireAction(String, Object)} , {@link #fireAction0(String, Object, MyAbstractAction)}.
 * <p/>
 * Finally, a {@link #dispose()} method is there to shut down all instanciated action when you want to dispose all uis.
 *
 * @param <A> type of boxed action
 * @author chemit
 * @see ActionProvider
 * @see MyAbstractAction
 */
public interface ActionFactory<A extends MyAbstractAction> {

    /**
     * Method to init the dictionary of knwon action implementations.
     *
     * @return the dictionary of known action implementations
     */
    Map<String, Class<? extends MyAbstractAction>> init();

    /** @return the class of the base action of the factory. */
    Class<A> getBaseClass();

    /** @return the set of all the action's classes known by the factory. */
    Set<Entry<String, Class<? extends MyAbstractAction>>> implsEntrySet();

    /** @return the array of names of all actions known by the factory */
    String[] getActionNames();

    /** @return the set of all actions cached in factory  indexed by their name */
    Set<Entry<String, A>> cacheEntrySet();

    /**
     * @param actionKey the action's key
     * @return the action in cache or <code>null</code> if action is not in cache
     */
    MyAbstractAction getActionFromCache(String actionKey);

    /** clear the cache of instanciated actions. */
    void resetCache();

    /**
     * @param actionKey the key of an action
     * @return the action with this key from cache, or <code>null</code> if this action is not in cache
     */

    //A get(String actionKey);

    /**
     * For a given ui, load all actions registred in factory.
     * <p/>
     * The id of the widget in ui is directly mapped to a action key.
     *
     * @param ui the ui to treate
     */
    void loadActions(JAXXObject ui);

    /**
     * Obtain an action instance given his key and widget
     *
     * @param actionKey the key of action
     * @param component the component using the action
     * @return the instanciated action (could come from cache if already instanciated {@link #getActionFromCache(String)}
     */
    A newAction(String actionKey, JComponent component);

    /**
     * Obtain an action instance given his key (should call {@link #newAction(String, JComponent)}
     * <p/>
     * This is a convinient method when you want to obtain an action with no attached widget.
     *
     * @param actionKey the key of action
     * @return the instanciated action (could come from cache if already instanciated {@link #getActionFromCache(String)}
     */
    A newAction(String actionKey);


    /**
     * Fire an action given his key, his source and tthe widget responsible of action
     *
     * @param actionKey the action's key
     * @param source    the object source of action
     * @param component the component doing the action
     */
    void fireAction(String actionKey, Object source, JComponent component);

    /**
     * Fire an action given his key and his source, no widget are involved here
     *
     * @param actionKey the action's key
     * @param source    the object source of action
     */
    void fireAction(String actionKey, Object source);

    /**
     * Fire an action given his action's key, his source and the real action.
     * <p/>
     * This is a convinient method when you need to modified action before fire it.
     *
     * @param actionKey action's key
     * @param source    source of action
     * @param action    real action
     */
    void fireAction0(String actionKey, Object source, A action);


    /**
     * dispose all actions in cache using {@link MyAbstractAction#disposeUI()} on each
     * action, then {@link #resetCache()}
     */
    void dispose();
}
