/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: BundleMojo.java 1839 2011-01-12 16:41:52Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.2/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/bundle/BundleMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.bundle;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.bundle.I18nBundleUtil;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.io.SortedProperties;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Generate an aggregate i18n bundle for all dependencies of the project.
 * <p/>
 * The main idea is to have a final unique i18n bundle for a application to
 * launch, this really improve i18n loading time to have a unique named bundle,
 * no need to seek in all dependencies...
 * <p/>
 * Moreover, this permits also to deal with order of i18n keys, more precisly,
 * we want to use the higher level i18n key for an application. If the i18n
 * key is present on a library, we want to be able to override it in
 * application (or user wants it:)).
 * <p/>
 * This goal permits this using the dependencies graph order of artifacts.
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal bundle
 * @phase generate-resources
 * @execute goal=collect-i18n-artifacts
 * @requiresProject true
 * @requiresDependencyResolution runtime
 * @since 0.12
 */
public class BundleMojo extends AbstractMakeI18nBundleMojo {

    /**
     * A flag to use the parent of directory {@link #bundleOutputDir}
     * as directory to add in build ressources.
     * <p/>
     * If the flag is not activated, the, we use directly the given
     * {@link #bundleOutputDir} directory.
     * <p/>
     * <b>Note :</b> By default we use the parent directory, since this is more
     * natural, only special cases should desactivate this parameter.
     *
     * @parameter expression="${i18n.addBundleOuputDirParent}" default-value="true"
     * @since 2.0
     */
    protected boolean addBundleOuputDirParent;

    /**
     * A flag to generate the i18n definition file.
     * <p/>
     * This file contains all generated bundles and the paths of all i18n
     * artifacts used to make it.
     *
     * @parameter expression="${i18n.generateDefinitionFile}" default-value="true"
     * @since 2.0
     */
    protected boolean generateDefinitionFile;

    @Override
    public void init() throws Exception {
        super.init();

        // ajout de repertoire de generation (le parent en fait)
        // dans les resources du projet

        File ressourceDir = bundleOutputDir;

        if (addBundleOuputDirParent) {
            ressourceDir = ressourceDir.getParentFile();
        }
        addResourceDir(ressourceDir, "**/*.properties");
    }

    @Override
    protected void doAction() throws Exception {
        long t00 = System.nanoTime();

        String version = getProject().getVersion();
        version = PluginHelper.removeSnapshotSuffix(version);

        if (!silent) {
            getLog().info("config - bundle name : " + bundleOutputName);
            getLog().info("config - basedir     : " + bundleOutputDir);
            getLog().info("config - locales     : " + Arrays.toString(locales));
            getLog().info("config - version     : " + version);
        }

        Map<Locale, String> bundleDico =
                new LinkedHashMap<Locale, String>(locales.length);

        for (Locale locale : locales) {

            long t0 = System.nanoTime();

            File bundleOut = getI18nFile(bundleOutputDir,
                                         bundleOutputName,
                                         locale,
                                         false
            );

            SortedProperties propertiesOut =
                    new SortedProperties(encoding, false);
            StringBuilder buffer = new StringBuilder();

            URL[] urls = getCollectI18nResources(locale);
            if (urls.length == 0) {
                getLog().warn("no bundle for locale " + locale);
                continue;
            }

            if (!silent) {
                getLog().info("generate bundle for locale " + locale +
                              " from " + urls.length + " i18n resource(s)");
            }

            List<String> bundlesUrls = new ArrayList<String>();
            for (URL url : urls) {
                long t000 = System.nanoTime();
                I18nBundleEntry bundleEntry =
                        new I18nBundleEntry(url, locale, null);
                bundleEntry.load(propertiesOut);
                String strPath = bundleEntry.getPath().toString();
                int index = strPath.indexOf("i18n/");

                String str = strPath.substring(index);
                bundlesUrls.add(str);
                buffer.append(',').append(str);
                if (verbose) {
                    getLog().info(
                            "loaded " + bundleEntry.getPath() + " in " +
                            PluginHelper.convertTime(t000, System.nanoTime()));
                }
            }

            if (!bundlesUrls.isEmpty()) {
                bundleDico.put(locale, buffer.substring(1));
                if (!silent) {
                    if (getLog().isDebugEnabled()) {
                        getLog().debug(bundlesUrls.size() +
                                       " i18n resource(s) detected");
                    }
                    for (String u : bundlesUrls) {
                        getLog().info(u);
                    }
                }
            }
            propertiesOut.store(bundleOut);
            if (!silent && verbose) {
                getLog().info(
                        "bundle created in " +
                        PluginHelper.convertTime(t0, System.nanoTime()) +
                        " (detected sentences : " + propertiesOut.size() + ")");
            }
            if (checkBundle) {
                checkBundle(locale, propertiesOut, showEmpty, unsafeMapping);
            }
        }

        failsIfWarning();

        if (generateDefaultLocale) {
            generateDefaultBundle();
        }

        if (generateDefinitionFile) {

            generateDefinitionFile(version, bundleDico);
        }
        if (!silent && verbose) {
            getLog().info("done in " +
                          PluginHelper.convertTime(t00, System.nanoTime()));
        }
    }

    @Override
    protected File getBundleFile(File root,
                                 String artifactId,
                                 Locale locale,
                                 boolean create) throws IOException {
        return getI18nFile(root, artifactId, locale, create);
    }

    protected void generateDefinitionFile(String version,
                                          Map<Locale, String> bundleDico) throws IOException {

        // ecriture du ficher des definitions i18n (permet de faire une
        // recherche exacte sur un fichier puis d'en deduire les bundles a
        // charger
        String f = String.format(DefaultI18nInitializer.UNIQUE_BUNDLE_DEF,
                                 bundleOutputName);
        File defOut = new File(bundleOutputDir, f);
        if (!silent) {
            getLog().info("prepare i18n definition file in " +
                          defOut.getAbsolutePath());
        }
        SortedProperties p = new SortedProperties(encoding, false);
        p.setProperty(DefaultI18nInitializer.BUNDLE_DEF_LOCALES, bundles);
        p.setProperty(DefaultI18nInitializer.BUNDLE_DEF_VERSION, version);
        for (Entry<Locale, String> e : bundleDico.entrySet()) {
            p.setProperty(DefaultI18nInitializer.BUNDLES_FOR_LOCALE +
                          e.getKey().toString(), e.getValue());
        }
        FileOutputStream out = new FileOutputStream(defOut);
        try {
            p.store(out, null);
        } finally {
            out.close();
        }
    }

    @Override
    protected URL[] getCollectI18nResources(Locale locale) throws IOException {
        File file = getCollectOutputFile(locale, false);
        if (!file.exists()) {
            return I18nBundleUtil.EMPTY_URL_ARRAY;
        }
        URL[] urls = PluginHelper.getLinesAsURL(file);
        return urls;
    }
}
