/**
 * *##% maven-guix-plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix;

import java.io.File;
import java.util.ArrayList;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.nuiton.guix.generator.GwtGenerator;
import org.nuiton.guix.generator.SwingGenerator;

/**
 * Launches the program
 *
 * @author morin
 * @goal generate
 * @phase process-sources
 * @requiresDependencyResolution compile
 * @requiresProject
 */
public class GuixMojo extends AbstractMojo {

    /**
     * Project dependencies.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     */
    protected MavenProject project;
    /**
     * Directory of the files to compile.
     * @parameter expression="${guix.guixFilesDir}" default-value="${project.basedir}/src/main/java"
     * @since 0.0.1
     */
    private String guixFilesDir;
    /**
     * Base Directory of the files to compile. Used to determine the package names.
     * @parameter expression="${guix.guixFilesBaseDir}" default-value="${project.basedir}/src/main/java"
     * @since 0.0.1
     */
    private String guixFilesBaseDir;
    /**
     * Directory of the generated files.
     * @parameter expression="${guix.targetDirectory}" default-value="${basedir}/target/generatedFiles"
     * @since 0.0.1
     */
    private String targetDirectory;
    /**
     * Main class of the application to generate.
     * @parameter expression="${guix.mainClass}"
     * @required
     * @since 0.0.1
     */
    private String mainClass;
    /**
     * Generation language
     * @parameter expression="${guix.generationLanguage}"
     * @required
     * @since 0.0.1
     */
    private String generationLanguage;
    /**
     * Generation language
     * @parameter expression="${guix.launcherName}" default-value="Main"
     * @since 0.0.1
     */
    private String launcherName;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        File guixFilesBaseDir = new File(this.guixFilesBaseDir);
        File guixFilesDir = new File(this.guixFilesDir);
        File targetDirectory = new File(this.targetDirectory);
        Class generatorClass = null;
        if (generationLanguage.equalsIgnoreCase("Swing")) {
            generatorClass = SwingGenerator.class;
            SwingGuixInitializer.initialize();
            fixCompileSourceRoots(targetDirectory);
        }
        else if (generationLanguage.equalsIgnoreCase("GWT")) {
            generatorClass = GwtGenerator.class;
            GwtGuixInitializer.initialize();
            fixCompileSourceRoots(new File(targetDirectory, "java"));
            
        }
                
        if (guixFilesDir.exists() && guixFilesDir.isDirectory()) {
            ArrayList<File> guixFiles = goDeeperInto(guixFilesDir);

            GuixLauncher gcl = new GuixLauncher(guixFiles.toArray(
                    new File[guixFiles.size()]), targetDirectory,
                    guixFilesBaseDir, mainClass, generatorClass, generationLanguage, launcherName);
            boolean result = gcl.compile();
            if (result) {
                getLog().info("Compilation succeeded !");
            }
            else {
                getLog().error("Compilation failed...");
            }
            getLog().info("Compilation over");
        }
        else {
            getLog().error("Sources not found.");
        }
    }

    private ArrayList<File> goDeeperInto(File dir) {
        getLog().debug("goind deeper into " + dir.getPath());
        ArrayList<File> result = new ArrayList<File>();
        for (File f : dir.listFiles()) {
            if (f.isDirectory() && !f.isHidden()) {
                result.addAll(goDeeperInto(f));
            }
            else if (f.getName().endsWith(".guix") || f.getName().endsWith(".jaxx")) {
                getLog().debug("found " + f.getName());
                result.add(f);
            }
        }
        return result;
    }

    protected void fixCompileSourceRoots(File targetDirectory) {
        //fixme should remove this silly test when we will make real maven plugin tests :)
        if (project != null) {
            if (!project.getCompileSourceRoots().contains(targetDirectory.getPath())) {
                project.addCompileSourceRoot(targetDirectory.getPath());

            }
        }
    }
}
